#[cfg(test)] mod test;

pub trait Item {
	type Period;

	fn matches(&self, new: &Self, within: &Self::Period) -> bool;
}

impl<T: Clone + std::ops::Sub> Item for T
where T::Output: PartialOrd,
{
	type Period = T::Output;

	fn matches(&self, old: &T, within: &T::Output) -> bool {
		self.clone() - old.clone() >= *within
	}
}

pub struct Tagged<Item, T>(pub Item, pub T);

impl<I: crate::Item, T> Item for Tagged<I, T> {
	type Period = I::Period;

	fn matches(&self, old: &Self, within: &I::Period) -> bool {
		self.0.matches(&old.0, within)
	}
}

pub struct Bucket<Period> {
	period: Period,
	count: usize,
}

impl<Period> Bucket<Period> {
	pub fn new(period: Period, count: usize) -> Self {
		Bucket {
			period,
			count,
		}
	}
}

pub struct Config<Period> {
	buckets: Vec<Bucket<Period>>,
}

impl<Period> Default for Config<Period> {
	fn default() -> Self {
		Self::empty()
	}
}

impl<Period> Config<Period> {
	pub fn empty() -> Self {
		Config {
			buckets: Default::default(),
		}
	}

	/// Insert a new retention bucket.
	///
	/// For optimal performance insert buckets from shortest retention to maximum retention.
	pub fn add_bucket(&mut self, bucket: Bucket<Period>)
	where Period: PartialOrd
	{
		for i in (0..self.buckets.len()).rev() {
			if bucket.period >= self.buckets[i].period {
				self.buckets.insert(i+1, bucket);
				return
			}
		}

		self.buckets.insert(0, bucket);
	}
}

struct Iter<'a, Item: crate::Item, I: Iterator<Item=Item>> {
	config: &'a Config<Item::Period>,
	candidates: I,
	anchor: Vec<usize>,
	last: Vec<usize>,
	owned: Vec<usize>,
	removed: Vec<Item>,
	keep: Vec<Item>,
}

impl<'a, Item: crate::Item, I: Iterator<Item=Item>> Iter<'a, Item, I> {
	fn disown(&mut self, bid: usize) {
		self.owned[bid] -= 1;

		let last = self.last[bid];
		if let Some(bucket) = self.config.buckets.get(bid+1) {
			if last > 0 && !self.keep[last].matches(&self.keep[last-1], &bucket.period) {
				self.removed.push(self.keep.remove(last));
				for last in &mut self.last[..bid] {
					*last -= 1;
				}
				for anchor in &mut self.anchor {
					if *anchor >= last {
						*anchor -= 1;
					}
				}
			} else {
				self.last[bid] += 1;
				if self.owned[bid+1] >= bucket.count {
					self.disown(bid+1)
				}
			}
		} else {
			debug_assert_eq!(last, 0);
			self.removed.push(self.keep.remove(last));
			for last in &mut self.last {
				if *last > 0 {
					*last -= 1;
				}
			}
			for anchor in &mut self.anchor {
				*anchor -= 1;
			}
		}
	}
}

impl<'a, Item: crate::Item, I: Iterator<Item=Item>> Iterator for Iter<'a, Item, I> {
	type Item = Item;

	fn next(&mut self) -> Option<Self::Item> {
		if let Some(removed) = self.removed.pop() {
			return Some(removed)
		}

		let e = self.candidates.next()?;

		if self.keep.is_empty() {
			self.keep.push(e);
			self.owned[0] += 1;
			return self.next()
		}

		let mut accept = false;
		for (bid, bucket) in self.config.buckets.iter().enumerate() {
			if e.matches(&self.keep[self.anchor[bid]], &bucket.period) {
				if bid == 0 {
					accept = true;
				}
				if accept {
					self.owned[bid] += 1;
					self.anchor[bid] = self.keep.len();
				}
			}
		}

		if !accept {
			return Some(e)
		}

		self.keep.push(e);
		for bid in 0..self.config.buckets.len() {
			if self.owned[bid] > self.config.buckets[bid].count {
				self.disown(bid);
			}
		}

		self.next()
	}
}

pub fn prune<'a, Item: crate::Item + 'a>(
	config: &'a Config<Item::Period>,
	candidates: impl IntoIterator<Item=Item> + 'a)
-> impl Iterator<Item=Item> + 'a
{
	let last = config.buckets.iter().map(|_| 0).collect();
	let anchor = config.buckets.iter().map(|_| 0).collect();
	let owned = config.buckets.iter().map(|_| 0).collect();
	
	Iter {
		candidates: candidates.into_iter(),
		last,
		owned,
		anchor,
		config,
		keep: Vec::new(),
		removed: Vec::new(),
	}
}

#[test]
fn daily() {
	let c = Config {
		buckets: vec![Bucket{period: 3, count: 3}],
	};

	// For cases like this we essentially pick an arbitrary reference point. It isn't elegant but it works.
	test::assert_keep(&c,
		1..=12,
		[4, 7, 10]);

	test::assert_keep(&c,
		2..=12,
		[5, 8, 11]);

	// However subsequent runs stay synchronized.
	test::assert_keep(&c,
		[4, 7, 10, 13],
		[7, 10, 13]);

	test::assert_keep(&c,
		[7, 10, 13, 14],
		[7, 10, 13]);

	test::assert_keep(&c,
		[4, 7, 10, 13, 14],
		[7, 10, 13]);

	test::assert_keep(&c,
		[4, 7, 10, 13, 14, 15, 16, 17],
		[10, 13, 16]);

	test::assert_keep(&c,
		[5, 8, 11, 13, 14],
		[8, 11, 14]);
}

#[test]
fn daily_weekly() {
	let c = Config {
		buckets: vec![
			Bucket{period: 3, count: 3},
			Bucket{period: 6, count: 4},
		],
	};

	test::assert_keep(&c,
		1..=10,
		[1, 4, 7, 10]);

	test::assert_keep(&c,
		1..=20,
		[1, 7, 13, 16, 19]);

	test::assert_keep(&c,
		1..=30,
		[7, 13, 19, 22, 25, 28]);

	test::assert_keep(&c,
		1..=50,
		[25, 31, 37, 43, 46, 49]);
}

#[test]
fn daily_weekly_unaligned() {
	let c = Config {
		buckets: vec![
			Bucket{period: 3, count: 2},
			Bucket{period: 7, count: 4},
		],
	};

	// test::assert_keep(&c,
	// 	1..=10,
	// 	[1, 4, 7, 10]);

	test::assert_keep(&c,
		1..=20,
		[1, 10, 16, 19]);

	test::assert_keep(&c,
		1..=30,
		[1, 10, 19, 25, 28]);

	test::assert_keep(&c,
		1..=50,
		[10, 19, 28, 37, 46, 49]);
}


#[test]
fn dates() {
	let c = Config {
		buckets: vec![
			Bucket{period: chrono::Duration::days(2), count: 3},
			Bucket{period: chrono::Duration::weeks(1), count: 4},
		],
	};

	test::assert_keep(&c,
		(1..=30).map(|d| chrono::NaiveDate::from_ymd(1, 1, d)),
		[
			chrono::NaiveDate::from_ymd(1, 1, 1),
			chrono::NaiveDate::from_ymd(1, 1, 9),
			chrono::NaiveDate::from_ymd(1, 1, 17),
			chrono::NaiveDate::from_ymd(1, 1, 25),
			chrono::NaiveDate::from_ymd(1, 1, 27),
			chrono::NaiveDate::from_ymd(1, 1, 29),
		]);
}
