use super::*;
use crate::asg::*;
use std::{collections::HashMap, sync::Arc};

#[derive(Debug)]
pub enum Constructable {
    Struct {
        name: String,
        items: Vec<(String, usize)>,
    },
    Tuple(Vec<usize>),
    TaggedTuple {
        name: String,
        items: Vec<usize>,
    },
    TaggedEnum {
        name: String,
        discriminant: String,
        values: Vec<usize>,
    },
    TaggedEnumStruct {
        name: String,
        discriminant: String,
        values: Vec<(String, usize)>,
    },
}

#[derive(Debug)]
pub enum Instruction {
    Eval(usize, Expression),
    Construct(usize, Constructable),
    // source, new_stream, len constraint
    Constrict(Target, usize, usize),
    WrapStream(Target, usize, Arc<Transform>, Vec<usize>), // stream, new stream, transformer, arguments
    ConditionalWrapStream(
        usize,
        Vec<Instruction>,
        Target,
        usize,
        Arc<Transform>,
        Vec<usize>,
    ), // condition, prelude, stream, new stream, transformer, arguments

    DecodeForeign(Target, usize, Arc<ForeignType>, Vec<usize>),
    DecodeRef(Target, usize, String, Vec<usize>),
    DecodeRepr(String, PrimitiveType, usize, Target),
    DecodePrimitive(Target, usize, PrimitiveType),
    DecodePrimitiveArray(Target, usize, PrimitiveType, Option<usize>),
    DecodeReprArray(Target, usize, String, PrimitiveType, Option<usize>),
    // target, register of length
    Skip(Target, usize),

    // register representing: internal stream, end index, terminator, output handle, inner
    Loop(
        Target,
        Option<usize>,
        Option<usize>,
        usize,
        Vec<Instruction>,
    ),
    LoopOutput(usize, usize), // output handle, item
    Conditional(usize, usize, usize, Vec<Instruction>), // target, interior_register, condition, if_true
    ConditionalPredicate(usize, Vec<Instruction>), // target, interior_register, condition, if_true
    /// returns from decoder early
    Return(usize),
    Error(String),
}

#[derive(Debug)]
pub struct Context {
    pub register_count: usize,
    pub field_register_map: HashMap<String, usize>,
    pub instructions: Vec<Instruction>,
    pub name: String,
}

impl Context {
    fn alloc_register(&mut self) -> usize {
        let x = self.register_count;
        self.register_count += 1;
        x
    }
}

impl Context {
    pub fn new() -> Context {
        Context {
            name: String::new(),
            instructions: vec![],
            field_register_map: HashMap::new(),
            register_count: 0,
        }
    }

    pub fn decode_field_top(&mut self, field: &Arc<Field>) {
        assert!(field.toplevel);
        self.name = field.name.clone();
        let mut value = self.decode_field(Target::Direct, field);
        match &*field.type_.borrow() {
            Type::Foreign(_) => (),
            Type::Container(_) => (),
            Type::Enum(_) => (),
            Type::Bitfield(_) => (),
            _ => {
                if let Some(old_value) = value {
                    let extra_value = self.alloc_register();
                    self.instructions.push(Instruction::Construct(
                        extra_value,
                        Constructable::TaggedTuple {
                            name: field.name.clone(),
                            items: vec![old_value],
                        },
                    ));
                    value = Some(extra_value);
                }
            }
        }
        if let Some(value) = value {
            self.instructions.push(Instruction::Return(
                value,
            ));    
        }
    }

    fn decode_field_condition(&mut self, field: &Arc<Field>) -> Option<usize> {
        if let Some(condition) = field.condition.borrow().as_ref() {
            let value = self.alloc_register();
            self.instructions
                .push(Instruction::Eval(value, condition.clone()));
            Some(value)
        } else {
            None
        }
    }

    /// return of `None` means interior container
    pub fn decode_field(&mut self, source: Target, field: &Arc<Field>) -> Option<usize> {
        let field_condition = self.decode_field_condition(field);
        let start = self.instructions.len();
        
        let emitted = self.decode_field_unconditional(source, field);

        if let Some(field_condition) = field_condition {
            if emitted.is_none() {
                unimplemented!("cannot have interior containers with field conditional");
            }
            let target = self.alloc_register();
            let drained = self.instructions.drain(start..).collect();
            self.instructions.push(Instruction::Conditional(
                target,
                emitted.unwrap(),
                field_condition,
                drained,
            ));
            Some(target)
        } else {
            emitted
        }
    }

    fn decode_field_unconditional(&mut self, mut source: Target, field: &Arc<Field>) -> Option<usize> {
        let mut new_streams = vec![];

        for transform in field.transforms.borrow().iter().rev() {
            let condition = if let Some(condition) = &transform.condition {
                let value = self.alloc_register();
                self.instructions
                    .push(Instruction::Eval(value, condition.clone()));
                Some(value)
            } else {
                None
            };

            let argument_start = self.instructions.len();
            let mut args = vec![];
            for arg in transform.arguments.iter() {
                let r = self.alloc_register();
                self.instructions.push(Instruction::Eval(r, arg.clone()));
                args.push(r);
            }
            let new_stream = self.alloc_register();
            new_streams.push(new_stream);

            if let Some(condition) = condition {
                let drained = self.instructions.drain(argument_start..).collect();
                self.instructions.push(Instruction::ConditionalWrapStream(
                    condition,
                    drained,
                    source,
                    new_stream,
                    transform.transform.clone(),
                    args,
                ));
            } else {
                self.instructions.push(Instruction::WrapStream(
                    source,
                    new_stream,
                    transform.transform.clone(),
                    args,
                ));
            }
            source = Target::Stream(new_stream);
        }

        //todo: assert condition matching actual presence
        let emitted = match &*field.type_.borrow() {
            Type::Container(c) => {
                let buf_target = if let Some(length) = &c.length {
                    //todo: use limited stream
                    let len_register = self.alloc_register();
                    self.instructions
                        .push(Instruction::Eval(len_register, length.clone()));
                    let buf = self.alloc_register();
                    self.instructions
                        .push(Instruction::Constrict(source, buf, len_register));
                    Target::Stream(buf)
                } else {
                    source
                };
                if c.is_enum.get() {
                    for (name, child) in c.items.iter() {
                        let condition = self.decode_field_condition(child);
                        let start = self.instructions.len();
                        let decoded = self.decode_field_unconditional(buf_target, child);
                        let target = self.alloc_register();
                        
                        let subtype = child.type_.borrow();
                        match &*subtype {
                            Type::Container(c) => {
                                let mut values = vec![];
                                for (subname, subchild) in c.flatten_view() {
                                    if subchild.is_pad.get() || matches!(&*subchild.type_.borrow(), Type::Container(_)) {
                                        continue;
                                    }
    
                                    values.push((
                                        subname.clone(),
                                        *self
                                            .field_register_map
                                            .get(&subname)
                                            .expect("missing field in field_register_map"),
                                    ));
                                }

                                self.instructions.push(Instruction::Construct(target, Constructable::TaggedEnumStruct {
                                    name: field.name.clone(),
                                    discriminant: name.clone(),
                                    values,
                                }));
                            },
                            _ => {
                                let decoded = decoded.expect("enum discriminant was proper interior container, which is illegal");
                                self.instructions.push(Instruction::Construct(target, Constructable::TaggedEnum {
                                    name: field.name.clone(),
                                    discriminant: name.clone(),
                                    values: vec![decoded],
                                }));
                            },
                        }

                        if let Some(condition) = condition {
                            self.instructions.push(Instruction::Return(target));
                            let drained = self.instructions.drain(start..).collect();
                            self.instructions.push(Instruction::ConditionalPredicate(
                                condition,
                                drained,
                            ));
                        } else {
                            return Some(target);
                        }
                    }
                    self.instructions.push(Instruction::Error(format!("no enum conditions matched for {}", field.name)));
                    None
                } else {
                    for (name, child) in c.items.iter() {
                        let decoded = self.decode_field(buf_target, child);
                        if let Some(decoded) = decoded {
                            self.field_register_map.insert(name.clone(), decoded);
                        }
                    }
                    if field.toplevel {
                        let emitted = self.alloc_register();
                        let mut items = vec![];
                        for (name, child) in c.flatten_view() {
                            if child.is_pad.get() || matches!(&*child.type_.borrow(), Type::Container(_)) {
                                continue;
                            }
                            items.push((
                                name.clone(),
                                *self
                                    .field_register_map
                                    .get(&name)
                                    .expect("missing field in field_register_map"),
                            ));
                        }
                        self.instructions.push(Instruction::Construct(
                            emitted,
                            Constructable::Struct {
                                name: field.name.clone(),
                                items,
                            },
                        ));
                        Some(emitted)
                    } else {
                        None
                    }
                }
            }
            _ => self.decode_type(source, field),
        };

        emitted
    }

    pub fn decode_type(&mut self, source: Target, field: &Arc<Field>) -> Option<usize> {
        let output = self.alloc_register();
        Some(match &*field.type_.borrow() {
            Type::Container(_) => unimplemented!(),
            Type::Array(c) => {
                if field.is_pad.get() {
                    let array_type = field.type_.borrow();
                    let array_type = match &*array_type {
                        Type::Array(a) => &**a,
                        _ => panic!("invalid type for pad"),
                    };
                    let len = array_type.length.value.as_ref().cloned().unwrap();
                    let length_register = self.alloc_register();
                    self.instructions.push(Instruction::Eval(length_register, len));
                    self.instructions.push(Instruction::Skip(source, length_register));
                    return None;
                }
                let terminator = if c.length.expandable && c.length.value.is_some() {
                    let len = c.length.value.as_ref().cloned().unwrap();
                    let r = self.alloc_register();
                    self.instructions.push(Instruction::Eval(r, len));
                    Some(r)
                } else {
                    None
                };

                let len = if c.length.expandable {
                    None
                } else {
                    let len = c.length.value.as_ref().cloned().unwrap();
                    let r = self.alloc_register();
                    self.instructions.push(Instruction::Eval(r, len));
                    Some(r)
                };

                if c.element.condition.borrow().is_none()
                    && c.element.transforms.borrow().len() == 0
                    && terminator.is_none()
                {
                    let type_ = c.element.type_.borrow();
                    let type_ = type_.resolved();
                    match &*type_ {
                        // todo: const-length type optimizations for container/array/foreign
                        Type::Container(_) | Type::Array(_) | Type::Foreign(_) | Type::Ref(_) => (),
                        Type::Enum(x) => {
                            self.instructions.push(Instruction::DecodeReprArray(
                                source,
                                output,
                                field.name.clone(),
                                PrimitiveType::Scalar(x.rep.clone()),
                                len,
                            ));
                            return Some(output);
                        }
                        Type::Bitfield(x) => {
                            self.instructions.push(Instruction::DecodeReprArray(
                                source,
                                output,
                                field.name.clone(),
                                PrimitiveType::Scalar(x.rep.clone()),
                                len,
                            ));
                            return Some(output);
                        }
                        Type::Scalar(x) => {
                            self.instructions.push(Instruction::DecodePrimitiveArray(
                                source,
                                output,
                                PrimitiveType::Scalar(*x),
                                len,
                            ));
                            return Some(output);
                        }
                        Type::F32 => {
                            self.instructions.push(Instruction::DecodePrimitiveArray(
                                source,
                                output,
                                PrimitiveType::F32,
                                len,
                            ));
                            return Some(output);
                        }
                        Type::F64 => {
                            self.instructions.push(Instruction::DecodePrimitiveArray(
                                source,
                                output,
                                PrimitiveType::F64,
                                len,
                            ));
                            return Some(output);
                        }
                        Type::Bool => {
                            self.instructions.push(Instruction::DecodePrimitiveArray(
                                source,
                                output,
                                PrimitiveType::Bool,
                                len,
                            ));
                            return Some(output);
                        }
                    }
                }

                let current_pos = self.instructions.len();
                let item = self.decode_field(source, &c.element);
                if item.is_none() {
                    unimplemented!("cannot have inline container inside array");
                }
                self.instructions
                    .push(Instruction::LoopOutput(output, item.unwrap()));
                let drained = self.instructions.drain(current_pos..).collect();
                self.instructions
                    .push(Instruction::Loop(source, len, terminator, output, drained));
                output
            }
            Type::Enum(e) => {
                self.instructions.push(Instruction::DecodeRepr(
                    field.name.clone(),
                    PrimitiveType::Scalar(e.rep.clone()),
                    output,
                    source,
                ));
                output
            }
            Type::Bitfield(e) => {
                self.instructions.push(Instruction::DecodeRepr(
                    field.name.clone(),
                    PrimitiveType::Scalar(e.rep.clone()),
                    output,
                    source,
                ));
                output
            }
            Type::Scalar(s) => {
                self.instructions.push(Instruction::DecodePrimitive(
                    source,
                    output,
                    PrimitiveType::Scalar(*s),
                ));
                output
            }
            Type::F32 => {
                self.instructions.push(Instruction::DecodePrimitive(
                    source,
                    output,
                    PrimitiveType::F32,
                ));
                output
            }
            Type::F64 => {
                self.instructions.push(Instruction::DecodePrimitive(
                    source,
                    output,
                    PrimitiveType::F64,
                ));
                output
            }
            Type::Bool => {
                self.instructions.push(Instruction::DecodePrimitive(
                    source,
                    output,
                    PrimitiveType::Bool,
                ));
                output
            }
            Type::Foreign(f) => {
                self.instructions.push(Instruction::DecodeForeign(
                    source,
                    output,
                    f.clone(),
                    vec![],
                ));
                output
            }
            Type::Ref(r) => {
                let mut args = vec![];
                for arg in r.arguments.iter() {
                    let r = self.alloc_register();
                    self.instructions.push(Instruction::Eval(r, arg.clone()));
                    args.push(r);
                }
                if let Type::Foreign(f) = &*r.target.type_.borrow() {
                    self.instructions.push(Instruction::DecodeForeign(
                        source,
                        output,
                        f.clone(),
                        args,
                    ));
                } else {
                    self.instructions.push(Instruction::DecodeRef(
                        source,
                        output,
                        r.target.name.clone(),
                        args,
                    ));
                }
                output
            }
        })
    }
}
