use wasm_bindgen::prelude::*;
extern crate nalgebra as na;

use crate::transform::image;

// Wasm bindgen for javascript console output
#[wasm_bindgen]
extern "C" {
    #[wasm_bindgen(js_namespace = console)]
    pub fn log(s: &str); 

    #[wasm_bindgen(js_namespace = console, js_name = log)]
    pub fn log_u32(s: &str); 

    #[wasm_bindgen(js_namespace = console, js_name = log)]
    pub fn log_many(a: &str, b: &str); 
}

// macro for using javascript log bind
macro_rules! console_log {
    ($($t:tt)*) => (log(&format_args!($($t)*).to_string()));
}

// Point is a struct containing x and y coordinates
#[derive(Debug)]
pub struct Point {
    pub x: f32,
    pub y: f32, 
}

impl Default for Point {
    fn default() -> Self { Point{x: 0.0, y:0.0} }
}

#[derive(Debug)]
struct ProjectiveTransform {
    h_matrix: na::Matrix3<f32>,
}

impl ProjectiveTransform {
    fn new(points_a: [Point; 4], points_b: [Point; 4]) -> ProjectiveTransform {
        let m = construct_decompose_matrix(points_a, points_b);

        let svd_compute = na::SVD::new(m, true, true);
        let h_matrix = construct_homography_matrix(svd_compute);

        return ProjectiveTransform{ h_matrix: h_matrix };
    }


    fn transform_image(&self, _img: image::Image) {
    }

}

// homography/svd implementation on js/wasm, takes two parameters, a and b, being
// two sets of coordinates, with the x values followed by the y values: [x, y, x, y]
// and having 8 coordinates each set. Not having 8 coordinates will lead to a panic
// TODO: make assertions to avoid panic
#[allow(dead_code)]
#[wasm_bindgen]
pub fn projective_transform(width: u32, height: u32, image_buffer: &[u8], first_set: &[f32], second_set: &[f32]) {
    
    console_log!("First set of points {:?}", first_set);
    console_log!("Second set of points {:?}", second_set);
    
    console_log!("Image bytes: {:?}", image_buffer);
    
    let mut points_a: [Point; 4] = Default::default();
    let mut points_b: [Point; 4] = Default::default();
    
    for i in 0..4 {
        points_a[i].x = first_set[i*2];
        points_a[i].y = first_set[i*2+1];
        
        points_b[i].x = second_set[i];
        points_b[i].y = second_set[i*2+1];
    }

    let proj_transform = ProjectiveTransform::new(points_a, points_b);
    let img = image::Image::new(width, height, image_buffer);
    
    console_log!("{:?}", proj_transform.h_matrix);
    
    proj_transform.transform_image(img);
    // order the bytes into pixels(arrays) and these pixel into more arrays
    // each pixel will then have x and y coordinates, then we are able to reorder
    // them following the homography matrix
}

// construct_decompose_matrix takes the two sets of points, a and b and constructs the
// matrix to be decomposed by svd
pub fn construct_decompose_matrix(first_set: [Point; 4], second_set: [Point; 4]) -> na::SMatrix<f32, 8, 9> {
    let mut m = na::SMatrix::<f32, 8, 9>::repeat(0.0);
    
    for i in 0..4 {
        let vec_a = na::RowSVector::<f32, 9>::from_iterator([ -first_set[i].x, -first_set[i].y, -1.0, 0.0, 0.0, 0.0, first_set[i].x * second_set[i].x, first_set[i].y * second_set[i].x, second_set[i].x ]);
        let vec_b = na::RowSVector::<f32, 9>::from_iterator([ 0.0, 0.0, 0.0, -first_set[i].x, -first_set[i].y, -1.0, first_set[i].x * second_set[i].y, first_set[i].y * second_set[i].y, second_set[i].y ]);
        
        
        m.set_row(2*i, &vec_a);
        m.set_row(2*i+1, &vec_b);
    }
    
    return m;
}

fn construct_homography_matrix(svd: na::SVD<f32, na::Const<8>, na::Const<9>>) -> na::Matrix3<f32> {
    let mut h_matrix = na::Matrix3::repeat(0.0);
    
    match svd.v_t {
        Some(v_values) => {
            let v_row = v_values.row(v_values.nrows()-1);
            h_matrix[(0, 0)] = v_row[(0, 0)];
            h_matrix[(0, 1)] = v_row[(0, 1)];
            h_matrix[(0, 2)] = v_row[(0, 2)];
            h_matrix[(1, 0)] = v_row[(0, 3)];
            h_matrix[(1, 1)] = v_row[(0, 4)];
            h_matrix[(1, 2)] = v_row[(0, 5)];
            h_matrix[(2, 0)] = v_row[(0, 6)];
            h_matrix[(2, 1)] = v_row[(0, 7)];
            h_matrix[(2, 2)] = 1.0; //v_row[(0, 8)]; 8 degrees of freedoms
        },
        None => println!("Invalid matrix"),
    }
    
    return h_matrix;
}

// construct_decompose_matrix_with_ninth_row takes two sets of points, a and b and construct the
// matrix to be decomposed by svd, this adds a ninth row to the matrix to be decomposed
// need to research more about this, some say to just add the last row, some just add the 1 later to the
// homography matrix
#[allow(dead_code)]
fn construct_decompose_matrix_with_ninth_row(a: [Point; 4], b: [Point; 4]) -> na::SMatrix<f32, 9, 9> {
    let mut m = na::SMatrix::<f32, 9, 9>::repeat(0.0);
    
    for i in 0..4 {
        let vec_a = na::RowSVector::<f32, 9>::from_iterator([ -a[i].x, -a[i].y, -1.0, 0.0, 0.0, 0.0, a[i].x * b[i].x, a[i].y * b[i].x, b[i].x ]);
        let vec_b = na::RowSVector::<f32, 9>::from_iterator([ 0.0, 0.0, 0.0, -a[i].x, -a[i].y, -1.0, a[i].x * b[i].y, a[i].y * b[i].y, b[i].y ]);
        
        
        m.set_row(2*i, &vec_a);
        m.set_row(2*i+1, &vec_b);    
    }
    
    m.set_row(8, &na::RowSVector::<f32, 9>::from_iterator([ 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0 ]));
    
    return m;
}

// homography/svd implementation on development console (no bindings for js/wasm)
pub fn projective_transform_debug(width: u32, height: u32, image_buffer: &[u8], first_set: [Point; 4], second_set: [Point; 4]) {
    println!("First set of points {:?}", first_set);
    println!("Second set of points {:?}", second_set);

    let transform = ProjectiveTransform::new(first_set, second_set);
    println!("{:?}", transform);
    println!("{:?}", transform.h_matrix);

    let image = image::Image::new(width, height, image_buffer);

    println!("{:?}", image_buffer[0]); // buffer must have at least 30 indexes or panics
    println!("{:?}", image_buffer[1]); // buffer must have at least 30 indexes or panics
    println!("{:?}", image_buffer[2]); // buffer must have at least 30 indexes or panics
    println!("{:?}", image.get_pixel(0, 0));
}