#[cfg(feature = "localtime")]
mod localtime {
    use std::time::SystemTime;

    /// Return a string representing the current date and time as localtime.
    ///
    /// Available with the `localtime` feature toggle.
    pub fn format_now_datetime_seconds() -> String {
        let t = time::OffsetDateTime::now_utc();
        t.to_offset(time::UtcOffset::try_local_offset_at(t).unwrap_or(time::UtcOffset::UTC))
            .format("%F %T")
    }

    /// Return a string representing the current time as localtime.
    ///
    /// Available with the `localtime` feature toggle.
    pub fn format_time_for_messages(time: SystemTime) -> String {
        time::OffsetDateTime::from(time)
            .to_offset(time::UtcOffset::try_current_local_offset().unwrap_or(time::UtcOffset::UTC))
            .format("%T")
    }
}

/// An `hours:minute:seconds` format.
pub const DATE_TIME_HMS: usize = "00:51:45".len();

#[cfg(not(feature = "localtime"))]
mod utc {
    use super::DATE_TIME_HMS;
    use std::time::SystemTime;
    const DATE_TIME_YMD: usize = "2020-02-13T".len();

    /// Return a string representing the current date and time as UTC.
    ///
    /// Available without the `localtime` feature toggle.
    pub fn format_time_for_messages(time: SystemTime) -> String {
        String::from_utf8_lossy(
            &humantime::format_rfc3339_seconds(time).to_string().as_bytes()
                [DATE_TIME_YMD..DATE_TIME_YMD + DATE_TIME_HMS],
        )
        .into_owned()
    }

    /// Return a string representing the current time as UTC.
    ///
    /// Available without the `localtime` feature toggle.
    pub fn format_now_datetime_seconds() -> String {
        String::from_utf8_lossy(
            &humantime::format_rfc3339_seconds(std::time::SystemTime::now())
                .to_string()
                .as_bytes()[.."2020-02-13T00:51:45".len()],
        )
        .into_owned()
    }
}

#[cfg(feature = "localtime")]
pub use localtime::*;

#[cfg(not(feature = "localtime"))]
pub use utc::*;
