//     A stand-alone jack application that listens to a midi channel and prints the chord names.
//     Copyright (C) 2021  Pieter Penninckx
//
//     This program is free software: you can redistribute it and/or modify
//     it under the terms of the GNU Affero General Public License as
//     published by the Free Software Foundation, either version 3 of the
//     License, or (at your option) any later version that is approved by Pieter Penninckx.
//
//     This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty of
//     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//     GNU Affero General Public License for more details.
//
//     You should have received a copy of the GNU Affero General Public License
//     along with this program.  If not, see <https://www.gnu.org/licenses/>.
//! # Print-chords
//! A stand-alone jack application that listens to a midi channel and prints the chord names.
//!
//! # Usage
//! See the [command line options](`Options`).
#[macro_use]
extern crate enum_primitive;

mod chord_listener;
mod chord_printer;

use crate::chord_listener::MultiChannelChordListener;
use anyhow::{anyhow, Context, Result};
use chord_printer::string_representation;
use doc_comment::doc_comment;
use jack::{RingBufferReader, RingBufferWriter};
use midi_consts::channel_event::control_change::{ALL_NOTES_OFF, ALL_SOUND_OFF};
use midi_consts::channel_event::{CONTROL_CHANGE, EVENT_TYPE_MASK, NOTE_OFF, NOTE_ON};
use rsynth::backend::jack_backend::run;
use rsynth::backend::HostInterface;
use rsynth::buffer::AudioBufferInOut;
use rsynth::event::event_queue::{AlwaysInsertNewAfterOld, EventQueue};
use rsynth::event::{ContextualEventHandler, Indexed, RawMidiEvent, SysExEvent, Timed};
use rsynth::meta::{InOut, Meta, MetaData};
use rsynth::{AudioHandler, ContextualAudioRenderer};
use std::error::Error;
use std::fmt::{Display, Formatter};
use std::io::Write;
use std::thread::spawn;
use std::{env, fmt};

const USAGE: &'static str = include_str!("../USAGE.txt");
const FIRST_BIT: u128 = (0b1000000000000000 as u128) << (128 - 16);

struct JackChordListener {
    meta_data: MetaData<&'static str, &'static str, &'static str>,
    inner: MultiChannelChordListener,
    event_queue: EventQueue<RawMidiEvent>,
    writer: RingBufferWriter,
    previous_chord: u128,
}

impl JackChordListener {
    fn meta_data() -> MetaData<&'static str, &'static str, &'static str> {
        MetaData {
            general_meta: "Print chords",
            audio_port_meta: InOut {
                inputs: Vec::new(),
                outputs: Vec::new(),
            },
            midi_port_meta: InOut {
                inputs: vec!["midi in"],
                outputs: Vec::new(),
            },
        }
    }

    pub fn new(writer: RingBufferWriter) -> Self {
        Self {
            meta_data: Self::meta_data(),
            inner: MultiChannelChordListener::new_with_all_channels(),
            event_queue: EventQueue::new(1024),
            writer,
            previous_chord: 0,
        }
    }
}

impl Meta for JackChordListener {
    type MetaData = MetaData<&'static str, &'static str, &'static str>;

    fn meta(&self) -> &Self::MetaData {
        &self.meta_data
    }
}

impl AudioHandler for JackChordListener {
    fn set_sample_rate(&mut self, _sample_rate: f64) {}
}

impl<Context> ContextualAudioRenderer<f32, Context> for JackChordListener
where
    Context: HostInterface,
{
    fn render_buffer(&mut self, _buffer: &mut AudioBufferInOut<f32>, _context: &mut Context) {
        let mut must_print_chord = false;

        for event in self.event_queue.iter() {
            self.inner.handle_event(event.event.data());

            let chord = self.inner.get_chord();
            if chord.count_ones() > self.previous_chord.count_ones() {
                must_print_chord = true;
            }
            self.previous_chord = chord;
        }
        self.event_queue.clear();

        if must_print_chord {
            self.writer
                .write_buffer(&self.previous_chord.to_ne_bytes()[..]);
        }
    }
}

impl<Context> ContextualEventHandler<Indexed<Timed<RawMidiEvent>>, Context> for JackChordListener {
    fn handle_event(&mut self, event: Indexed<Timed<RawMidiEvent>>, _context: &mut Context) {
        let data = event.event.event.data();
        if matches!(
            (data[0] & EVENT_TYPE_MASK, data[1]),
            (NOTE_OFF, _)
                | (NOTE_ON, _)
                | (CONTROL_CHANGE, ALL_NOTES_OFF)
                | (CONTROL_CHANGE, ALL_SOUND_OFF)
        ) {
            self.event_queue
                .queue_event(event.event, AlwaysInsertNewAfterOld);
        }
    }
}

impl<'a, Context> ContextualEventHandler<Indexed<Timed<SysExEvent<'a>>>, Context>
    for JackChordListener
{
    fn handle_event(&mut self, _event: Indexed<Timed<SysExEvent>>, _context: &mut Context) {}
}

struct JackChordPrinter {
    reader: RingBufferReader,
}

impl JackChordPrinter {
    fn new(reader: RingBufferReader) -> Self {
        Self { reader }
    }

    fn run(&mut self) {
        let mut buffer = [0_u8; 16];
        loop {
            let mut chord = 0;
            let mut found = false;
            while self.reader.space() >= 16 {
                self.reader.read_buffer(&mut buffer[..]);
                chord = u128::from_ne_bytes(buffer);
                found = true;
            }
            if found {
                print!("\r{:10}", string_representation(chord));
                let _ = std::io::stdout().flush();
            }
            std::thread::sleep(std::time::Duration::from_millis(10));
        }
    }
}

doc_comment!(
    concat!("# Usage\n```txt\n", include_str!("../USAGE.txt"), "\n```\n"),
    pub struct Options {
        print_help: bool,
    }
);

impl Default for Options {
    fn default() -> Self {
        Self { print_help: false }
    }
}

impl Options {
    fn from_command_line_arguments() -> Result<Self> {
        let mut args: Vec<String> = env::args().rev().collect();
        let _ = args.pop();
        let mut result = Self::default();
        while let Some(arg) = args.pop() {
            match arg.as_str() {
                "--help" | "-h" => {
                    result.print_help = true;
                }
                a => return Err(anyhow!("Unknown command line argument: {}", a)),
            }
        }
        Ok(result)
    }
}

// TODO: remove this struct when switching to the jack crate version 0.7.
#[derive(Debug)]
struct RingBufferCreationFailed;

impl Display for RingBufferCreationFailed {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "failed to create jack ringbuffer")
    }
}

impl Error for RingBufferCreationFailed {}

fn main() -> Result<()> {
    let options = Options::from_command_line_arguments()?;
    if options.print_help {
        println!("{}", USAGE);
        return Ok(());
    }
    let ringbuf = jack::RingBuffer::new(1024)
        .map_err(|_| RingBufferCreationFailed)
        .context("Error while creating jack ringbuffer.")?;
    let (reader, writer) = ringbuf.into_reader_writer();
    let listener = JackChordListener::new(writer);
    let mut printer = JackChordPrinter::new(reader);
    spawn(move || printer.run());

    run(listener).context("Error while running or starting jack client.")?;
    Ok(())
}
