//     A stand-alone jack application that listens to a midi channel and prints the chord names.
//
//     Copyright (C) 2021  Pieter Penninckx
//
//     This program is free software: you can redistribute it and/or modify
//     it under the terms of the GNU Affero General Public License as
//     published by the Free Software Foundation, either version 3 of the
//     License, or (at your option) any later version that is approved by Pieter Penninckx.
//
//     This program is distributed in the hope that it will be useful,
//     but WITHOUT ANY WARRANTY; without even the implied warranty of
//     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//     GNU Affero General Public License for more details.
//
//     You should have received a copy of the GNU Affero General Public License
//     along with this program.  If not, see <https://www.gnu.org/licenses/>.
use crate::FIRST_BIT;
const FIRST_12_BITS: u128 = (0b1111111111110000 as u128) << (128 - 16);
const FIRST_12_BITS_U32: u32 = (0b1111111111110000 as u32) << (32 - 16);
use enum_primitive::FromPrimitive;

#[allow(unused)]
pub fn parse<'a>(name: &'a str) -> Option<u128> {
    let mut result = 0;
    'outer: for note in name.split(" ") {
        for i in 0..12 {
            if let Some(octave_num) = note.strip_prefix(note_name(i)) {
                if let Ok(oct) = octave_num.parse::<u32>() {
                    result |= FIRST_BIT >> (i + oct * 12);
                } else {
                    return None;
                }
                continue 'outer;
            }
        }
        return None;
    }
    Some(result)
}

fn wrap_chord(chord: u128) -> u16 {
    let mut chord = chord;
    let mut result = ((chord & FIRST_12_BITS) >> 128 - 16) as u16;
    while chord != 0 {
        chord <<= 12;
        result |= ((chord & FIRST_12_BITS) >> 128 - 16) as u16;
    }
    result & 0b1111111111110000
}

#[test]
fn wrap_chord_works() {
    let first_note = (1 as u128).rotate_right(1);
    assert_eq!(wrap_chord(first_note), 0b1000000000000000);
    assert_eq!(
        wrap_chord(first_note | (first_note >> 13)),
        0b1100000000000000
    );
}

const fn lowest(chord: u128) -> Option<u32> {
    let zeroes = chord.leading_zeros();
    if zeroes == 128 {
        None
    } else {
        Some(zeroes)
    }
}

const fn invert(chord: u16) -> u16 {
    let masked_chord = ((chord & (0b0111111111111111)) as u32) << 16;
    let mut chord = (chord as u32) << 16;
    chord = masked_chord | (chord >> 12);
    chord = chord << chord.leading_zeros();
    ((chord & FIRST_12_BITS_U32) >> 16) as u16
}

#[test]
fn invert_works() {
    let observed = invert(MAJOR);
    assert_eq!(observed, 0b1001000010000000);
}

const fn remove_highest(chord: u16) -> u16 {
    chord & (0b1111111111111110 << chord.trailing_zeros())
}

#[test]
fn remove_highest_works() {
    let a = 1 << 4;
    let b = 1 << 8;
    assert_eq!(remove_highest(a | b), b);
}

#[test]
fn remove_highest_still_works() {
    let c = 0b1000000000000000;
    let a = c >> 9;
    assert_eq!(remove_highest(a | c), c);
}

const fn relative_position_of_highest(chord: u16) -> u32 {
    chord.trailing_zeros() - 3
}

const fn relative_position_of_second_highest(chord: u16) -> u32 {
    relative_position_of_highest(remove_highest(chord))
}

#[test]
fn relative_position_of_second_highest_works() {
    let c = 0b1000000000000000;
    let a = c >> 9;
    assert_eq!(relative_position_of_second_highest(a | c), 12);
}

const fn relative_position_of_third_highest(chord: u16) -> u32 {
    relative_position_of_second_highest(remove_highest(chord))
}

// Adding a new chord type:
// Step 1: Add a bit representation here (do not add the inversions).
// This is a bit-wise representation of the chord,
// from left to right, a bit is 1 if it correspond to a note in the chord and 0 otherwise.
// Example, the C Major chord has the following notes: C, E and G.
// So if we write a 1 under the notes that are part of the chord and a 0 under the notes that are not,
// Then we get the following:
//     C  C♯ D  E♭ E  F  F♯ G  A♭ A  B♭ B (1  2  3  4)
//     1  0  0  0  1  0  0  1  0  0  0  0  0  0  0  0
// Add the end, we add four zeros.
// If the chord exceeds one octave, map the higher notes to the range of the first twelve notes.
const MAJOR: u16 = 0b1000100100000000;
const MINOR: u16 = 0b1001000100000000;
const MAJOR7: u16 = 0b1000100100010000;
const MINOR7: u16 = 0b1001000100010000;

// Adding a new chord type:
// Step 2: Add it to the list below, also add the inversions (if any).
enum_from_primitive! {
#[derive(Debug, PartialEq, Clone, Copy)]
#[repr(u16)]
enum ChordType {
    Major      = MAJOR,
    MajorInv1  = invert(MAJOR),
    MajorInv2  = invert(invert(MAJOR)),
    Minor      = MINOR,
    MinorInv1  = invert(MINOR),
    MinorInv2  = invert(invert(MINOR)),
    Major7     = MAJOR7,
    Major7Inv1 = invert(MAJOR7),
    Major7Inv2 = invert(invert(MAJOR7)),
    Major7Inv3 = invert(invert(invert(MAJOR7))),
    Minor7     = MINOR7,
    Minor7Inv1 = invert(MINOR7),
    Minor7Inv2 = invert(invert(MINOR7)),
    Minor7Inv3 = invert(invert(invert(MINOR7))),
    Other      = 0b1111111111111111,
    Silence    = 0,
}
}

impl ChordType {
    // Adding a new chord type:
    // Step 3: Add the suffix to the list below
    const fn suffix(&self) -> &'static str {
        use ChordType::*;
        match self {
            Major | MajorInv1 | MajorInv2 => "",
            Minor | MinorInv1 | MinorInv2 => "m",
            Major7 | Major7Inv1 | Major7Inv2 | Major7Inv3 => "7",
            Minor7 | Minor7Inv1 | Minor7Inv2 | Minor7Inv3 => "m7",
            Silence => "NC",
            Other => "???",
        }
    }

    // Adding a new chord type:
    // Step 4: Add the inversions to the list below.
    const fn inversion(&self) -> u8 {
        use ChordType::*;
        match self {
            Major | Minor | Major7 | Minor7 | Other | Silence => 0,
            MajorInv1 | MinorInv1 | Major7Inv1 | Minor7Inv1 => 1,
            MajorInv2 | MinorInv2 | Major7Inv2 | Minor7Inv2 => 2,
            Major7Inv3 | Minor7Inv3 => 3,
        }
    }

    fn root_difference(&self) -> u32 {
        match self.inversion() {
            0 => 0,
            1 => relative_position_of_highest(*self as u16),
            2 => relative_position_of_second_highest(*self as u16),
            3 => relative_position_of_third_highest(*self as u16),
            _ => unreachable!(),
        }
    }
}

fn chord_type(chord: u128) -> ChordType {
    if let Some(l) = lowest(chord) {
        let mut wrapped = (wrap_chord(chord) as u32) << 16;
        wrapped = wrapped | (wrapped >> 12);
        wrapped = wrapped << (l % 12);
        wrapped = wrapped & FIRST_12_BITS_U32;
        ChordType::from_u16((wrapped >> 16) as u16).unwrap_or(ChordType::Other)
    } else {
        ChordType::Silence
    }
}

fn note_name(note: u32) -> &'static str {
    match note % 12 {
        0 => "C",
        1 => "C♯",
        2 => "D",
        3 => "E♭",
        4 => "E",
        5 => "F",
        6 => "F♯",
        7 => "G",
        8 => "A♭",
        9 => "A",
        10 => "B♭",
        11 => "B",
        _ => unreachable!(),
    }
}

#[allow(unused)]
fn note_from_name<'a>(name: &'a str) -> Option<u8> {
    match name {
        "C" => Some(0),
        "C♯" => Some(1),
        "D" => Some(2),
        "E♭" => Some(3),
        "E" => Some(4),
        "F" => Some(5),
        "F♯" => Some(6),
        "G" => Some(7),
        "A♭" => Some(8),
        "A" => Some(9),
        "B♭" => Some(10),
        "B" => Some(11),
        _ => None,
    }
}

pub fn string_representation(chord: u128) -> String {
    if let Some(l) = lowest(chord) {
        let chord_type = chord_type(chord);
        let root_difference = chord_type.root_difference();
        let root = (12 + l - root_difference) % 12;
        if root_difference == 0 {
            format!("{}{}", note_name(root), chord_type.suffix())
        } else {
            format!(
                "{}{}/{}",
                note_name(root),
                chord_type.suffix(),
                note_name(l)
            )
        }
    } else {
        "".to_string()
    }
}
