use criterion::{criterion_group, criterion_main, Criterion};
use preserves::value::{self, Reader, Writer, PackedReader, PackedWriter};
use preserves::{de, ser};
use std::io::Read;
use std::io::BufReader;
use std::io::{Seek, SeekFrom};

#[path = "../tests/samples/mod.rs"]
mod samples;
use samples::TestCases;

pub fn bench_decoder_bytes(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let mut bs = vec![];
    fh.read_to_end(&mut bs).ok();
    c.bench_function("decode samples.bin via bytes", |b| b.iter_with_large_drop(
        || PackedReader::decode_bytes(&bs[..]).demand_next(true).unwrap()));
}

pub fn bench_decoder_file(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    c.bench_function("decode samples.bin via file", |b| b.iter_with_large_drop(|| {
        fh.seek(SeekFrom::Start(0)).ok();
        PackedReader::decode_read(&mut fh).demand_next(true).unwrap()
    }));
}

pub fn bench_decoder_buffered_file(c: &mut Criterion) {
    let mut fh = BufReader::new(std::fs::File::open("../../../tests/samples.bin").unwrap());
    c.bench_function("decode samples.bin via buffered file", |b| b.iter_with_large_drop(|| {
        fh.seek(SeekFrom::Start(0)).ok();
        PackedReader::decode_read(&mut fh).demand_next(true).unwrap()
    }));
}

pub fn bench_encoder(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let v = PackedReader::decode_read(&mut fh).demand_next(true).unwrap();
    c.bench_function("encode samples.bin", |b| b.iter_with_large_drop(|| {
        let mut bs = vec![];
        PackedWriter::new(&mut bs).write(&v).unwrap();
        bs
    }));
}

pub fn bench_de(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let mut bs = vec![];
    fh.read_to_end(&mut bs).ok();
    c.bench_function("deserialize samples.bin", |b| b.iter_with_large_drop(
        || de::from_bytes::<TestCases>(&bs[..]).unwrap()));
}

pub fn bench_ser(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let v: TestCases = de::from_read(&mut fh).unwrap();
    c.bench_function("serialize samples.bin", |b| b.iter_with_large_drop(|| {
        let mut bs = vec![];
        ser::to_writer(&mut PackedWriter::new(&mut bs), &v).unwrap();
        bs
    }));
}

pub fn bench_decoder_de(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let mut bs = vec![];
    fh.read_to_end(&mut bs).ok();
    c.bench_function("decode-then-deserialize samples.bin", |b| b.iter_with_large_drop(
        || value::de::from_value::<TestCases>(&PackedReader::decode_bytes(&bs[..]).demand_next(true).unwrap()).unwrap()));
}

pub fn bench_ser_encoder(c: &mut Criterion) {
    let mut fh = std::fs::File::open("../../../tests/samples.bin").unwrap();
    let v: TestCases = de::from_read(&mut fh).unwrap();
    c.bench_function("serialize-then-encode samples.bin", |b| b.iter_with_large_drop(|| {
        let mut bs = vec![];
        PackedWriter::new(&mut bs).write(&value::ser::to_value(&v)).unwrap();
        bs
    }));
}

pub fn large_testdata_decoder_with_ann(c: &mut Criterion) {
    c.bench_function("decode testdata.bin with annotations", |b| {
        let mut fh = std::fs::File::open("benches/testdata.bin").unwrap();
        let mut bs = vec![];
        fh.read_to_end(&mut bs).ok();
        b.iter(|| {
            let mut r = PackedReader::decode_bytes(&bs[..]);
            while let Some(_) = r.next(true).unwrap() {}
        })
    });
}

pub fn large_testdata_decoder_without_ann(c: &mut Criterion) {
    c.bench_function("decode testdata.bin without annotations", |b| {
        let mut fh = std::fs::File::open("benches/testdata.bin").unwrap();
        let mut bs = vec![];
        fh.read_to_end(&mut bs).ok();
        b.iter(|| {
            let mut r = PackedReader::decode_bytes(&bs[..]);
            while let Some(_) = r.next(false).unwrap() {}
        })
    });
}

pub fn large_testdata_encoder(c: &mut Criterion) {
    c.bench_function("encode testdata.bin", |b| {
        let mut fh = BufReader::new(std::fs::File::open("benches/testdata.bin").unwrap());
        let mut vs = vec![];
        let mut r = PackedReader::decode_read(&mut fh);
        while let Some(v) = r.next(true).unwrap() {
            vs.push(v);
        }
        b.iter_with_large_drop(|| {
            let mut bs = vec![];
            let mut w = PackedWriter::new(&mut bs);
            for v in &vs {
                w.write(&v).unwrap();
            }
            bs
        })
    });
}

criterion_group!(codec,
                 bench_decoder_bytes, bench_decoder_file, bench_decoder_buffered_file,
                 bench_encoder);
criterion_group!(serde, bench_de, bench_ser);
criterion_group!(codec_then_serde, bench_decoder_de, bench_ser_encoder);
criterion_group!{
    name = large_testdata;
    config = Criterion::default().sample_size(10);
    targets = large_testdata_decoder_with_ann, large_testdata_decoder_without_ann, large_testdata_encoder
}
criterion_main!(codec, serde, codec_then_serde, large_testdata);
