use num::bigint::BigInt;
use num::traits::cast::ToPrimitive;
use std::borrow::Cow;
use std::cmp::Ordering;
use std::convert::TryFrom;
use std::convert::TryInto;
use std::fmt::Debug;
use std::hash::{Hash, Hasher};
use std::marker::PhantomData;
use std::ops::Index;
use std::ops::IndexMut;
use std::string::String;
use std::vec::Vec;

pub use std::collections::BTreeSet as Set;
pub use std::collections::BTreeMap as Map;

use super::signed_integer::SignedInteger;
use crate::error::{Error, ExpectedKind, Received};

pub trait Domain: Sized + Debug + Clone + Eq + Hash + Ord {
    fn from_preserves(v: IOValue) -> Self;
    fn as_preserves(&self) -> IOValue;
}

pub trait NestedValue<D: Domain>: Sized + Debug + Clone + Eq + Hash + Ord {
    fn wrap(anns: Annotations<Self, D>, v: Value<Self, D>) -> Self;

    fn annotations(&self) -> &Annotations<Self, D>;
    fn value(&self) -> &Value<Self, D>;
    fn pieces(self) -> (Annotations<Self, D>, Value<Self, D>);
    fn value_owned(self) -> Value<Self, D>;

    fn debug_fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        for ann in self.annotations().slice() {
            write!(f, "@{:?} ", ann)?;
        }
        self.value().fmt(f)
    }

    fn copy_via<M: NestedValue<E>, E: Domain, F>(&self, f: &F) -> M
    where
        F: Fn(&D) -> Value<M, E>
    {
        M::wrap(self.annotations().copy_via(f), self.value().copy_via(f))
    }

    fn to_io_value(&self) -> IOValue {
        self.copy_via(&|d| d.as_preserves().value().clone())
    }

    fn from_io_value(v: &IOValue) -> Self {
        v.copy_via(&|d| Value::Domain(D::from_preserves(d.clone())))
    }
}

/// The `Value`s from the specification.
#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub enum Value<N, D> where N: NestedValue<D>, D: Domain {
    Boolean(bool),
    Float(Float),
    Double(Double),
    SignedInteger(SignedInteger),
    String(String),
    ByteString(Vec<u8>),
    Symbol(String),
    Record(Record<N>),
    Sequence(Vec<N>),
    Set(Set<N>),
    Dictionary(Map<N, N>),
    Domain(D),
}

/// Single-precision IEEE 754 Value
#[derive(Clone, Debug)]
pub struct Float(pub f32);

/// Double-precision IEEE 754 Value
#[derive(Clone, Debug)]
pub struct Double(pub f64);

/// A Record `Value` -- INVARIANT: length always non-zero
#[derive(Clone, Debug, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub struct Record<N>(pub Vec<N>);

impl<N> Record<N> {
    pub fn label(&self) -> &N {
        &self.0[0]
    }

    pub fn label_mut(&mut self) -> &mut N {
        &mut self.0[0]
    }

    pub fn arity(&self) -> usize {
        self.0.len() - 1
    }

    pub fn fields(&self) -> &[N] {
        &self.0[1..]
    }

    pub fn fields_mut(&mut self) -> &mut [N] {
        &mut self.0[1..]
    }

    pub fn fields_vec(&self) -> &Vec<N> {
        &self.0
    }

    pub fn fields_vec_mut(&mut self) -> &mut Vec<N> {
        &mut self.0
    }

    pub fn finish<D: Domain>(self) -> Value<N, D> where N: NestedValue<D> {
        Value::Record(self)
    }
}

impl From<f32> for Float {
    fn from(v: f32) -> Self {
        Float(v)
    }
}

impl From<Float> for f32 {
    fn from(v: Float) -> Self {
        v.0
    }
}

impl Hash for Float {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.0.to_bits().hash(state);
    }
}

impl PartialEq for Float {
    fn eq(&self, other: &Self) -> bool {
        self.0.to_bits() == other.0.to_bits()
    }
}

impl Ord for Float {
    fn cmp(&self, other: &Self) -> Ordering {
        let mut a: u32 = self.0.to_bits();
        let mut b: u32 = other.0.to_bits();
        if a & 0x8000_0000 != 0 { a ^= 0x7fff_ffff; }
        if b & 0x8000_0000 != 0 { b ^= 0x7fff_ffff; }
        (a as i32).cmp(&(b as i32))
    }
}

impl PartialOrd for Float {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Eq for Float {}

impl From<f64> for Double {
    fn from(v: f64) -> Self {
        Double(v)
    }
}

impl From<Double> for f64 {
    fn from(v: Double) -> Self {
        v.0
    }
}

impl Hash for Double {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.0.to_bits().hash(state);
    }
}

impl PartialEq for Double {
    fn eq(&self, other: &Self) -> bool {
        self.0.to_bits() == other.0.to_bits()
    }
}

impl Ord for Double {
    fn cmp(&self, other: &Self) -> Ordering {
        let mut a: u64 = self.0.to_bits();
        let mut b: u64 = other.0.to_bits();
        if a & 0x8000_0000_0000_0000 != 0 { a ^= 0x7fff_ffff_ffff_ffff; }
        if b & 0x8000_0000_0000_0000 != 0 { b ^= 0x7fff_ffff_ffff_ffff; }
        (a as i64).cmp(&(b as i64))
    }
}

impl PartialOrd for Double {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Eq for Double {}

impl<N: NestedValue<D>, D: Domain> From<bool> for Value<N, D> { fn from(v: bool) -> Self { Value::Boolean(v) } }

impl<N: NestedValue<D>, D: Domain> From<f32> for Value<N, D> { fn from(v: f32) -> Self { Value::Float(Float::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<f64> for Value<N, D> { fn from(v: f64) -> Self { Value::Double(Double::from(v)) } }

impl<N: NestedValue<D>, D: Domain> From<u8> for Value<N, D> { fn from(v: u8) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<i8> for Value<N, D> { fn from(v: i8) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<u16> for Value<N, D> { fn from(v: u16) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<i16> for Value<N, D> { fn from(v: i16) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<u32> for Value<N, D> { fn from(v: u32) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<i32> for Value<N, D> { fn from(v: i32) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<u64> for Value<N, D> { fn from(v: u64) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<i64> for Value<N, D> { fn from(v: i64) -> Self { Value::from(i128::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<u128> for Value<N, D> { fn from(v: u128) -> Self { Value::SignedInteger(SignedInteger::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<i128> for Value<N, D> { fn from(v: i128) -> Self { Value::SignedInteger(SignedInteger::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<&BigInt> for Value<N, D> { fn from(v: &BigInt) -> Self { Value::SignedInteger(SignedInteger::from(Cow::Borrowed(v))) } }

impl<N: NestedValue<D>, D: Domain> From<&str> for Value<N, D> { fn from(v: &str) -> Self { Value::String(String::from(v)) } }
impl<N: NestedValue<D>, D: Domain> From<String> for Value<N, D> { fn from(v: String) -> Self { Value::String(v) } }

impl<N: NestedValue<D>, D: Domain> From<&[u8]> for Value<N, D> { fn from(v: &[u8]) -> Self { Value::ByteString(Vec::from(v)) } }
// impl<N: NestedValue<D>, D: Domain> From<Vec<u8>> for Value<N, D> { fn from(v: Vec<u8>) -> Self { Value::ByteString(v) } }

impl<N: NestedValue<D>, D: Domain> From<Vec<N>> for Value<N, D> { fn from(v: Vec<N>) -> Self { Value::Sequence(v) } }
impl<N: NestedValue<D>, D: Domain> From<Set<N>> for Value<N, D> { fn from(v: Set<N>) -> Self { Value::Set(v) } }
impl<N: NestedValue<D>, D: Domain> From<Map<N, N>> for Value<N, D> { fn from(v: Map<N, N>) -> Self { Value::Dictionary(v) } }

impl<N: NestedValue<D>, D: Domain> Debug for Value<N, D> {
    // Not *quite* a formatter for the Preserves text syntax, since it
    // doesn't escape strings/symbols properly.
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Value::Boolean(false) => f.write_str("#f"),
            Value::Boolean(true) => f.write_str("#t"),
            Value::Float(Float(v)) => write!(f, "{:?}f", v),
            Value::Double(Double(v)) => write!(f, "{:?}", v),
            Value::SignedInteger(v) => write!(f, "{}", v),
            Value::String(ref v) => write!(f, "{:?}", v), // TODO: proper escaping!
            Value::ByteString(ref v) => {
                f.write_str("#x\"")?;
                for b in v { write!(f, "{:02x}", b)? }
                f.write_str("\"")
            }
            Value::Symbol(ref v) => {
                // TODO: proper escaping!
                if v.is_empty() {
                    f.write_str("||")
                } else {
                    write!(f, "{}", v)
                }
            }
            Value::Record(ref r) => {
                f.write_str("<")?;
                r.label().fmt(f)?;
                for v in r.fields() {
                    f.write_str(" ")?;
                    v.fmt(f)?;
                }
                f.write_str(">")
            }
            Value::Sequence(ref v) => v.fmt(f),
            Value::Set(ref v) => {
                f.write_str("#")?;
                f.debug_set().entries(v.iter()).finish()
            }
            Value::Dictionary(ref v) => f.debug_map().entries(v.iter()).finish(),
            Value::Domain(ref d) => write!(f, "#!{:?}", d),
        }
    }
}

//---------------------------------------------------------------------------

impl<N: NestedValue<D>, D: Domain> Value<N, D> {
    pub fn wrap(self) -> N {
        N::wrap(Annotations::empty(), self)
    }

    fn expected(&self, k: ExpectedKind) -> Error {
        Error::Expected(k, Received::ReceivedOtherValue(self.clone().wrap().to_io_value()))
    }

    pub fn is_boolean(&self) -> bool {
        self.as_boolean().is_some()
    }

    pub fn as_boolean(&self) -> Option<bool> {
        if let Value::Boolean(b) = *self {
            Some(b)
        } else {
            None
        }
    }

    pub fn as_boolean_mut(&mut self) -> Option<&mut bool> {
        if let Value::Boolean(ref mut b) = *self {
            Some(b)
        } else {
            None
        }
    }

    pub fn to_boolean(&self) -> Result<bool, Error> {
        self.as_boolean().ok_or_else(|| self.expected(ExpectedKind::Boolean))
    }

    pub fn is_float(&self) -> bool {
        self.as_float().is_some()
    }

    pub fn as_float(&self) -> Option<f32> {
        if let Value::Float(Float(f)) = *self {
            Some(f)
        } else {
            None
        }
    }

    pub fn as_float_mut(&mut self) -> Option<&mut f32> {
        if let Value::Float(Float(ref mut f)) = *self {
            Some(f)
        } else {
            None
        }
    }

    pub fn to_float(&self) -> Result<f32, Error> {
        self.as_float().ok_or_else(|| self.expected(ExpectedKind::Float))
    }

    pub fn is_double(&self) -> bool {
        self.as_double().is_some()
    }

    pub fn as_double(&self) -> Option<f64> {
        if let Value::Double(Double(f)) = *self {
            Some(f)
        } else {
            None
        }
    }

    pub fn as_double_mut(&mut self) -> Option<&mut f64> {
        if let Value::Double(Double(ref mut f)) = *self {
            Some(f)
        } else {
            None
        }
    }

    pub fn to_double(&self) -> Result<f64, Error> {
        self.as_double().ok_or_else(|| self.expected(ExpectedKind::Double))
    }

    pub fn is_signedinteger(&self) -> bool {
        self.as_signedinteger().is_some()
    }

    pub fn as_signedinteger(&self) -> Option<&SignedInteger> {
        if let Value::SignedInteger(ref n) = *self {
            Some(n)
        } else {
            None
        }
    }

    pub fn as_signedinteger_mut(&mut self) -> Option<&mut SignedInteger> {
        if let Value::SignedInteger(ref mut n) = *self {
            Some(n)
        } else {
            None
        }
    }

    pub fn to_signedinteger(&self) -> Result<&SignedInteger, Error> {
        self.as_signedinteger().ok_or_else(|| self.expected(ExpectedKind::SignedInteger))
    }

    pub fn is_i(&self) -> bool {
        self.as_i().is_some()
    }

    pub fn as_i(&self) -> Option<i128> {
        self.as_signedinteger().and_then(|n| n.try_into().ok())
    }

    pub fn to_i(&self) -> Result<i128, Error> {
        self.as_i().ok_or_else(|| self.expected(ExpectedKind::SignedIntegerI128))
    }

    pub fn is_u(&self) -> bool {
        self.as_u().is_some()
    }

    pub fn as_u(&self) -> Option<u128> {
        self.as_signedinteger().and_then(|n| n.try_into().ok())
    }

    pub fn to_u(&self) -> Result<u128, Error> {
        self.as_u().ok_or_else(|| self.expected(ExpectedKind::SignedIntegerU128))
    }

    pub fn as_u8(&self) -> Option<u8> { self.as_u().and_then(|i| i.to_u8()) }
    pub fn as_i8(&self) -> Option<i8> { self.as_i().and_then(|i| i.to_i8()) }
    pub fn as_u16(&self) -> Option<u16> { self.as_u().and_then(|i| i.to_u16()) }
    pub fn as_i16(&self) -> Option<i16> { self.as_i().and_then(|i| i.to_i16()) }
    pub fn as_u32(&self) -> Option<u32> { self.as_u().and_then(|i| i.to_u32()) }
    pub fn as_i32(&self) -> Option<i32> { self.as_i().and_then(|i| i.to_i32()) }
    pub fn as_u64(&self) -> Option<u64> { self.as_u().and_then(|i| i.to_u64()) }
    pub fn as_i64(&self) -> Option<i64> { self.as_i().and_then(|i| i.to_i64()) }
    pub fn as_u128(&self) -> Option<u128> { self.as_u().and_then(|i| i.to_u128()) }
    pub fn as_i128(&self) -> Option<i128> { self.as_i().and_then(|i| i.to_i128()) }

    pub fn to_i8(&self) -> Result<i8, Error> {
        match self.as_i() {
            Some(i) => i.to_i8().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_u8(&self) -> Result<u8, Error> {
        match self.as_u() {
            Some(i) => i.to_u8().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_i16(&self) -> Result<i16, Error> {
        match self.as_i() {
            Some(i) => i.to_i16().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_u16(&self) -> Result<u16, Error> {
        match self.as_u() {
            Some(i) => i.to_u16().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_i32(&self) -> Result<i32, Error> {
        match self.as_i() {
            Some(i) => i.to_i32().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_u32(&self) -> Result<u32, Error> {
        match self.as_u() {
            Some(i) => i.to_u32().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_i64(&self) -> Result<i64, Error> {
        match self.as_i() {
            Some(i) => i.to_i64().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_u64(&self) -> Result<u64, Error> {
        match self.as_u() {
            Some(i) => i.to_u64().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_i128(&self) -> Result<i128, Error> {
        match self.as_i() {
            Some(i) => i.to_i128().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_u128(&self) -> Result<u128, Error> {
        match self.as_u() {
            Some(i) => i.to_u128().ok_or_else(|| Error::NumberOutOfRange(BigInt::from(i))),
            None => Err(self.expected(ExpectedKind::SignedInteger)),
        }
    }

    pub fn to_char(&self) -> Result<char, Error> {
        let fs = self.to_simple_record("UnicodeScalar", Some(1))?;
        let c = fs[0].value().to_u32()?;
        char::try_from(c).map_err(|_| Error::InvalidUnicodeScalar(c))
    }

    pub fn is_string(&self) -> bool {
        self.as_string().is_some()
    }

    pub fn as_string(&self) -> Option<&String> {
        if let Value::String(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_string_mut(&mut self) -> Option<&mut String> {
        if let Value::String(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_string(&self) -> Result<&String, Error> {
        self.as_string().ok_or_else(|| self.expected(ExpectedKind::String))
    }

    pub fn is_bytestring(&self) -> bool {
        self.as_bytestring().is_some()
    }

    pub fn as_bytestring(&self) -> Option<&Vec<u8>> {
        if let Value::ByteString(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_bytestring_mut(&mut self) -> Option<&mut Vec<u8>> {
        if let Value::ByteString(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_bytestring(&self) -> Result<&Vec<u8>, Error> {
        self.as_bytestring().ok_or_else(|| self.expected(ExpectedKind::ByteString))
    }

    pub fn symbol(s: &str) -> Value<N, D> {
        Value::Symbol(s.to_string())
    }

    pub fn is_symbol(&self) -> bool {
        self.as_symbol().is_some()
    }

    pub fn as_symbol(&self) -> Option<&String> {
        if let Value::Symbol(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_symbol_mut(&mut self) -> Option<&mut String> {
        if let Value::Symbol(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_symbol(&self) -> Result<&String, Error> {
        self.as_symbol().ok_or_else(|| self.expected(ExpectedKind::Symbol))
    }

    pub fn record(label: N, expected_arity: usize) -> Record<N> {
        let mut v = Vec::with_capacity(expected_arity + 1);
        v.push(label);
        Record(v)
    }

    pub fn is_record(&self) -> bool {
        matches!(*self, Value::Record(_))
    }

    pub fn as_record(&self, arity: Option<usize>) -> Option<&Record<N>> {
        if let Value::Record(ref r) = *self {
            match arity {
                Some(expected) if r.arity() == expected => Some(r),
                Some(_other) => None,
                None => Some(r)
            }
        } else {
            None
        }
    }

    pub fn as_record_mut(&mut self, arity: Option<usize>) -> Option<&mut Record<N>> {
        if let Value::Record(ref mut r) = *self {
            match arity {
                Some(expected) if r.arity() == expected => Some(r),
                Some(_other) => None,
                None => Some(r)
            }
        } else {
            None
        }
    }

    pub fn to_record(&self, arity: Option<usize>) -> Result<&Record<N>, Error> {
        self.as_record(arity).ok_or_else(|| self.expected(ExpectedKind::Record(arity)))
    }

    pub fn simple_record(label: &str, expected_arity: usize) -> Record<N> {
        Self::record(Value::symbol(label).wrap(), expected_arity)
    }

    pub fn simple_record0(label: &str) -> Value<N, D> {
        Self::simple_record(label, 0).finish()
    }

    pub fn simple_record1(label: &str, field: N) -> Value<N, D> {
        let mut r = Self::simple_record(label, 1);
        r.fields_vec_mut().push(field);
        r.finish()
    }

    pub fn is_simple_record(&self, label: &str, arity: Option<usize>) -> bool {
        self.as_simple_record(label, arity).is_some()
    }

    pub fn as_simple_record(&self, label: &str, arity: Option<usize>) -> Option<&[N]> {
        self.as_record(arity).and_then(|r| {
            match r.label().value() {
                Value::Symbol(ref s) if s == label => Some(r.fields()),
                _ => None
            }
        })
    }

    pub fn to_simple_record(&self, label: &str, arity: Option<usize>) ->
        Result<&[N], Error>
    {
        self.as_simple_record(label, arity)
            .ok_or_else(|| self.expected(ExpectedKind::SimpleRecord(label.to_owned(), arity)))
    }

    pub fn to_option(&self) -> Result<Option<&N>, Error> {
        match self.as_simple_record("None", Some(0)) {
            Some(_fs) => Ok(None),
            None => match self.as_simple_record("Some", Some(1)) {
                Some(fs) => Ok(Some(&fs[0])),
                None => Err(self.expected(ExpectedKind::Option))
            }
        }
    }

    pub fn is_sequence(&self) -> bool {
        self.as_sequence().is_some()
    }

    pub fn as_sequence(&self) -> Option<&Vec<N>> {
        if let Value::Sequence(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_sequence_mut(&mut self) -> Option<&mut Vec<N>> {
        if let Value::Sequence(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_sequence(&self) -> Result<&Vec<N>, Error> {
        self.as_sequence().ok_or_else(|| self.expected(ExpectedKind::Sequence))
    }

    pub fn is_set(&self) -> bool {
        self.as_set().is_some()
    }

    pub fn as_set(&self) -> Option<&Set<N>> {
        if let Value::Set(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_set_mut(&mut self) -> Option<&mut Set<N>> {
        if let Value::Set(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_set(&self) -> Result<&Set<N>, Error> {
        self.as_set().ok_or_else(|| self.expected(ExpectedKind::Set))
    }

    pub fn is_dictionary(&self) -> bool {
        self.as_dictionary().is_some()
    }

    pub fn as_dictionary(&self) -> Option<&Map<N, N>> {
        if let Value::Dictionary(ref s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn as_dictionary_mut(&mut self) -> Option<&mut Map<N, N>> {
        if let Value::Dictionary(ref mut s) = *self {
            Some(s)
        } else {
            None
        }
    }

    pub fn to_dictionary(&self) -> Result<&Map<N, N>, Error> {
        self.as_dictionary().ok_or_else(|| self.expected(ExpectedKind::Dictionary))
    }

    pub fn copy_via<M: NestedValue<E>, E: Domain, F>(&self, f: &F) -> Value<M, E>
    where
        F: Fn(&D) -> Value<M, E>
    {
        match self {
            Value::Boolean(b) => Value::Boolean(*b),
            Value::Float(f) => Value::Float(f.clone()),
            Value::Double(d) => Value::Double(d.clone()),
            Value::SignedInteger(ref n) => Value::SignedInteger(n.clone()),
            Value::String(ref s) => Value::String(s.clone()),
            Value::ByteString(ref v) => Value::ByteString(v.clone()),
            Value::Symbol(ref v) => Value::Symbol(v.clone()),
            Value::Record(ref r) =>
                Value::Record(Record(r.fields_vec().iter().map(|a| a.copy_via(f)).collect())),
            Value::Sequence(ref v) => Value::Sequence(v.iter().map(|a| a.copy_via(f)).collect()),
            Value::Set(ref v) => Value::Set(v.iter().map(|a| a.copy_via(f)).collect()),
            Value::Dictionary(ref v) =>
                Value::Dictionary(v.iter().map(|(a,b)| (a.copy_via(f), b.copy_via(f))).collect()),
            Value::Domain(d) => f(d),
        }
    }
}

impl<N: NestedValue<D>, D: Domain> Index<usize> for Value<N, D> {
    type Output = N;

    fn index(&self, i: usize) -> &Self::Output {
        &self.as_sequence().unwrap()[i]
    }
}

impl<N: NestedValue<D>, D: Domain> IndexMut<usize> for Value<N, D> {
    fn index_mut(&mut self, i: usize) -> &mut Self::Output {
        &mut self.as_sequence_mut().unwrap()[i]
    }
}

impl<N: NestedValue<D>, D: Domain> Index<&N> for Value<N, D> {
    type Output = N;

    fn index(&self, i: &N) -> &Self::Output {
        &(*self.as_dictionary().unwrap())[i]
    }
}

//---------------------------------------------------------------------------
// This part is a terrible hack

impl serde::Serialize for UnwrappedIOValue {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: serde::Serializer {
        super::magic::output_value(serializer, self.clone().wrap())
    }
}

impl<'de> serde::Deserialize<'de> for UnwrappedIOValue {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::Deserializer<'de> {
        Ok(super::magic::input_value::<'de, D>(deserializer)?.value_owned())
    }
}

pub fn serialize_nested_value<S, N, D: Domain>(v: &N, serializer: S) ->
    Result<S::Ok, S::Error> where S: serde::Serializer, N: NestedValue<D> + serde::Serialize
{
    super::magic::output_value(serializer, v.to_io_value())
}

pub fn deserialize_nested_value<'de, D, N, Dom: Domain>(deserializer: D) ->
    Result<N, D::Error> where D: serde::Deserializer<'de>, N: NestedValue<Dom> + serde::Deserialize<'de>
{
    Ok(N::from_io_value(&super::magic::input_value(deserializer)?))
}

//---------------------------------------------------------------------------

#[derive(Clone)]
pub struct Annotations<N: NestedValue<D>, D: Domain>(Option<Vec<N>>, PhantomData<D>);

impl<N: NestedValue<D>, D: Domain> Annotations<N, D> {
    pub fn empty() -> Self {
        Annotations(None,
                    PhantomData)
    }

    pub fn new(anns: Option<Vec<N>>) -> Self {
        Annotations(anns, PhantomData)
    }

    pub fn maybe_slice(&self) -> Option<&[N]> {
        match self.0 {
            None => None,
            Some(ref b) => Some(&&b[..]),
        }
    }

    pub fn slice(&self) -> &[N] {
        self.maybe_slice().unwrap_or(&[])
    }

    pub fn modify<F>(&mut self, f: F) -> &mut Self
    where
        F: FnOnce(&mut Vec<N>)
    {
        let mut v: Vec<N> = self.0.take().unwrap_or_else(Vec::new);
        f(&mut v);
        self.0 = if v.is_empty() { None } else { Some(v) };
        self
    }

    pub fn copy_via<M: NestedValue<E>, E: Domain, F>(&self, f: &F) -> Annotations<M, E>
    where
        F: Fn(&D) -> Value<M, E>
    {
        match self.0 {
            None =>
                Annotations(None, PhantomData),
            Some(ref b) =>
                Annotations(Some(b.iter().map(|a| a.copy_via(f)).collect()), PhantomData),
        }
    }
}

/// A possibly-annotated Value, with annotations (themselves
/// possibly-annotated) in order of appearance.
#[derive(Clone)]
pub struct AnnotatedValue<N: NestedValue<D>, D: Domain>(pub Annotations<N, D>, pub Value<N, D>);

impl<N: NestedValue<D>, D: Domain> AnnotatedValue<N, D> {
    fn new(anns: Annotations<N, D>, value: Value<N, D>) -> Self {
        AnnotatedValue(anns, value)
    }
}

impl<N: NestedValue<D>, D: Domain> PartialEq for AnnotatedValue<N, D> {
    fn eq(&self, other: &Self) -> bool {
        self.1.eq(&other.1)
    }
}

impl<N: NestedValue<D>, D: Domain> Eq for AnnotatedValue<N, D> {}

impl<N: NestedValue<D>, D: Domain> Hash for AnnotatedValue<N, D> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.1.hash(state);
    }
}

impl<N: NestedValue<D>, D: Domain> PartialOrd for AnnotatedValue<N, D> {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl<N: NestedValue<D>, D: Domain> Ord for AnnotatedValue<N, D> {
    fn cmp(&self, other: &Self) -> Ordering {
        self.1.cmp(&other.1)
    }
}

//---------------------------------------------------------------------------

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub struct PlainValue<D: Domain>(AnnotatedValue<PlainValue<D>, D>);

impl<D: Domain> PlainValue<D> {
    pub fn annotations_mut(&mut self) -> &mut Annotations<Self, D> {
        &mut (self.0).0
    }

    pub fn value_mut(&mut self) -> &mut Value<Self, D> {
        &mut (self.0).1
    }
}

impl<D: Domain> NestedValue<D> for PlainValue<D> {
    fn wrap(anns: Annotations<Self, D>, v: Value<Self, D>) -> Self {
        PlainValue(AnnotatedValue::new(anns, v))
    }

    fn annotations(&self) -> &Annotations<Self, D> {
        &(self.0).0
    }

    fn value(&self) -> &Value<Self, D> {
        &(self.0).1
    }

    fn pieces(self) -> (Annotations<Self, D>, Value<Self, D>) {
        let AnnotatedValue(anns, v) = self.0;
        (anns, v)
    }

    fn value_owned(self) -> Value<Self, D> {
        (self.0).1
    }
}

impl<D: Domain> Debug for PlainValue<D> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.debug_fmt(f)
    }
}

impl<D: Domain> serde::Serialize for PlainValue<D> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: serde::Serializer {
        serialize_nested_value(self, serializer)
    }
}

impl<'de, Dom: Domain> serde::Deserialize<'de> for PlainValue<Dom> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::Deserializer<'de> {
        deserialize_nested_value(deserializer)
    }
}

//---------------------------------------------------------------------------

use std::rc::Rc;

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub struct RcValue<D: Domain>(Rc<AnnotatedValue<RcValue<D>, D>>);

impl<D: Domain> NestedValue<D> for RcValue<D> {
    fn wrap(anns: Annotations<Self, D>, v: Value<Self, D>) -> Self {
        RcValue(Rc::new(AnnotatedValue::new(anns, v)))
    }

    fn annotations(&self) -> &Annotations<Self, D> {
        &(self.0).0
    }

    fn value(&self) -> &Value<Self, D> {
        &(self.0).1
    }

    fn pieces(self) -> (Annotations<Self, D>, Value<Self, D>) {
        match Rc::try_unwrap(self.0) {
            Ok(AnnotatedValue(anns, v)) => (anns, v),
            Err(r) => (r.0.clone(), r.1.clone()),
        }
    }

    fn value_owned(self) -> Value<Self, D> {
        Rc::try_unwrap(self.0).unwrap_or_else(|_| panic!("value_owned on RcValue with refcount greater than one")).1
    }
}

impl<D: Domain> Debug for RcValue<D> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.debug_fmt(f)
    }
}

impl<D: Domain> serde::Serialize for RcValue<D> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: serde::Serializer {
        serialize_nested_value(self, serializer)
    }
}

impl<'de, Dom: Domain> serde::Deserialize<'de> for RcValue<Dom> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::Deserializer<'de> {
        deserialize_nested_value(deserializer)
    }
}

//---------------------------------------------------------------------------

use std::sync::Arc;

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub struct ArcValue<D: Domain>(Arc<AnnotatedValue<ArcValue<D>, D>>);

impl<D: Domain> NestedValue<D> for ArcValue<D> {
    fn wrap(anns: Annotations<Self, D>, v: Value<Self, D>) -> Self {
        ArcValue(Arc::new(AnnotatedValue::new(anns, v)))
    }

    fn annotations(&self) -> &Annotations<Self, D> {
        &(self.0).0
    }

    fn value(&self) -> &Value<Self, D> {
        &(self.0).1
    }

    fn pieces(self) -> (Annotations<Self, D>, Value<Self, D>) {
        match Arc::try_unwrap(self.0) {
            Ok(AnnotatedValue(anns, v)) => (anns, v),
            Err(r) => (r.0.clone(), r.1.clone()),
        }
    }

    fn value_owned(self) -> Value<Self, D> {
        Arc::try_unwrap(self.0).unwrap_or_else(|_| panic!("value_owned on ArcValue with refcount greater than one")).1
    }
}

impl<D: Domain> Debug for ArcValue<D> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.debug_fmt(f)
    }
}

impl<D: Domain> serde::Serialize for ArcValue<D> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: serde::Serializer {
        serialize_nested_value(self, serializer)
    }
}

impl<'de, Dom: Domain> serde::Deserialize<'de> for ArcValue<Dom> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::Deserializer<'de> {
        deserialize_nested_value(deserializer)
    }
}

//---------------------------------------------------------------------------

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub struct IOValue(Arc<AnnotatedValue<IOValue, IOValue>>);
pub type UnwrappedIOValue = Value<IOValue, IOValue>;

impl Domain for IOValue {
    fn from_preserves(v: IOValue) -> Self {
        v
    }

    fn as_preserves(&self) -> IOValue {
        self.clone()
    }
}

lazy_static! {
    pub static ref FALSE: IOValue = IOValue(Arc::new(AnnotatedValue(Annotations::empty(), Value::Boolean(false))));
    pub static ref TRUE: IOValue = IOValue(Arc::new(AnnotatedValue(Annotations::empty(), Value::Boolean(true))));
    pub static ref EMPTY_SEQ: IOValue = IOValue(Arc::new(AnnotatedValue(Annotations::empty(), Value::Sequence(Vec::new()))));
}

impl NestedValue<IOValue> for IOValue {
    fn wrap(anns: Annotations<Self, IOValue>, v: Value<Self, IOValue>) -> Self {
        IOValue(Arc::new(AnnotatedValue::new(anns, v)))
    }

    fn annotations(&self) -> &Annotations<Self, IOValue> {
        &(self.0).0
    }

    fn value(&self) -> &Value<Self, IOValue> {
        &(self.0).1
    }

    fn pieces(self) -> (Annotations<Self, IOValue>, Value<Self, IOValue>) {
        match Arc::try_unwrap(self.0) {
            Ok(AnnotatedValue(anns, v)) => (anns, v),
            Err(r) => (r.0.clone(), r.1.clone()),
        }
    }

    fn value_owned(self) -> Value<Self, IOValue> {
        match Arc::try_unwrap(self.0) {
            Ok(AnnotatedValue(_anns, v)) => v,
            Err(r) => r.1.clone(),
        }
    }

    fn to_io_value(&self) -> IOValue {
        self.clone()
    }

    fn from_io_value(v: &IOValue) -> Self {
        v.clone()
    }
}

impl Debug for IOValue {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.debug_fmt(f)
    }
}

impl serde::Serialize for IOValue {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error> where S: serde::Serializer {
        serialize_nested_value(self, serializer)
    }
}

impl<'de> serde::Deserialize<'de> for IOValue {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error> where D: serde::Deserializer<'de> {
        deserialize_nested_value(deserializer)
    }
}
