pub mod types;
pub mod names;
pub mod context;
pub mod parsers;
pub mod unparsers;
pub mod codegen;

use std::path::PathBuf;
use preserves::value::Map;

use crate::syntax::block::constructors::*;
use crate::metaschema::*;

pub type ModulePath = Vec<String>;

#[derive(Debug)]
pub struct CompilerConfig {
    pub bundle: Map<ModulePath, Schema>,
    pub output_dir: PathBuf,
}

impl CompilerConfig {
    pub fn new(output_dir: PathBuf) -> Self {
        CompilerConfig {
            bundle: Map::new(),
            output_dir: output_dir,
        }
    }
}

pub fn compile(config: &CompilerConfig) -> Result<(), std::io::Error> {
    for (k, v) in config.bundle.iter() {
        let mut output_path = config.output_dir.clone();
        output_path.extend(k);
        output_path.set_extension("rs");
        let mut m = context::ModuleContext::new();

        // println!("\n{:?}", &output_path);

        // TODO: embedded type

        for (n, d) in &v.definitions.0 {
            m.define_type(item(types::render_definition_type(n, &types::definition_type(d))));
            parsers::gen_definition_parser(&mut m, n, d);
            unparsers::gen_definition_unparser(&mut m, n, d);
        }

        //---------------------------------------------------------------------------

        println!("#![allow(unused_parens)]");
        println!();
        println!("use std::convert::TryFrom;");
        println!("use preserves::value::NestedValue;");
        println!("use lazy_static::lazy_static;");
        println!();

        println!("lazy_static! {{");
        for (value, name) in m.literals {
            let bs = preserves::value::PackedWriter::encode(&value).unwrap();
            println!("    pub static ref {}: preserves::value::IOValue = /* {:?} */ preserves::value::packed::from_bytes(&vec!{:?}).unwrap();",
                     name,
                     value,
                     bs);
        }
        println!("}}\n");

        for i in m.typedefs { println!("{:?}\n", i); }
        for i in m.functiondefs { println!("{:?}\n", i); }
    }
    Ok(())
}
