use preserves::value::{Map, IOValue};

use crate::syntax::block::Item;

use super::types;

pub struct ModuleContext {
    pub literals: Map<IOValue, String>,
    pub typedefs: Vec<Item>,
    pub functiondefs: Vec<Item>,
}

pub struct FunctionContext<'a> {
    pub m: &'a mut ModuleContext,
    pub temp_counter: usize,
    pub captures: Vec<Capture>,
}

pub struct Capture {
    pub field_name: String,
    pub ty: types::TField,
    pub source_expr: String,
}

impl ModuleContext {
    pub fn new() -> Self {
        ModuleContext {
            literals: Map::new(),
            typedefs: Vec::new(),
            functiondefs: Vec::new(),
        }
    }

    pub fn define_literal(&mut self, v: &IOValue) -> String {
        let next_id = format!("LIT{}", self.literals.len());
        "&*".to_owned() + self.literals.entry(v.clone()).or_insert(next_id)
    }

    pub fn define_type(&mut self, i: Item) {
        self.typedefs.push(i)
    }

    pub fn define_function<F: FnOnce(FunctionContext) -> Item>(&mut self, f: F) {
        let i = f(FunctionContext::new(self));
        self.functiondefs.push(i)
    }
}

impl<'a> FunctionContext<'a> {
    pub fn new(m: &'a mut ModuleContext) -> Self {
        FunctionContext {
            m: m,
            temp_counter: 0,
            captures: Vec::new(),
        }
    }

    pub fn capture(&mut self, field_name: String, ty: types::TField, source_expr: String) {
        self.captures.push(Capture {
            field_name: field_name,
            ty: ty,
            source_expr: source_expr,
        })
    }

    pub fn lookup_capture(&self, field_name: &str) -> &Capture {
        for c in &self.captures {
            if c.field_name == field_name {
                return c;
            }
        }
        panic!("No capture for field {:?} available", field_name)
    }

    pub fn gentempname(&mut self) -> String {
        let i = self.temp_counter;
        self.temp_counter += 1;
        format!("_tmp{}", i)
    }

    pub fn branch<R, F: FnOnce(&mut Self) -> R>(&mut self, f: F) -> R {
        let saved_temp_counter = self.temp_counter;
        let saved_capture_count = self.captures.len();
        let result = f(self);
        self.temp_counter = saved_temp_counter;
        self.captures.truncate(saved_capture_count);
        result
    }
}
