use glob::glob;
use preserves::value::packed::PackedReader;
use preserves::value::Reader;
use std::convert::TryFrom;
use std::fs::File;
use std::io::Error;
use std::io::ErrorKind;
use std::path::PathBuf;
use structopt::StructOpt;

use preserves_schema::compiler::{CompilerConfig, compile};
use preserves_schema::metaschema::*;

#[derive(Clone, StructOpt, Debug)]
struct CommandLine {
    #[structopt(short, long)]
    output_dir: PathBuf,

    #[structopt(long)]
    prefix: Option<String>,

    input_glob: Vec<String>,
}

fn inputs(globs: &Vec<String>) -> Vec<PathBuf> {
    let mut result = Vec::new();
    for g in globs.iter() {
        match glob(g) {
            Ok(paths) =>
                for p in paths {
                    match p {
                        Ok(s) => result.push(s),
                        Err(e) => println!("warning: {:?}", e),
                    }
                }
            Err(e) => println!("warning: {:?}", e),
        }
    }
    result
}

fn main() -> Result<(), Error> {
    let args = CommandLine::from_args();

    let prefix = match &args.prefix {
        Some(s) => s.split(".").map(str::to_string).collect(),
        None => vec![],
    };

    let mut config = CompilerConfig::new(args.output_dir);

    for i in inputs(&args.input_glob) {
        let mut f = File::open(&i)?;
        let mut reader = PackedReader::decode_read(&mut f);
        let schema = reader.demand_next(false)?;
        match Schema::try_from(&schema) {
            Ok(s) => {
                config.bundle.insert(prefix.clone(), s);
                ()
            },
            Err(()) => match Bundle::try_from(&schema) {
                Ok(Bundle { modules }) => {
                    for (ModulePath(k), v) in modules.0 {
                        let mut name = prefix.clone();
                        name.extend(k);
                        config.bundle.insert(name, v);
                    }
                },
                Err(()) => return Err(ErrorKind::InvalidData)?,
            },
        }
    }

    compile(&config)
}
