use crate::file_writer;
use crate::ucd_parsers;
use crate::Error;
use std::fs::File;
use std::io::Write;
use std::path::Path;
use ucd_parse::{CodepointRange, Codepoints};

pub struct BidiClassGen {}

fn add_range(range: &CodepointRange, bidi: &str, vec: &mut Vec<(Codepoints, String)>) {
    if range.start.value() == range.end.value() {
        vec.push((Codepoints::Single(range.start), String::from(bidi)));
    } else {
        vec.push((Codepoints::Range(*range), String::from(bidi)));
    }
}

impl BidiClassGen {
    pub fn generate_file(ucd_dir: &Path, out_dir: &Path, file_name: &str) -> Result<(), Error> {
        let dest_path = out_dir.join(file_name);
        let mut file = File::create(dest_path)?;

        file_writer::generate_file_header(&mut file)?;
        BidiClassGen::generate_bidi_class_enum(&mut file)?;
        BidiClassGen::generate_bidi_class_table(&mut file, ucd_dir)
    }

    fn generate_bidi_class_table(file: &mut File, ucd_dir: &Path) -> Result<(), Error> {
        let mut vec = Vec::new();
        let ucds: Vec<ucd_parsers::UnicodeData> = ucd_parsers::UnicodeData::parse(ucd_dir)?;

        for udata in ucds.iter() {
            vec.push((udata.codepoints, udata.bidi_class.clone()));
        }

        let vec = BidiClassGen::compress_into_ranges(&vec);
        BidiClassGen::write_table_to_file(file, &vec)
    }

    fn compress_into_ranges(input: &[(Codepoints, String)]) -> Vec<(Codepoints, String)> {
        let mut out = Vec::new();
        let mut range: Option<CodepointRange> = None;
        let mut val: Option<String> = None;

        for (cp, bidi) in input.iter() {
            if val.is_none() {
                val = Some(bidi.clone());
            }

            if val.as_ref() != Some(bidi) {
                // No same bidi class
                match range.as_ref() {
                    Some(r) => {
                        add_range(r, &val.unwrap(), &mut out);
                        range = None;
                    }
                    None => out.push((*cp, val.unwrap())),
                }
                val = Some(bidi.clone());
            }

            match cp {
                Codepoints::Single(cp) => {
                    match range.as_mut() {
                        Some(mut r) => {
                            if cp.value() - r.end.value() == 1 {
                                r.end = *cp;
                            } else {
                                // there is a gap, non consecutive numbers
                                add_range(r, bidi, &mut out);
                                // Start a new range
                                range = Some(CodepointRange {
                                    start: *cp,
                                    end: *cp,
                                });
                            }
                        }
                        None => {
                            range = Some(CodepointRange {
                                start: *cp,
                                end: *cp,
                            });
                        }
                    }
                }
                Codepoints::Range(cp) => {
                    match range.as_mut() {
                        Some(mut r) => {
                            if cp.start.value() - r.end.value() == 1 {
                                // This range can be included in the previous one
                                r.end = cp.end;
                            } else {
                                // no consecutive ranges
                                out.push((Codepoints::Range(*r), bidi.clone()));
                                out.push((Codepoints::Range(*cp), bidi.clone()));
                                range = None;
                            }
                        }
                        None => {
                            range = Some(*cp);
                        }
                    }
                }
            }
        }

        out
    }

    fn generate_bidi_class_enum(file: &mut File) -> Result<(), Error> {
        writeln!(
            file,
            "/// Represents values of the Unicode character property"
        )?;
        writeln!(
            file,
            "/// [Bidi_Class](http://www.unicode.org/reports/tr44/#Bidi_Class),"
        )?;
        writeln!(
            file,
            "/// also known as the bidirectional character type. Possible values"
        )?;
        writeln!(file,
		"/// are listed in [UAX44, Table 13](http://www.unicode.org/reports/tr44/#Bidi_Class_Values)"
		)?;
        writeln!(file)?;

        // Values in BidiClass enum are generated by parsing the UCD files, they are upper case
        // which goes against the rust naming rules. Let's allow it just for this time
        writeln!(file, "#[allow(clippy::upper_case_acronyms)]")?;

        writeln!(file, "#[derive(Copy, Clone, PartialEq, Debug)]")?;
        writeln!(file, "pub enum BidiClass {{")?;
        writeln!(file, "\tAL,")?;
        writeln!(file, "\tAN,")?;
        writeln!(file, "\tB,")?;
        writeln!(file, "\tBN,")?;
        writeln!(file, "\tCS,")?;
        writeln!(file, "\tEN,")?;
        writeln!(file, "\tES,")?;
        writeln!(file, "\tET,")?;
        writeln!(file, "\tFSI,")?;
        writeln!(file, "\tL,")?;
        writeln!(file, "\tLRE,")?;
        writeln!(file, "\tLRI,")?;
        writeln!(file, "\tLRO,")?;
        writeln!(file, "\tNSM,")?;
        writeln!(file, "\tON,")?;
        writeln!(file, "\tPDF,")?;
        writeln!(file, "\tPDI,")?;
        writeln!(file, "\tR,")?;
        writeln!(file, "\tRLE,")?;
        writeln!(file, "\tRLI,")?;
        writeln!(file, "\tRLO,")?;
        writeln!(file, "\tS,")?;
        writeln!(file, "\tWS,")?;
        writeln!(file, "}}")?;

        Ok(writeln!(file)?)
    }

    fn write_table_to_file(file: &mut File, vec: &[(Codepoints, String)]) -> Result<(), Error> {
        writeln!(
            file,
            "static BIDI_CLASS_TABLE: [(Codepoints, BidiClass); {}] = [",
            vec.len()
        )?;

        for (cp, bidi) in vec.iter() {
            writeln!(
                file,
                "\t({}, BidiClass::{}),",
                file_writer::generate_codepoint_str(cp),
                bidi
            )?;
        }

        writeln!(file, "];")?;
        Ok(writeln!(file)?)
    }
}
