// build.rs
use precis_tools::{CodeGenerator, UnicodeVersionGen};
use std::env;
use std::fs;
use std::path::Path;

const UNICODE_VERSION: &str = "6.3.0";

fn generate_code(ucd: &Path) {
    let gen = CodeGenerator::new(ucd);
    let out_dir = env::var_os("OUT_DIR").unwrap();
    let out_path = Path::new(&out_dir);
    gen.generate_definitions(out_path, "precis_defs.rs");
    gen.generate_code(out_path, "precis_tables.rs");

    UnicodeVersionGen::generate_code(out_path, UNICODE_VERSION, "unicode_version.rs").unwrap();
}

fn create_dir(path: &Path) {
    if !path.is_dir() {
        fs::create_dir(&path).unwrap();
    }
}

fn main() {
    let out_dir = env::var_os("OUT_DIR").unwrap();
    let ucd_path = Path::new(&out_dir).join("ucd");

    create_dir(&ucd_path);

    let csv_path = Path::new(&out_dir).join("csv");
    create_dir(&csv_path);

    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "UnicodeData.txt").unwrap();

    // JoinControl (H)
    // Noncharacter_Code_Point
    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "PropList.txt").unwrap();
    // 9.9.  OldHangulJamo (I)
    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "HangulSyllableType.txt")
        .unwrap();

    // Default_Ignorable_Code_Point
    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "DerivedCoreProperties.txt")
        .unwrap();

    // for long value aliases for General_Category values
    // Used to generate function names
    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "PropertyValueAliases.txt")
        .unwrap();

    // Required for context rules
    precis_tools::download::get_ucd_file(UNICODE_VERSION, &ucd_path, "Scripts.txt").unwrap();

    let extracted_path = ucd_path.join("extracted");
    create_dir(&extracted_path);
    precis_tools::download::get_ucd_file(
        UNICODE_VERSION,
        &ucd_path,
        "extracted/DerivedJoiningType.txt",
    )
    .unwrap();

    precis_tools::download::get_csv_file(UNICODE_VERSION, &csv_path).unwrap();

    generate_code(&ucd_path);

    println!("cargo:rerun-if-changed=build.rs");
}
