use std::error::Error;
use std::fmt;

/// A trait that describes a
/// [`Newtype`](https://rust-unofficial.github.io/patterns/patterns/behavioural/newtype.html)
/// wrapper struct generated by [`define!`](crate::define) and
/// [`extend!`](crate::extend) macros.
pub trait Wrapper: Sized {
    /// Name of the wrapper. Will be used for nice error messages.
    const NAME: &'static str;

    /// The inner type of the wrapper.
    type Inner;

    /// The type of an error that can occur during the construction or mutation
    /// of the wrapper's value.
    type Error;
    /// The function that will mutate and validate wrapper's inner value on
    /// every construction and mutation.
    ///
    /// It's behaviour is based on the closures that were provided during the
    /// invocation of [`define!`](crate::define)/[`extend!`](crate::extend)
    /// macros (e.g. `adjust`, `ensure` and `validate`).
    ///
    /// If no closures were provided, this function will not mutate the value
    /// and always return `Ok(())`.
    const PROCESS: fn(&mut Self::Inner) -> Result<(), Self::Error>;

    /// Construct a new wrapper.
    ///
    /// It will return an error if the provided `value` doesn't pass
    /// [`Self::PROCESS`](Self::PROCESS).
    fn new(value: impl Into<Self::Inner>) -> Result<Self, ConstructionError<Self>>;

    /// Get a shared reference to the inner value.
    fn get(&self) -> &Self::Inner;

    // TODO: maybe change `ConstructionError` to `ReplacementError`?
    /// Replace inner value with the provided one.
    ///
    /// It will return an error if the provided `value` doesn't pass
    /// [`Self::PROCESS`](Self::PROCESS).
    fn set(&mut self, value: impl Into<Self::Inner>) -> Result<(), ConstructionError<Self>>;

    /// Mutate inner value using provided closure.
    ///
    /// To make sure that the closure doesn't corrupt the inner value, this
    /// method is only available when the inner type implements
    /// [`Clone`](Clone). This way, the closure receives a copy of the inner
    /// value, and then, if the mutated value passes
    /// [`Self::PROCESS`](Self::PROCESS), it will replace the inner value.
    fn mutate(&mut self, f: impl FnOnce(&mut Self::Inner)) -> Result<(), MutationError<Self>>
    where
        Self::Inner: Clone,
    {
        self.__mutate_with(Self::Inner::clone, f)
    }

    /// This is a helper method that should be implemented in order for `mutate`
    /// method to work in a generic way. This method should not be used directly
    /// by the user (hence `#[doc(hidden)]` and a weird name).
    ///
    /// Using this method, we allowing the trait to be implemented for both
    /// `Clone` and `!Clone` types. `Clone` types will have the `mutate` method,
    /// while the `!Clone` types won't.
    ///
    /// The idea for this method originated here:
    /// https://users.rust-lang.org/t/is-it-possible-to-implement-trait-method-with-where-clause-on-a-concrete-type/72846/12?u=teenjuna
    #[doc(hidden)]
    fn __mutate_with(
        &mut self,
        _clone: impl Fn(&Self::Inner) -> Self::Inner,
        _f: impl FnOnce(&mut Self::Inner),
    ) -> Result<(), MutationError<Self>> {
        unimplemented!()
    }

    /// Construct a new wrapper without calling
    /// [`Self::PROCESS`](Self::PROCESS).
    #[cfg(feature = "unprocessed")]
    #[cfg_attr(docsrs, doc(cfg(feature = "unprocessed")))]
    fn new_unprocessed(value: impl Into<Self::Inner>) -> Self;

    /// Replace inner value with the provided one without calling
    /// [`Self::PROCESS`](Self::PROCESS).
    #[cfg(feature = "unprocessed")]
    #[cfg_attr(docsrs, doc(cfg(feature = "unprocessed")))]
    fn set_unprocessed(&mut self, value: impl Into<Self::Inner>);

    /// Mutate inner value using provided closure without calling
    /// [`Self::PROCESS`](Self::PROCESS).
    #[cfg(feature = "unprocessed")]
    #[cfg_attr(docsrs, doc(cfg(feature = "unprocessed")))]
    fn mutate_unprocessed(&mut self, f: impl FnOnce(&mut Self::Inner));

    /// Verify that inner value still passes [`Self::PROCESS`](Self::PROCESS).
    #[cfg(feature = "unprocessed")]
    #[cfg_attr(docsrs, doc(cfg(feature = "unprocessed")))]
    fn verify(self) -> Result<Self, VerificationError<Self>>;
}

/// A wrapper-error that will be returned if the
/// [`Wrapper::new`](crate::Wrapper::new) or
/// [`Wrapper::set`](crate::Wrapper::set) methods receive a value that doesn't
/// pass [`Wrapper::PROCESS`](crate::Wrapper::PROCESS) function.
///
/// This wrapper contains both the value that caused the error and the original
/// error returned by the [`Wrapper::PROCESS`](crate::Wrapper::PROCESS)
/// function.
#[derive(Debug)]
pub struct ConstructionError<W: Wrapper> {
    /// Value that caused the error.
    pub value: W::Inner,
    /// Original error.
    pub original: W::Error,
}

impl<W> fmt::Display for ConstructionError<W>
where
    W: Wrapper,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "failed to construct type {} from value {:?}: {}",
            W::NAME,
            self.value,
            self.original,
        )
    }
}

impl<W> Error for ConstructionError<W>
where
    W: Wrapper + fmt::Debug,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display + fmt::Debug,
{
    // NOTE: `self.error` could be used for `source` function.
    // However, it would require `W::Error: Error + 'static`,
    // which is more restrictive, therefore less appealing.
    // It's also not clear for me if this change would be
    // useful.
    // Waiting for the stabilization of specialization?
}

/// A wrapper-error that will be returned if the
/// [`Wrapper::mutate`](crate::Wrapper::mutate) method receives a closure that
/// mutates the inner value in such a way that it no longer passes
/// the [`Wrapper::PROCESS`](crate::Wrapper::PROCESS) function.
///
/// This wrapper contains the mutated value that caused the error, the
/// value before the mutation and the original error returned by
/// [`Wrapper::PROCESS`](crate::Wrapper::PROCESS) function.
#[derive(Debug)]
pub struct MutationError<W: Wrapper> {
    /// Value before the mutation.
    pub old_value: W::Inner,
    /// Value after mutation (the cause of the error).
    pub new_value: W::Inner,
    /// Original error.
    pub original: W::Error,
}

impl<W> fmt::Display for MutationError<W>
where
    W: Wrapper,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "failed to mutate type {} from value {:?} to value {:?}: {}",
            W::NAME,
            self.old_value,
            self.new_value,
            self.original,
        )
    }
}

impl<W> Error for MutationError<W>
where
    W: Wrapper + fmt::Debug,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display + fmt::Debug,
{
}

/// A wrapper-error that will be returned if the
/// [`Wrapper::verify`](crate::Wrapper::verify) method is called on a wrapper
/// whose inner value no longer passes the
/// [`Wrapper::PROCESS`](crate::Wrapper::PROCESS) function.
///
/// This wrapper contains both the value that caused the error and the original
/// error returned by the [`Wrapper::PROCESS`](crate::Wrapper::PROCESS)
/// function.
#[derive(Debug)]
pub struct VerificationError<W: Wrapper> {
    /// Value that caused the error.
    pub value: W::Inner,
    /// Original error.
    pub original: W::Error,
}

impl<W> fmt::Display for VerificationError<W>
where
    W: Wrapper,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "verification of type {} with value {:?} failed: {}",
            W::NAME,
            self.value,
            self.original,
        )
    }
}

impl<W> Error for VerificationError<W>
where
    W: Wrapper + fmt::Debug,
    W::Inner: fmt::Debug,
    W::Error: fmt::Display + fmt::Debug,
{
}

/// Convenience trait that allows mapping from `Result<_,
/// ConstructionError<Wrapper>>`, `Result<_, MutationError<Wrapper>` and
/// `Result<_, VerificationError<Wrapper>>` to `Result<_, Wrapper::Error>`.
pub trait MapOriginalError<O, E> {
    fn map_original(self) -> Result<O, E>;
}

impl<O, W: Wrapper> MapOriginalError<O, W::Error> for Result<O, ConstructionError<W>> {
    fn map_original(self) -> Result<O, W::Error> {
        self.map_err(|err| err.original)
    }
}

impl<O, W: Wrapper> MapOriginalError<O, W::Error> for Result<O, MutationError<W>> {
    fn map_original(self) -> Result<O, W::Error> {
        self.map_err(|err| err.original)
    }
}

impl<O, W: Wrapper> MapOriginalError<O, W::Error> for Result<O, VerificationError<W>> {
    fn map_original(self) -> Result<O, W::Error> {
        self.map_err(|err| err.original)
    }
}
