/// Helpers for serializing to/deserializing from toml
///
/// In order to use this module, set the "toml" feature in your Cargo.toml:
///
/// ```toml,ignore
/// [dependencies.potestas]
/// version = "0.1.0"
/// features = ["toml"]
/// ```
pub mod toml;

/// Helpers for serializing to/deserializing from json
///
/// In order to use this module, set the "json" feature in your Cargo.toml:
///
/// ```toml,ignore
/// [dependencies.elefen]
/// version = "0.1.0"
/// features = ["json"]
/// ```
pub mod json;

/// Helpers for working with the command line
pub mod cli;

/// # Low-level API for extending
// Convert the HTTP response body from JSON. Pass up deserialization errors
// transparently.
pub async fn deserialise_response<T: for<'de> serde::Deserialize<'de>>(
	response: reqwest::Response,
) -> crate::Result<T> {
	let bytes = response.bytes().await?;
	match serde_json::from_slice(&bytes) {
		Ok(t) => {
			trace!("Resp: {}", String::from_utf8_lossy(&bytes));
			Ok(t)
		}
		// If deserializing into the desired type fails try again to
		// see if this is an error response.
		Err(e) => {
			error!("Error resp: {}", String::from_utf8_lossy(&bytes));
			if let Ok(error) = serde_json::from_slice(&bytes) {
				return Err(crate::Error::Api(error));
			}
			Err(e.into())
		}
	}
}
