use std::{error, fmt, io::Error as IoError};

use ::toml::{de::Error as TomlDeError, ser::Error as TomlSerError};
use hyper_old_types::Error as HeaderParseError;
use reqwest::{header::ToStrError as HeaderStrError, Error as HttpError, StatusCode};
use serde::Deserialize;
use serde_json::Error as SerdeError;
use serde_qs::Error as SerdeQsError;
use serde_urlencoded::ser::Error as UrlEncodedError;
// use tungstenite::error::Error as WebSocketError;
use url::ParseError as UrlError;

/// Convience type over `std::result::Result` with `Error` as the error type.
pub type Result<T> = ::std::result::Result<T, Error>;

/// enum of possible errors encountered using the mastodon API.
#[derive(Debug, Error)]
pub enum Error {
	/// Error from the Mastodon API. This typically means something went
	/// wrong with your authentication or data.
	#[error(transparent)]
	Api(#[from] ApiError),
	/// Error deserialising to json. Typically represents a breaking change in
	/// the Mastodon API
	#[error(transparent)]
	Serde(#[from] SerdeError),
	#[error(transparent)]
	UrlEncoded(#[from] UrlEncodedError),
	#[error(transparent)]
	Http(#[from] HttpError),
	#[error(transparent)]
	Io(#[from] IoError),
	#[error(transparent)]
	Url(#[from] UrlError),
	#[error("Missing Client Id.")]
	ClientIdRequired,
	#[error("Missing Client Secret.")]
	ClientSecretRequired,
	#[error("Missing Access Token.")]
	AccessTokenRequired,
	#[error("Generic client error, code {0}")]
	Client(StatusCode),
	#[error("Generic server error, code {0}")]
	Server(StatusCode),
	#[error("Missing field {0}")]
	MissingField(&'static str),
	#[error(transparent)]
	TomlSer(#[from] TomlSerError),
	#[error(transparent)]
	TomlDe(#[from] TomlDeError),
	#[error(transparent)]
	HeaderStrError(#[from] HeaderStrError),
	#[error(transparent)]
	HeaderParseError(#[from] HeaderParseError),
	#[error(transparent)]
	SerdeQs(#[from] SerdeQsError),
	#[error(transparent)]
	WebSocket(#[from] tokio_tungstenite::tungstenite::Error),
	#[error("Error in streaming API data format")]
	StreamingFormat,
	#[error("Other error: {0}")]
	Other(String),
}

/// Error returned from the Mastodon API.
#[derive(Clone, Debug, Deserialize)]
pub struct ApiError {
	/// The type of error.
	pub error: Option<String>,
	/// The description of the error.
	pub error_description: Option<String>,
}

impl fmt::Display for ApiError {
	fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
		write!(f, "{:?}", self)
	}
}

impl error::Error for ApiError {}
