use std::fmt::{Display, Formatter};

use crate::entities::{
	event::Event,
	notification::{Notification, NotificationType},
};

pub struct NotificationDisplay<'a>(pub &'a Notification);

impl<'a> Display for NotificationDisplay<'a> {
	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		let n = self.0;
		match n.notification_type {
			NotificationType::Follow => {
				write!(f, "Follow {{ #{}, @{} }}", n.id, n.account.acct)
			}
			NotificationType::Favourite => {
				if let Some(ref s) = n.status {
					write!(
						f,
						"Favourite {{ #{}, acct: @{}, status: «{}» }}",
						n.id, n.account.acct, s.content
					)
				} else {
					write!(
						f,
						"Favourite {{ #{}, acct: @{}, status: -- }}",
						n.id, n.account.acct
					)
				}
			}
			NotificationType::Mention => {
				if let Some(ref s) = n.status {
					write!(
						f,
						"Mention {{ #{}, acct: @{}, status: «{}» }}",
						n.id, n.account.acct, s.content
					)
				} else {
					write!(
						f,
						"Mention {{ #{}, acct: @{}, status: -- }}",
						n.id, n.account.acct
					)
				}
			}
			NotificationType::Reblog => {
				if let Some(ref s) = n.status {
					write!(
						f,
						"Reblog {{ #{}, acct: @{}, status: «{}» }}",
						n.id, n.account.acct, s.content
					)
				} else {
					write!(
						f,
						"Reblog {{ #{}, acct: @{}, status: -- }}",
						n.id, n.account.acct
					)
				}
			}
		}
	}
}

pub struct EventDisplay<'a>(pub &'a Event);

impl<'a> Display for EventDisplay<'a> {
	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
		let n = self.0;
		match n {
			Event::Notification(n) => NotificationDisplay(n).fmt(f),
			Event::Delete(id) => {
				write!(f, "Delete {{ #{} }}", id)
			}
			Event::FiltersChanged => {
				write!(f, "FiltersChanged")
			}
			Event::Update(s) => {
				write!(
					f,
					"Status {{ #{}, acct: @{}, status: «{}», vis: {:?} }}",
					s.id, s.account.acct, s.content, s.visibility
				)
			}
		}
	}
}
