use etcd_client::Client;
use hyper::service::{make_service_fn, service_fn};
use hyper::{Body, Request, Response, Server, StatusCode};
use log::*;
use std::convert::Infallible;
use std::net::SocketAddr;
use std::sync::Arc;
use tokio::sync::watch::*;
use tokio::sync::Mutex;
use tokio_postgres::NoTls;

const LEASE_DURATION: i64 = 10;

#[tokio::main]
async fn main() {
    env_logger::init();
    let matches = clap::App::new("Postrep")
        .version("0.1")
        .author("Pierre-Étienne Meunier <pe@pijul.org>")
        .arg(
            clap::Arg::with_name("host")
                .long("host")
                .help("Name of this host")
                .required(true)
                .takes_value(true),
        )
        .arg(
            clap::Arg::with_name("signal")
                .long("signal")
                .help("Path to the postgres signal file")
                .required(true)
                .takes_value(true),
        )
        .arg(
            clap::Arg::with_name("port")
                .long("port")
                .required(true)
                .takes_value(true),
        )
        .arg(
            clap::Arg::with_name("password")
                .long("password")
                .required(true)
                .takes_value(true),
        )
        .get_matches();

    let (db, connection) = tokio_postgres::connect("host=localhost user=postgres", NoTls)
        .await
        .unwrap();
    let db = Arc::new(Mutex::new(db));
    let dbt = tokio::spawn(async move {
        if let Err(e) = connection.await {
            eprintln!("connection error: {}", e);
        }
    });
    tokio::pin!(dbt);
    let host = matches.value_of("host").unwrap().to_string();
    let signal = matches.value_of("signal").unwrap().to_string();
    let password = matches.value_of("password").unwrap().to_string();

    let (lead, is_leader) = channel(false);
    let lead = Arc::new(lead);
    let mut election_observer = tokio::spawn(observe_elections(
        host.clone(),
        signal.clone(),
        password.clone(),
        lead.clone(),
        db.clone(),
    ));
    let mut race = tokio::spawn(race(
        host.clone(),
        signal.clone(),
        lead.clone(),
        is_leader.clone(),
    ));
    let addr = SocketAddr::from((
        [0, 0, 0, 0],
        matches
            .value_of("port")
            .and_then(|x| x.parse().ok())
            .unwrap_or(8008),
    ));
    let make_service = make_service_fn(move |_conn| {
        let is_leader = is_leader.clone();
        async move { Ok::<_, Infallible>(service_fn(move |req| handle(req, is_leader.clone()))) }
    });
    loop {
        tokio::select!(
            _ = Server::bind(&addr).serve(make_service.clone()) => {
                debug!("http server stopping");
                break
            },
            _ = &mut dbt => {
                // If this database breaks, release the lock.
                tokio::time::sleep(std::time::Duration::from_secs(1)).await;
                info!("Restarting database connection");
                let (db_, connection) = if let Ok(x) = tokio_postgres::connect("host=localhost user=postgres", NoTls).await {
                    x
                } else {
                    tokio::time::sleep(std::time::Duration::from_millis(100)).await;
                    continue
                };
                lead.send(false).unwrap();
                dbt.set(tokio::spawn(async move {
                    if let Err(e) = connection.await {
                        eprintln!("connection error: {}", e);
                    }
                }));
                *db.lock().await = db_
            }
            _ = &mut race => {
                debug!("race stopped");
                break
            }
            _ = &mut election_observer => {
                debug!("election observer stopped");
                break
            }
        )
    }
}

async fn handle(
    _req: Request<Body>,
    is_leader: Receiver<bool>,
) -> Result<Response<Body>, Infallible> {
    if *is_leader.borrow() {
        Ok(Response::new(Body::from("OK")))
    } else {
        let mut resp = Response::new(Body::from("Not OK"));
        *resp.status_mut() = StatusCode::NOT_FOUND;
        Ok(resp)
    }
}

async fn race(
    host: String,
    signal: String,
    lead: Arc<Sender<bool>>,
    mut is_leader: Receiver<bool>,
) -> Result<(), anyhow::Error> {
    loop {
        let mut client = Client::connect(["localhost:2379"], None).await.unwrap();
        let lease = client.lease_grant(LEASE_DURATION, None).await?;
        let id = lease.id();
        info!("Taking lock {:x?}", id);
        let mut lock = client.campaign("postrep", host.as_str(), id).await?;

        debug!("lock = {:?}", lock);
        let (mut keeper, _stream) = client.lease_keep_alive(id).await?;
        let is_leader_ = is_leader.clone();
        let t = tokio::spawn(async move {
            tokio::time::sleep(std::time::Duration::from_secs(LEASE_DURATION as u64 / 2)).await;
            while *is_leader_.borrow() {
                debug!("sending keep alive");
                if let Err(e) = keeper.keep_alive().await {
                    error!("{:?}", e);
                    break;
                }
                tokio::time::sleep(std::time::Duration::from_secs(LEASE_DURATION as u64 / 2)).await;
            }
            debug!("done renewing");
        });

        let leader = lock.take_leader().unwrap();
        info!("Lock acquired: {:?}", leader);
        std::fs::remove_file(&signal).unwrap_or(());
        lead.send(true).unwrap();
        debug!("waiting for instance to become non-leader");
        loop {
            is_leader.changed().await.unwrap();
            debug!("changed: {:?}", is_leader.borrow());
            if !*is_leader.borrow() {
                info!("Lock released");
                // at.store(false, Ordering::SeqCst);
                t.await.unwrap();
                client
                    .resign(Some(etcd_client::ResignOptions::new().with_leader(leader)))
                    .await?;
                break;
            }
        }
    }
}

async fn observe_elections(
    host: String,
    signal: String,
    password: String,
    lead: Arc<Sender<bool>>,
    pool: Arc<Mutex<tokio_postgres::Client>>,
) -> Result<(), anyhow::Error> {
    let mut client = Client::connect(["localhost:2379"], None).await?;
    let mut obs = client.observe("postrep").await?;
    while let Some(m) = obs.message().await? {
        debug!("obs: {:?}", m);
        let leader = m.kv().unwrap().value_str()?;
        if leader != host {
            info!("The leader is {:?}", leader);
            std::fs::write(&signal, "").unwrap_or(());
            lead.send(false).unwrap();
        } else {
            info!("I'm the leader");
            std::fs::remove_file(&signal).unwrap_or(());
            lead.send(true).unwrap();
        }
        let pool = pool.lock().await;
        pool.execute(format!("ALTER SYSTEM SET primary_conninfo='host={} port=5432 user=replication password={}'", leader, password).as_str(), &[]).await?;
    }
    Ok(())
}
