use std::{
    fmt::Debug,
    marker::PhantomData,
};

use postgres::Client;
use serde::{
    de::DeserializeOwned,
    Serialize,
};

pub struct QueryContext<V>
where
    V: Debug + Default + Serialize + DeserializeOwned + Default, {
    query_name: String,
    query_instance_id: String,
    version: i64,
    _phantom: PhantomData<V>,
}

impl<V> QueryContext<V>
where
    V: Debug + Default + Serialize + DeserializeOwned + Default,
{
    pub fn new(
        query_name: String,
        query_instance_id: String,
        version: i64,
        _phantom: PhantomData<V>,
    ) -> QueryContext<V> {
        QueryContext {
            query_name,
            query_instance_id,
            version,
            _phantom: PhantomData,
        }
    }

    pub fn commit(
        &mut self,
        conn: &mut Client,
        view: V,
    ) {
        let sql = match self.version {
            0 => {
                format!(
                    "INSERT INTO {} (payload, version, \
                     query_instance_id) VALUES ( $1, $2, $3 )",
                    &self.query_name
                )
            },
            _ => {
                format!(
                    "UPDATE {} SET payload= $1 , version= $2 WHERE \
                     query_instance_id= $3",
                    &self.query_name
                )
            },
        };
        let version = self.version + 1;
        // let query_instance_id = &self.query_instance_id;
        let payload = match serde_json::to_value(&view) {
            Ok(payload) => payload,
            Err(err) => {
                panic!(
                    "unable to covert view '{}' with id: '{}', to \
                     value: {}\n  view: {:?}",
                    &self.query_instance_id,
                    &self.query_name,
                    err,
                    &view
                );
            },
        };
        match conn.execute(
            sql.as_str(),
            &[
                &payload,
                &version,
                &self.query_instance_id,
            ],
        ) {
            Ok(_) => {},
            Err(err) => {
                panic!(
                    "unable to update view '{}' with id: '{}', \
                     encountered: {}",
                    &self.query_instance_id, &self.query_name, err
                );
            },
        };
    }
}
