#![warn(clippy::pedantic)]

use std::collections::BTreeMap;
use std::fmt;

mod parser;
mod error;
pub use error::Error;

#[derive(Default, Debug)]
pub struct PoFile{
    /// Path to the file PoFile was created from.
    pub path: String,
    /// Size of the `langs` vector
    pub size: usize,
    /// Language entries
    pub langs: Vec<Po>,
}

impl PoFile{
    /// Creates a new `PoFile` struct and reads it from a file.
    /// # Errors
    /// If the path doesn't exist.
    pub fn new(path: &str) -> Result<Self, Box<dyn std::error::Error>>{
        let langs = parser::string_to_langs(path)?;
        Ok(PoFile{
            path: path.to_owned(),
            size: langs.len(),
            langs
        })
    }

    /// Writes the `PoFile` into a file specified in `path`
    /// # Errors
    /// If it's unable to write to the file
    pub fn write(&self, path: &str) -> Result<(), Box<dyn std::error::Error>>{
        let langs = self.to_string();
        std::fs::write(path, langs)?;
        Ok(())
    }

    /// Updates the file `PoFile` was created from.
    /// # Errors
    /// If it's unable to write to the file
    pub fn update(&self) -> Result<(), Box<dyn std::error::Error>>{
        let langs = self.to_string();
        std::fs::write(&self.path, langs)?;
        Ok(())
    }
}

impl fmt::Display for PoFile{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result{
        f.write_str(&parser::langs_to_string(&self.langs))
    }
}

#[derive(Default, Debug)]
pub struct Po{
    /// Comments maintained by a translator
    pub translator_comments: String,
    /// Automatic comments generated by a tool
    pub auto_comments: Vec<AutoComments>,
    /// References to the use in files
    pub reference: String,
    /// Message context
    pub msgctxt: String,
    /// Message id (untranslated text)
    pub msgid: String,
    /// Message string (translated text)
    pub msgstr: BTreeMap<usize, String>,
}

#[derive(Debug)]
pub enum AutoComments{
    /// Automatically extracted comment
    ExtractedComments(String),
    /// Flag
    Flag(String),
    /// Other comment like previous msgid or msgctxt
    Other(String),
}

#[test]
fn test_parse_file(){
    let path = "tests/en.po";
    let file = PoFile::new(path);
    assert!(file.is_ok());
    let file = dbg!(file.unwrap());
    assert!(file.size == 69);
}

#[test]
fn test_multi_number(){
    let path = "tests/multi_number.po";
    let file = PoFile::new(path);
    assert!(file.is_ok());
    let file = dbg!(file.unwrap());
    assert!(file.size == 1);
    let lang = file.langs.get(0);
    assert!(lang.is_some());
    let lang = lang.unwrap();
    assert!(lang.msgstr.len() == 2);
    assert!(!lang.msgstr.values().any(|a|{a.starts_with(" ")}));
    assert!(lang.msgid.contains("\n"));
}

#[test]
fn test_write(){
    let path = "tests/en.po";
    let file = PoFile::new(path);
    assert!(file.is_ok());
    let file = file.unwrap();
    let str = file.to_string();
    std::fs::write("tests/output_en.po", str).unwrap();
}
