use crate::agents::{DecrLimits, DeleteSession, SaveAuthCode};
use crate::config::LimitInput;
use crate::crypto::{create_jwt, session_id};
use crate::error::BrokerError;
use crate::web::{json_response, return_to_relier, Context, HandlerResult, ResponseType};
use serde::{Deserialize, Serialize};
use serde_json::json;

/// Session data stored by bridges.
#[derive(Clone, Serialize, Deserialize)]
#[serde(tag = "type")]
pub enum BridgeData {
    Email(email::EmailBridgeData),
    Oidc(oidc::OidcBridgeData),
}

/// Once a bridge has authenticated the user, this function can be used to finish up the redirect
/// to the relying party with an `id_token` or `code` generated by us.
pub async fn complete_auth(ctx: &mut Context) -> HandlerResult {
    let data = ctx
        .session_data
        .take()
        .expect("complete_auth called without a session");
    ctx.app
        .store
        .send(DeleteSession {
            session_id: ctx.session_id.clone(),
        })
        .await
        .map_err(|e| BrokerError::Internal(format!("could not remove a session: {}", e)))?;

    let state = data.return_params.state.clone();
    let origin = data
        .return_params
        .redirect_uri
        .origin()
        .ascii_serialization();

    ctx.app
        .store
        .send(DecrLimits {
            input: LimitInput {
                email_addr: data.email_addr.clone(),
                origin: origin.clone(),
                ip: data.original_ip,
            },
        })
        .await
        .map_err(|e| BrokerError::Internal(format!("could not decrement rate limits: {}", e)))?;

    let (auth_field, auth_value) = match data.response_type {
        ResponseType::IdToken => {
            let jwt = create_jwt(
                &ctx.app,
                &data.email,
                &data.email_addr,
                &origin,
                &data.nonce,
                data.signing_alg,
            )
            .await
            .map_err(|err| BrokerError::Internal(format!("Could not create a JWT: {:?}", err)))?;
            ("id_token", jwt)
        }
        ResponseType::Code => {
            let code = session_id(&data.email_addr, &origin, &ctx.app.rng).await;
            ctx.app
                .store
                .send(SaveAuthCode {
                    code: code.clone(),
                    data,
                })
                .await
                .map_err(|e| BrokerError::Internal(format!("could not save auth code: {}", e)))?;
            ("code", code)
        }
    };

    if ctx.want_json {
        Ok(json_response(&json!({
            auth_field: &auth_value,
            "state": &state,
        })))
    } else {
        Ok(return_to_relier(
            ctx,
            &[(auth_field, &auth_value), ("state", &state)],
        ))
    }
}

pub mod email;
pub mod oidc;
