use crate::agents::{DecrLimits, DeleteSession};
use crate::config::LimitInput;
use crate::crypto;
use crate::error::BrokerError;
use crate::web::{json_response, return_to_relier, Context, HandlerResult};
use serde::{Deserialize, Serialize};
use serde_json::json;

/// Session data stored by bridges.
#[derive(Clone, Serialize, Deserialize)]
#[serde(tag = "type")]
pub enum BridgeData {
    Email(email::EmailBridgeData),
    Oidc(oidc::OidcBridgeData),
}

/// Once a bridge has authenticated the user, this function can be used to finish up the redirect
/// to the relying party with a token generated by us.
pub async fn complete_auth(ctx: &mut Context) -> HandlerResult {
    let data = ctx
        .session_data
        .as_ref()
        .expect("complete_auth called without a session");
    ctx.app
        .store
        .send(DeleteSession {
            session_id: ctx.session_id.clone(),
        })
        .await
        .map_err(|e| BrokerError::Internal(format!("could not remove a session: {}", e)))?;

    let origin = data
        .return_params
        .redirect_uri
        .origin()
        .ascii_serialization();

    let jwt = crypto::create_jwt(
        &ctx.app,
        &data.email,
        &data.email_addr,
        &origin,
        &data.nonce,
        data.signing_alg,
    )
    .await
    .map_err(|err| {
        // Currently, the only possible failure here is that we accepted a signing algorithm
        // from the RP that suddenly disappeared from our config. Treat as an internal error.
        BrokerError::Internal(format!("Could not create a JWT: {:?}", err))
    })?;

    ctx.app
        .store
        .send(DecrLimits {
            input: LimitInput {
                email_addr: data.email_addr.clone(),
                origin,
                ip: data.original_ip,
            },
        })
        .await
        .map_err(|e| BrokerError::Internal(format!("could not decrement rate limits: {}", e)))?;

    if ctx.want_json() {
        Ok(json_response(
            &json!({
                "id_token": &jwt,
                "state": &data.return_params.state,
            }),
            None,
        ))
    } else {
        Ok(return_to_relier(
            ctx,
            &[("id_token", &jwt), ("state", &data.return_params.state)],
        ))
    }
}

pub mod email;
pub mod oidc;
