use std::{iter::Iterator, str::FromStr};
use serde_derive::{Deserialize, Serialize};
use anyhow::Result;
use surf::http::{Url, Method, Mime};
use lazy_static::lazy_static;
use std::sync::Arc;
use serde_json::json;
use tide::Request;
use http_types::headers::HeaderValue;
use tide::security::{CorsMiddleware, Origin};
use tide_acme::{AcmeConfig, TideRustlsExt};
use sonic_channel::*;
extern crate biscuit_auth as biscuit;
use biscuit::{crypto::PublicKey, token::Biscuit};

lazy_static! {
    static ref DB : Arc<rocksdb::DB> = {

        let prefix_extractor = rocksdb::SliceTransform::create_fixed_prefix(5);

        let mut opts = rocksdb::Options::default();
        opts.create_if_missing(true);
        opts.set_prefix_extractor(prefix_extractor);

        let configure = env_var_config();
        let db = rocksdb::DB::open(&opts, configure.db).unwrap();
        Arc::new(db)
    };
    
    static ref JWTS : Arc<lockfree::map::Map<String, (i64, String)>> = {
        Arc::new(lockfree::map::Map::new())
    };
}

#[derive(Debug, Serialize, Deserialize)]
pub struct Claims {
    pub exp: i64,          
    pub iat: i64,         
    pub iss: String,         
    pub sub: String,
}

#[derive(Deserialize, Debug, Clone)]
pub struct EnvVarConfig {
  pub port: u16,
  pub origin: String,
  pub db: String,
  pub secure: bool,
  pub certs: String,
  pub domain: String,
  pub sonic_server: String,
  pub sonic_password: String,
  pub broker: String,
  pub auto_cert: bool,
  pub key_path: String,
  pub cert_path: String,
}

#[derive(Debug, Serialize, Deserialize, Clone, PartialEq, Eq)]
pub struct Bucket {
    pub id: uuid::Uuid,
    pub name: String,
    pub username: String,
    pub items: Vec<uuid::Uuid>,
    pub collection: String,
}

#[derive(Debug, Serialize, Deserialize, Clone, PartialEq, Eq)]
pub struct BucketForm {
    pub collection: String,
    pub name: String,
}

#[derive(Debug, Serialize, Deserialize, Clone, PartialEq, Eq)]
pub struct Item {
    pub id: uuid::Uuid,
    pub collection: String,
    pub bucket: String,
    pub data: serde_json::Map<String, serde_json::Value>,
    pub indexes: Vec<String>,
    pub locale: Option<String>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct IndexForm {
    pub items: Vec<Item>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct DeindexForm {
    pub ids: Vec<uuid::Uuid>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct SearchForm {
    pub query: String,
    pub collection: String,
    pub bucket: String,
    pub limit: Option<usize>,
    pub offset: Option<usize>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct VerifyResponse {
    key: Vec<u8>,
    token: Vec<u8>,
    expiry: i64,
    username: String,
    scopes: Vec<String>,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct SuggestForm {
    pub query: String,
    pub collection: String,
    pub bucket: String,
    pub limit: Option<usize>,
}

fn replace(key: String, value: Vec<u8>) -> Result<()> {
    DB.put(key.clone(), value.clone())?;
    Ok(())
}

fn delete(key: String) -> Result<()> {
    DB.delete(key.clone())?;
    Ok(())
}

fn del_bucket(bucket: Bucket) -> Result<()> {
    let key = format!("buckets_{}_{}", bucket.collection, bucket.id);
    delete(key)?;
    Ok(())
}

fn get_items() -> Result<Vec<Item>> {
    let prefix = "items".as_bytes();
    let i = DB.prefix_iterator(prefix);
    let res : Vec<Item> = i.map(|(_, v)| {
        let data: Item = rmp_serde::from_read_ref(&v).unwrap();
        data
    }).collect();
    Ok(res)
}

fn get_buckets(collection_name: String) -> Result<Vec<Bucket>> {
    let prefix = format!("buckets_{}", collection_name);
    let prefix = prefix.as_bytes();
    let i = DB.prefix_iterator(prefix);
    let res : Vec<Bucket> = i.map(|(_, v)| {
        let data: Bucket = rmp_serde::from_read_ref(&v).unwrap();
        data
    }).collect();
    Ok(res)
}

fn is_bucket_unique(bucket_name: String, collection_name: String) -> Result<bool> {
    let buckets = get_buckets(collection_name.clone())?;
    for bucket in buckets {
        if bucket.name == bucket_name && bucket.collection == collection_name {
            return Ok(false);
        }
    }
    Ok(true)
}
   
fn puts_item(item: Item) -> Result<()> {
    let key = format!("items_{}", item.id);
    let value = rmp_serde::to_vec_named(&item)?;
    replace(key, value)?;
    Ok(())
}

fn index_items(items: Vec<Item>, username: String) -> Result<bool> {

    // check if all bucket calls are valid
    let mut check = true;
    for item in items.clone() {
        match get_bucket_by_name(item.bucket.clone(), username.clone(), item.collection.clone())? {
            Some(_) => {},
            None => { check = false; }
        }
    }

    if !check {
        return Ok(false);
    }

    for item in items {
        match get_bucket_by_name(item.bucket.clone(), username.clone(), item.collection.clone())? {
            Some(mut bucket) => {
                bucket.items.push(item.id.clone());
                puts_bucket(bucket)?;
                puts_item(item.clone())?;
                index_with_sonic(item.clone(), username.clone())?;
            },
            None => {}
        } 
    }
    Ok(true)
}

fn puts_bucket(bucket: Bucket) -> Result<()> {
    let key = format!("buckets_{}_{}", bucket.collection, bucket.id);
    let value = rmp_serde::to_vec_named(&bucket)?;
    replace(key, value)?;
    Ok(())
}

fn get_item_by_id(id: uuid::Uuid) -> Result<Option<Item>> {
    let items = get_items()?;
    Ok(items.into_iter().filter(|item| item.id == id).last())
}

fn get_bucket_by_name(name: String, username: String, collection_name: String) -> Result<Option<Bucket>> {
    let buckets = get_buckets(collection_name)?;
    Ok(buckets.into_iter().filter(|bucket| bucket.name == name && bucket.username == username).last())
}

fn del_items(ids: Vec<uuid::Uuid>) -> Result<()> {
    for id in ids {
        let key = format!("items_{}", id);
        delete(key)?;
    }
    Ok(())
}

fn env_var_config() -> EnvVarConfig {
 
    let mut port : u16 = 8888;
    let mut secure = false;
    let mut auto_cert = true;
    let mut origin = "*".to_string();
    let mut db: String = "db".to_string();
    let mut certs = "certs".to_string();
    let mut domain = "localhost".to_string();
    let mut sonic_server = "localhost:1491".to_string();
    let mut sonic_password = "SecretPassword".to_string();
    let mut broker = "http://localhost:8080".to_string();
    let mut key_path = "certs/private_key.pem".to_string();
    let mut cert_path = "certs/chain.pem".to_string();

    let _ : Vec<String> = go_flag::parse(|flags| {
        flags.add_flag("port", &mut port);
        flags.add_flag("origin", &mut origin);
        flags.add_flag("secure", &mut secure);
        flags.add_flag("db", &mut db);
        flags.add_flag("domain", &mut domain);
        flags.add_flag("certs", &mut certs);
        flags.add_flag("sonic_server", &mut sonic_server);
        flags.add_flag("sonic_password", &mut sonic_password);
        flags.add_flag("broker", &mut broker);
        flags.add_flag("auto_cert", &mut auto_cert);
        flags.add_flag("key_path", &mut key_path);
        flags.add_flag("cert_path", &mut cert_path);
    });

    EnvVarConfig{port, origin, secure, domain, certs, db, sonic_server, sonic_password, broker, auto_cert, cert_path, key_path}
}

async fn jwt_verify(token: String, operation: String) -> Result<(bool, String)> {

    let configure = env_var_config();

    let mut parts = token.split(" ");
    let auth_type = parts.next().unwrap();
    if auth_type == "Bearer" {
        let token = parts.next().unwrap();
        let time = nippy::get_unix_ntp_time().await?;

        match JWTS.get(&token.to_string()) {
            Some(entry) => {
                let (expiry, username) = entry.val();
                if expiry > &time {
                    return Ok((true, username.to_string()));
                } else {
                    return Ok((false, "".to_string()));
                }
            },
            None => {}
        }

        let broker_url = format!("{}/verify", configure.broker);
        let auth = format!("Bearer {}", token);
        let url = Url::parse(&broker_url)?;
        let mime = Mime::from_str("application/json").unwrap();
        let request = surf::Request::builder(Method::Get, url.clone())
        .header("authorization", &auth)
        .content_type(mime)
        .build();

        let mut res = surf::client().send(request).await.unwrap();
        if res.status() == 200 {
            let j: VerifyResponse = res.body_json().await.unwrap();
            let bis = Biscuit::from(&j.token)?;

            let public_key = PublicKey::from_bytes(&j.key).unwrap();
            let mut v1 = bis.verify(public_key)?;
            let f = format!("allow if right(#authority, \"files\", #{})", "full");
            let r = f.as_ref();
            v1.add_policy(r)?;
            let f = format!("allow if right(#authority, \"files\", #{})", operation);
            let r = f.as_ref();
            v1.add_policy(r)?;
            v1.deny()?;
            v1.verify()?;

            JWTS.insert(token.to_string(), (j.expiry, j.username.clone()));
            
            Ok((true, j.username.clone()))
        } else {
            Ok((false, "".to_string()))
        }
    } else {
        Ok((false, "".to_string()))
    }
}

fn index_with_sonic(item: Item, username: String) -> Result<()> {
    let configure = env_var_config();

    let channel = IngestChannel::start(configure.sonic_server, configure.sonic_password)?;
    match get_bucket_by_name(item.bucket.clone(), username.clone(), item.collection.clone())? {
        Some(_) => {
            match get_item_by_id(item.id)? {
                Some(item) => {
                    let mut ids = Vec::new();
                    ids.push(item.id);
                    deindex_items_with_sonic(ids)?;
                },
                None => {}
            }
            for (field, value) in item.clone().data {
                for index_field in item.indexes.clone() {
                    let oid = format!("{}_{}", item.id.to_string(), index_field);
                    if index_field == field && item.clone().locale == None {
                        channel.push(&item.collection, &item.bucket, &oid, &value.to_string())?;
                    }
                    else if index_field == field && item.clone().locale != None {
                        channel.push_with_locale(&item.collection, &item.bucket, &oid, &value.to_string(), &item.clone().locale.unwrap())?;
                    }
                }
            }
        },
        None => {} 
    }   
    Ok(())
}

fn deindex_items_with_sonic(ids: Vec<uuid::Uuid>) -> Result<()> {
    let configure = env_var_config();

    let channel = IngestChannel::start(configure.sonic_server, configure.sonic_password)?;
    for id in ids {
        match get_item_by_id(id)? {
            Some(item) => {
                for index in item.indexes {
                    let oid = format!("{}_{}", item.id.to_string(), index);
                    channel.flusho(&item.collection, &item.bucket, &oid)?;
                }
            },
            None => {}
        }
    }
    Ok(())
}

fn deindex_bucket_with_sonic(collection_name: String, bucket_name: String) -> Result<()> {
    let configure = env_var_config();

    let channel = IngestChannel::start(configure.sonic_server, configure.sonic_password)?;
    channel.flushb(&collection_name, &bucket_name)?;
    Ok(())
}

fn search_with_sonic(sf: SearchForm) -> Result<Vec<Item>> {
    let configure = env_var_config();

    let channel = SearchChannel::start(configure.sonic_server, configure.sonic_password)?;

    let mut items = Vec::new();

    if sf.offset != None && sf.limit != None {
        let ids: Vec<String> = channel.query_with_limit_and_offset(&sf.collection, &sf.bucket, &sf.query, sf.limit.unwrap(), sf.offset.unwrap())?;
        for id_str in ids {
            let uid: Vec<&str> = id_str.split("_").collect();
            let id = uuid::Uuid::parse_str(&uid[0])?;
            let item = get_item_by_id(id)?;
            items.push(item.unwrap());
        }
    }
    else if sf.offset == None && sf.limit != None {
        let ids: Vec<String> = channel.query_with_limit(&sf.collection, &sf.bucket, &sf.query, sf.limit.unwrap())?;
        for id_str in ids {
            let uid: Vec<&str> = id_str.split("_").collect();
            let id = uuid::Uuid::parse_str(&uid[0])?;
            match get_item_by_id(id)? {
                Some(item) => {
                    items.push(item);
                },
                None => {}
            }
        }
    }
    else {
        let ids: Vec<String> = channel.query(&sf.collection, &sf.bucket, &sf.query)?;
        for id_str in ids {
            let uid: Vec<&str> = id_str.split("_").collect();
            let id = uuid::Uuid::parse_str(&uid[0])?;
            match get_item_by_id(id)? {
                Some(item) => {
                    items.push(item);
                },
                None => {}
            }
        }
    }

    Ok(items)
}

fn suggest_with_sonic(sf: SuggestForm) -> Result<Vec<String>> {
    let configure = env_var_config();

    let channel = SearchChannel::start(configure.sonic_server, configure.sonic_password)?;

    if sf.limit != None {
        return Ok(channel.suggest_with_limit(&sf.collection, &sf.bucket, &sf.query, sf.limit.unwrap())?);
    }
    else {
        return Ok(channel.suggest(&sf.collection, &sf.bucket, &sf.query)?);
    }
}

fn bucket_create(bucket_form: BucketForm, username: String) -> Result<bool> {
    let check = is_bucket_unique(bucket_form.name.clone(), bucket_form.collection.clone())?;
    if check {
        let id = uuid::Uuid::new_v4();
        let bucket = Bucket{
            id,
            name: bucket_form.name,
            username,
            items: Vec::new(), 
            collection: bucket_form.collection,
        };
        puts_bucket(bucket)?;
        Ok(true)
    } else {
        Ok(false)
    }
}

fn bucket_delete(bucket_form: BucketForm, username: String) -> Result<()> {
    match get_bucket_by_name(bucket_form.name, username, bucket_form.collection.clone())? {
        Some(bucket) => {
            del_items(bucket.items.clone())?;
            deindex_bucket_with_sonic(bucket_form.collection, bucket.name.clone())?;
            del_bucket(bucket.clone())?;
        },
        None => {}
    }
    Ok(())
}

async fn index(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "index".to_string()).await?;
            if check {
                    let r =  req.body_string().await?;
                    let index_form : IndexForm = serde_json::from_str(&r)?;
                    let items = index_form.items;
                    let checker = index_items(items.clone(), username.clone())?;
                    if checker {
                        Ok(tide::Response::builder(200).header("content-type", "application/json").build())
                    } else {
                        let j = json!({"error": "a non existent bucket was specified in the items"}).to_string();
                        Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build())
                    }

            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

async fn deindex(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, _) = jwt_verify(token, "deindex".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let deindex_form : DeindexForm = serde_json::from_str(&r)?;
                let ids = deindex_form.ids;
                del_items(ids.clone())?;
                deindex_items_with_sonic(ids.clone())?;
                Ok(tide::Response::builder(200).header("content-type", "application/json").build())
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

async fn search(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, _) = jwt_verify(token, "search".to_string()).await?;
            if check {
                    let r =  req.body_string().await?;
                    let search_form : SearchForm = serde_json::from_str(&r)?;
                    let result = search_with_sonic(search_form)?;
                    Ok(tide::Response::builder(200).body(json!(result)).header("content-type", "application/json").build())
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

async fn suggest(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, _) = jwt_verify(token, "suggest".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let suggest_form : SuggestForm = serde_json::from_str(&r)?;
                let result = suggest_with_sonic(suggest_form)?;
                let j = json!({ "suggestions": result});
                Ok(tide::Response::builder(200).body(j).header("content-type", "application/json").build())
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

async fn health(_: Request<()>) -> tide::Result {
    Ok(tide::Response::builder(200).header("content-type", "application/json").build())
}

async fn create_bucket(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "create_bucket".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let create_bucket_form : BucketForm = serde_json::from_str(&r)?;
                let checker = bucket_create(create_bucket_form, username)?;
                if checker {
                    Ok(tide::Response::builder(200).header("content-type", "application/json").build())
                } else {
                    let j = json!({"error": "bucket with this name already exists"});
                    Ok(tide::Response::builder(400).body(j).header("content-type", "application/json").build())
                }
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

async fn delete_bucket(mut req: Request<()>) -> tide::Result {
    let token_value = req.header("authorization");
    match token_value {
        Some(token_header) => {
            let token = token_header.last().to_string();
            let (check, username) = jwt_verify(token, "delete_bucket".to_string()).await?;
            if check {
                let r =  req.body_string().await?;
                let delete_bucket_form : BucketForm = serde_json::from_str(&r)?;
                bucket_delete(delete_bucket_form, username)?;
                Ok(tide::Response::builder(200).header("content-type", "application/json").build())
            } else {
                Ok(tide::Response::builder(401).header("content-type", "application/json").build())
            }
        },
        None => { Ok(tide::Response::builder(401).header("content-type", "application/json").build()) }
    }
}

#[async_std::main]
async fn main() -> tide::Result<()> {

    let configure = env_var_config();

    let cors = CorsMiddleware::new()
    .allow_methods("GET, POST, OPTIONS".parse::<HeaderValue>().unwrap())
    .allow_headers("authorization".parse::<HeaderValue>().unwrap())
    .allow_origin(Origin::from(configure.origin))
    .allow_credentials(false);
    
    let mut app = tide::new();
    app.with(driftwood::DevLogger);
    app.with(cors);
    app.at("/").get(health);
    app.at("/").head(health);
    app.at("/index").post(index);
    app.at("/deindex").post(deindex);
    app.at("/search").post(search);
    app.at("/suggest").post(suggest);
    app.at("/create_bucket").post(create_bucket);
    app.at("/delete_bucket").post(delete_bucket);

    let ip = format!("0.0.0.0:{}", configure.port);

    if configure.secure && configure.auto_cert {
        app.listen(
            tide_rustls::TlsListener::build().addrs("0.0.0.0:443").acme(
                AcmeConfig::new()
                    .domains(vec![configure.domain])
                    .cache_dir(configure.certs)
                    .production(),
            ),
        )
        .await?;
    } else if configure.secure && !configure.auto_cert {
        app.listen(
            tide_rustls::TlsListener::build()
            .addrs("0.0.0.0:443")
            .cert(configure.cert_path)
            .key(configure.key_path)
        )
        .await?;
    } else {
        app.listen(ip).await?;
    }

    Ok(())
}