//! Instruction types

use solana_program::{
  instruction::{AccountMeta, Instruction},
  pubkey::{Pubkey, PUBKEY_BYTES},
  sysvar,
};
use std::{mem::size_of};

/// Instructions supported by the lending program.
#[derive(Clone, Debug, PartialEq)]
pub enum LendingInstruction {

  // 3
  /// Accrue interest and update market price of liquidity on a reserve.
  ///
  /// Accounts expected by this instruction:
  ///
  ///   0. `[writable]` Reserve account.
  ///   1. `[]` Clock sysvar.
  ///   2. `[]` Reserve liquidity oracle account.
  ///             Must be the Pyth price account specified at InitReserve.
  RefreshReserve,

  // 4
  /// Deposit liquidity into a reserve in exchange for collateral. Collateral represents a share
  /// of the reserve liquidity pool.
  ///
  /// Accounts expected by this instruction:
  ///
  ///   0. `[writable]` Source liquidity token account.
  ///                     $authority can transfer $liquidity_amount.
  ///   1. `[writable]` Destination collateral token account.
  ///   2. `[writable]` Reserve account.
  ///   3. `[writable]` Reserve liquidity supply SPL Token account.
  ///   4. `[writable]` Reserve collateral SPL Token mint.
  ///   5. `[]` Lending market account.
  ///   6. `[]` Derived lending market authority.
  ///   7. `[signer]` User transfer authority ($authority).
  ///   8. `[]` Clock sysvar.
  ///   9. `[]` Token program id.
  DepositReserveLiquidity {
      /// Amount of liquidity to deposit in exchange for collateral tokens
      liquidity_amount: u64,
  },

  // 5
  /// Redeem collateral from a reserve in exchange for liquidity.
  ///
  /// Accounts expected by this instruction:
  ///
  ///   0. `[writable]` Source collateral token account.
  ///                     $authority can transfer $collateral_amount.
  ///   1. `[writable]` Destination liquidity token account.
  ///   2. `[writable]` Reserve account.
  ///   3. `[writable]` Reserve collateral SPL Token mint.
  ///   4. `[writable]` Reserve liquidity supply SPL Token account.
  ///   5. `[]` Lending market account.
  ///   6. `[]` Derived lending market authority.
  ///   7. `[signer]` User transfer authority ($authority).
  ///   8. `[]` Clock sysvar.
  ///   9. `[]` Token program id.
  RedeemReserveCollateral {
      /// Amount of collateral tokens to redeem in exchange for liquidity
      collateral_amount: u64,
  },

}

impl LendingInstruction {
  /// Packs a [LendingInstruction](enum.LendingInstruction.html) into a byte buffer.
  pub fn pack(&self) -> Vec<u8> {
      let mut buf = Vec::with_capacity(size_of::<Self>());
      match *self {
          Self::RefreshReserve => {
              buf.push(3);
          }
          Self::DepositReserveLiquidity { liquidity_amount } => {
              buf.push(4);
              buf.extend_from_slice(&liquidity_amount.to_le_bytes());
          }
          Self::RedeemReserveCollateral { collateral_amount } => {
              buf.push(5);
              buf.extend_from_slice(&collateral_amount.to_le_bytes());
          }
      }
      buf
  }
}

/// Creates a `RefreshReserve` instruction
pub fn refresh_reserve(
  program_id: Pubkey,
  reserve_pubkey: Pubkey,
  reserve_liquidity_oracle_pubkey: Pubkey,
) -> Instruction {
  let accounts = vec![
      AccountMeta::new(reserve_pubkey, false),
      AccountMeta::new_readonly(reserve_liquidity_oracle_pubkey, false),
      AccountMeta::new_readonly(sysvar::clock::id(), false),
  ];
  Instruction {
      program_id,
      accounts,
      data: LendingInstruction::RefreshReserve.pack(),
  }
}

/// Creates a 'DepositReserveLiquidity' instruction.
#[allow(clippy::too_many_arguments)]
pub fn deposit_reserve_liquidity(
  program_id: Pubkey,
  liquidity_amount: u64,
  source_liquidity_pubkey: Pubkey,
  destination_collateral_pubkey: Pubkey,
  reserve_pubkey: Pubkey,
  reserve_liquidity_supply_pubkey: Pubkey,
  reserve_collateral_mint_pubkey: Pubkey,
  lending_market_pubkey: Pubkey,
  user_transfer_authority_pubkey: Pubkey,
) -> Instruction {
  let (lending_market_authority_pubkey, _bump_seed) = Pubkey::find_program_address(
      &[&lending_market_pubkey.to_bytes()[..PUBKEY_BYTES]],
      &program_id,
  );
  Instruction {
      program_id,
      accounts: vec![
          AccountMeta::new(source_liquidity_pubkey, false),
          AccountMeta::new(destination_collateral_pubkey, false),
          AccountMeta::new(reserve_pubkey, false),
          AccountMeta::new(reserve_liquidity_supply_pubkey, false),
          AccountMeta::new(reserve_collateral_mint_pubkey, false),
          AccountMeta::new_readonly(lending_market_pubkey, false),
          AccountMeta::new_readonly(lending_market_authority_pubkey, false),
          AccountMeta::new_readonly(user_transfer_authority_pubkey, true),
          AccountMeta::new_readonly(sysvar::clock::id(), false),
          AccountMeta::new_readonly(spl_token::id(), false),
      ],
      data: LendingInstruction::DepositReserveLiquidity { liquidity_amount }.pack(),
  }
}

/// Creates a 'RedeemReserveCollateral' instruction.
#[allow(clippy::too_many_arguments)]
pub fn redeem_reserve_collateral(
  program_id: Pubkey,
  collateral_amount: u64,
  source_collateral_pubkey: Pubkey,
  destination_liquidity_pubkey: Pubkey,
  reserve_pubkey: Pubkey,
  reserve_collateral_mint_pubkey: Pubkey,
  reserve_liquidity_supply_pubkey: Pubkey,
  lending_market_pubkey: Pubkey,
  user_transfer_authority_pubkey: Pubkey,
) -> Instruction {
  let (lending_market_authority_pubkey, _bump_seed) = Pubkey::find_program_address(
      &[&lending_market_pubkey.to_bytes()[..PUBKEY_BYTES]],
      &program_id,
  );
  Instruction {
      program_id,
      accounts: vec![
          AccountMeta::new(source_collateral_pubkey, false),
          AccountMeta::new(destination_liquidity_pubkey, false),
          AccountMeta::new(reserve_pubkey, false),
          AccountMeta::new(reserve_collateral_mint_pubkey, false),
          AccountMeta::new(reserve_liquidity_supply_pubkey, false),
          AccountMeta::new_readonly(lending_market_pubkey, false),
          AccountMeta::new_readonly(lending_market_authority_pubkey, false),
          AccountMeta::new_readonly(user_transfer_authority_pubkey, true),
          AccountMeta::new_readonly(sysvar::clock::id(), false),
          AccountMeta::new_readonly(spl_token::id(), false),
      ],
      data: LendingInstruction::RedeemReserveCollateral { collateral_amount }.pack(),
  }
}
