use std::mem::size_of;

use solana_program::instruction::{AccountMeta, Instruction};

use crate::error::StakingError;
use crate::instruction::StakingInstruction::*;
use crate::solana_program::{msg, program_error::ProgramError, pubkey::Pubkey, sysvar};

/// Instructions supported by the lending program.
#[derive(Clone, Debug, PartialEq)]
pub enum StakingInstruction {
    /// Accounts expected by this instruction:
    ///
    ///   0. `[writable]` Stake account - uninitialized.
    ///   1. `[]` Staking Pool.
    ///   2. `[]` Stake account owner.
    ///   3. `[]` Rent sysvar.
    CreateStakeAccount,
}

impl StakingInstruction {
    pub fn unpack(input: &[u8]) -> Result<Self, ProgramError> {
        input
            .split_first()
            .ok_or_else(|| StakingError::InstructionUnpackError.into())
            .and_then(|(&tag, rest)| match tag {
                1 => Ok((CreateStakeAccount, rest)),
                _ => {
                    msg!("Instruction cannot be unpacked");
                    Err(StakingError::InstructionUnpackError.into())
                }
            })
            .and_then(|(ins, rest)| {
                if rest.is_empty() {
                    Ok(ins)
                } else {
                    Err(StakingError::InstructionUnpackError.into())
                }
            })
    }

    pub fn pack(&self) -> Vec<u8> {
        let mut buf = Vec::with_capacity(size_of::<Self>());
        match *self {
            Self::CreateStakeAccount => {
                buf.push(1);
            }
        };
        buf
    }
}

//helpers

fn create_read_accounts(accounts: Vec<Pubkey>) -> impl Iterator<Item = AccountMeta> {
    accounts
        .into_iter()
        .map(|acc| AccountMeta::new_readonly(acc, false))
}

pub fn create_stake_account(
    program_id: Pubkey,
    stake_account: Pubkey,
    staking_pool: Pubkey,
    stake_account_owner: Pubkey,
) -> Instruction {
    let read_accounts =
        create_read_accounts(vec![staking_pool, stake_account_owner, sysvar::rent::id()]);

    let accounts = vec![AccountMeta::new(stake_account, false)]
        .into_iter()
        .chain(read_accounts)
        .collect();

    Instruction {
        program_id,
        accounts,
        data: StakingInstruction::CreateStakeAccount.pack(),
    }
}
