use num::complex::Complex;
use crate::cln::CLn;

/// Provides the 2nd order polylogarithm (dilogarithm) function
/// `li2()` of a number of type `T`.
pub trait Li2<T> {
    fn li2(&self) -> T;
}

impl Li2<f64> for f64 {
    /// Returns the real dilogarithm of a real number of type `f64`.
    ///
    /// Implemented as rational function approximation with a maximum
    /// error of 5e-17 [[arXiv:2201.01678]].
    ///
    /// [arXiv:2201.01678]: https://arxiv.org/abs/2201.01678
    ///
    /// # Example:
    /// ```
    /// use polylog::Li2;
    ///
    /// let z = 1.0;
    /// println!("Li2({}) = {}", z, z.li2());
    /// ```
    fn li2(&self) -> f64 {
        let pi = std::f64::consts::PI;
        let cp = [
            0.9999999999999999502e+0,
           -2.6883926818565423430e+0,
            2.6477222699473109692e+0,
           -1.1538559607887416355e+0,
            2.0886077795020607837e-1,
           -1.0859777134152463084e-2
        ];
        let cq = [
            1.0000000000000000000e+0,
           -2.9383926818565635485e+0,
            3.2712093293018635389e+0,
           -1.7076702173954289421e+0,
            4.1596017228400603836e-1,
           -3.9801343754084482956e-2,
            8.2743668974466659035e-4
        ];

        let x = *self;

        // transform to [0, 1/2]
        let (y, rest, sgn) = if x < -1. {
            let l = (1. - x).ln();
            (1./(1. - x), -pi*pi/6. + l*(0.5*l - (-x).ln()), 1.)
        } else if x == -1. {
            return -pi*pi/12.;
        } else if x < 0. {
            let l = (-x).ln_1p();
            (x/(x - 1.), -0.5*l*l, -1.)
        } else if x == 0. {
            return 0.;
        } else if x < 0.5 {
            (x, 0., 1.)
        } else if x < 1. {
            (1. - x, pi*pi/6. - x.ln()*(1. - x).ln(), -1.)
        } else if x == 1. {
            return pi*pi/6.;
        } else if x < 2. {
            let l = x.ln();
            (1. - 1./x, pi*pi/6. - l*((1. - 1./x).ln() + 0.5*l), 1.)
        } else {
            let l = x.ln();
            (1./x, pi*pi/3. - 0.5*l*l, -1.)
        };

        let y2 = y*y;
        let y4 = y2*y2;
        let p = cp[0] + y * cp[1] + y2 * (cp[2] + y * cp[3]) +
                y4 * (cp[4] + y * cp[5]);
        let q = cq[0] + y * cq[1] + y2 * (cq[2] + y * cq[3]) +
                y4 * (cq[4] + y * cq[5] + y2 * cq[6]);

        rest + sgn*y*p/q
    }
}

impl Li2<Complex<f64>> for Complex<f64> {
    /// Returns the dilogarithm of a complex number of type
    /// `Complex<f64>`.
    ///
    /// This function has been translated from the
    /// [SPheno](https://spheno.hepforge.org/) package.
    ///
    /// # Example:
    /// ```
    /// extern crate num;
    /// use num::complex::Complex;
    /// use polylog::Li2;
    ///
    /// fn main() {
    ///     let z = Complex::new(1.0, 1.0);
    ///     println!("Li2({}) = {}", z, z.li2());
    /// }
    /// ```
    fn li2(&self) -> Complex<f64> {
        let pi = std::f64::consts::PI;

        // bf[1..N-1] are the even Bernoulli numbers / (2 n + 1)!
        // generated by: Table[BernoulliB[2 n]/(2 n + 1)!, {n, 1, 19}]
        let bf = [
            - 1./4.,
              1./36.,
            - 1./3600.,
              1./211680.,
            - 1./10886400.,
              1./526901760.,
            - 4.0647616451442255e-11,
              8.9216910204564526e-13,
            - 1.9939295860721076e-14,
              4.5189800296199182e-16,
        ];

        let rz = self.re;
        let iz = self.im;

        // special cases
        if iz == 0. {
            if rz <= 1. {
                return Complex::new(rz.li2(), 0.0)
            } else { // rz > 1.
                return Complex::new(rz.li2(), -pi*rz.ln())
            }
        }

        let nz = self.norm_sqr();

        if nz < std::f64::EPSILON {
            return self*(1.0 + 0.25*self);
        }

        let (u, rest, sgn) = if rz <= 0.5 {
            if nz > 1. {
                let l = (-self).cln();
                (-(1. - 1. / self).cln(), -0.5 * l * l - pi * pi / 6., -1.)
            } else { // nz <= 1.
                (-(1. - self).cln(), Complex::new(0.,0.), 1.)
            }
        } else { // rz > 0.5
            if nz <= 2.0*rz {
                let l = -(self).cln();
                (l, l * (1. - self).cln() + pi * pi / 6., -1.)
            } else { // nz > 2.0*rz
                let l = (-self).cln();
                (-(1. - 1. / self).cln(), -0.5 * l * l - pi * pi / 6., -1.)
            }
        };

        let u2 = u*u;
        let u4 = u2*u2;
        let sum =
            u +
            u2 * (bf[0] +
            u  * (bf[1] +
            u2 * (
                bf[2] +
                u2*bf[3] +
                u4*(bf[4] + u2*bf[5]) +
                u4*u4*(bf[6] + u2*bf[7] + u4*(bf[8] + u2*bf[9]))
            )));

        sgn * sum + rest
    }
}
