use regex::Regex;
use serde::{Deserialize, Serialize};
use std::{collections::BTreeMap, fs, io};

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct PollapoYml {
    pub deps: Vec<String>,
    pub root: PollapoRoot,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct PollapoRoot {
    pub lock: BTreeMap<String, String>,
}

pub fn load_pollapo_yml(yml_path: &String) -> PollapoYml {
    let content_text = fs::read_to_string(yml_path).expect("failed to read pollapo.yml");
    serde_yaml::from_str(&content_text).unwrap()
}

pub fn write_pollao_yml(yml_path: &String, yml: PollapoYml) -> io::Result<()> {
    fs::write(yml_path, serde_yaml::to_string(&yml).unwrap())
}

#[derive(Debug)]
pub struct PollapoDepCore {
    pub owner: String,
    pub repo: String,
}

#[derive(Debug)]
pub struct PollapoDepFrag {
    pub core: PollapoDepCore,
    pub rev: Option<String>,
}

#[derive(Debug)]
pub struct PollapoDep {
    pub core: PollapoDepCore,
    pub rev: String,
}

impl ToString for PollapoDep {
    fn to_string(&self) -> String {
        format!("{}/{}@{}", self.core.owner, self.core.repo, self.rev).to_string()
    }
}

pub fn parse_dep_frag(dep: &str) -> PollapoDepFrag {
    let re = Regex::new(r"(?P<owner>.+?)/(?P<repo>.+?)(@(?P<rev>.+))?$").unwrap();
    let caps = re.captures(dep).unwrap();

    PollapoDepFrag {
        core: PollapoDepCore {
            owner: caps
                .name("owner")
                .expect("user not specified")
                .as_str()
                .to_string(),
            repo: caps
                .name("repo")
                .expect("repo not specified")
                .as_str()
                .to_string(),
        },
        rev: match caps.name("rev") {
            Some(match_res) => Some(match_res.as_str().to_string()),
            None => None,
        },
    }
}

pub fn push_dep(yml: &PollapoYml, dep: PollapoDep) -> PollapoYml {
    let mut new_deps = yml.deps.clone();
    new_deps.push(dep.to_string());
    PollapoYml {
        deps: new_deps,
        root: yml.root.clone(),
    }
}
