pub mod auth;
pub mod client;

use std::fs;
use std::io;
use github_rs::client::Executor;
use serde_json::Value;
use yaml_rust::YamlLoader;
use github_rs::client::Github;

use crate::pollapo_yml::PollapoDepCore;

pub fn get_host_location() -> String {
    format!(
        "{}/.config/gh/hosts.yml",
        std::env::var("HOME").expect("failed to get $HOME")
    )
}

pub fn get_token() -> Option<String> {
    match fs::read_to_string(get_host_location()) {
        Ok(content) => match &YamlLoader::load_from_str(&content) {
            Ok(parsed_content) => match parsed_content[0]["github.com"]["oauth_token"].as_str() {
                Some(str) => Some(str.to_string()),
                None => None,
            },
            Err(_) => None,
        },
        Err(_) => None,
    }
}

pub fn write_token(token: String) -> io::Result<()> {
    let host_data = format!(
        "github.com:\n   oauth_token: {}\n   git_protocol: ssh",
        token
    );
    fs::write(get_host_location(), host_data)
}

pub fn get_refs(client: &Github, pollapo_dep_core: &PollapoDepCore) -> Vec<String> {
    let branches = client.get().repos().owner(&pollapo_dep_core.owner).repo(&pollapo_dep_core.repo).branches().execute::<Value>();
    let tags = client.get().repos().owner(&pollapo_dep_core.owner).repo(&pollapo_dep_core.repo).tags().execute::<Value>();

    let mut result = vec![];

    match branches {
        Ok((_, _, json)) => {
            if let Some(json) = json {
                let vec = json.as_array().unwrap();
                for item in vec {
                    let name = item["name"].as_str().unwrap();
                    result.push(name.to_string());
                }
            }
        }
        Err(e) => panic!("{}", e),
    }

    match tags {
        Ok((_, _, json)) => {
            if let Some(json) = json {
                let vec = json.as_array().unwrap();
                for item in vec {
                    let name = item["name"].as_str().unwrap();
                    result.push(name.to_string());
                }
            }
        }
        Err(e) => panic!("{}", e),
    }

    result
}
