use crate::util::request;
use form_urlencoded;
use rand::prelude::*;
use reqwest::{self, StatusCode};
use spinners;
use std::borrow::Cow;
use std::thread::sleep;
use std::time::Duration as StdDuration;

static OAUTH_CLIENT_ID: &str = "178c6fc778ccc68e1d6a";
static SCOPES: [&str; 3] = ["repo", "read:org", "gist"];
static GRANT_TYPE: &str = "urn:ietf:params:oauth:grant-type:device_code";

#[derive(Debug)]
pub struct RequestCodeResp {
    pub device_code: String,
    pub expires_in: i64,
    pub interval: u64,
    pub user_code: String,
    pub verification_uri: String,
}

impl RequestCodeResp {
    pub fn new() -> Self {
        RequestCodeResp {
            device_code: String::new(),
            expires_in: 0,
            interval: 0,
            user_code: String::new(),
            verification_uri: String::new(),
        }
    }
}

pub async fn request_code() -> Result<RequestCodeResp, reqwest::Error> {
    let req_body = form_urlencoded::Serializer::new(String::new())
        .append_pair("client_id", OAUTH_CLIENT_ID)
        .append_pair("scope", &SCOPES.join(" "))
        .finish();

    let resp = request::client()
        .post("https://github.com/login/device/code")
        .header("Content-Type", "application/x-www-form-urlencoded")
        .body(req_body)
        .send()
        .await?
        .bytes()
        .await?;

    let parsed_resp = form_urlencoded::parse(&resp);
    Ok(
        parsed_resp.fold(RequestCodeResp::new(), |acc, kv| match kv {
            (Cow::Borrowed("device_code"), device_code) => RequestCodeResp {
                device_code: device_code.to_string(),
                ..acc
            },
            (Cow::Borrowed("expires_in"), expires_in) => RequestCodeResp {
                expires_in: expires_in.to_string().parse().unwrap(),
                ..acc
            },
            (Cow::Borrowed("interval"), interval) => RequestCodeResp {
                interval: interval.to_string().parse().unwrap(),
                ..acc
            },
            (Cow::Borrowed("user_code"), user_code) => RequestCodeResp {
                user_code: user_code.to_string(),
                ..acc
            },
            (Cow::Borrowed("verification_uri"), verification_uri) => RequestCodeResp {
                verification_uri: verification_uri.to_string(),
                ..acc
            },
            _ => unreachable!(),
        }),
    )
}

#[derive(Debug)]
pub struct AccessToken {
    pub access_token: String,
    pub token_type: String,
    pub scope: String,
}

impl AccessToken {
    pub fn new() -> Self {
        Self {
            access_token: String::new(),
            token_type: String::new(),
            scope: String::new(),
        }
    }
}

pub async fn poll_token(code: &RequestCodeResp) -> Result<AccessToken, reqwest::Error> {
    let spinner_list = [
        spinners::Spinners::Dots,
        spinners::Spinners::Line,
        spinners::Spinners::Pipe,
        spinners::Spinners::BoxBounce,
    ];
    let mut rng = rand::thread_rng();
    let rnd_idx = rng.gen_range(0..spinner_list.len());
    let spinner = &spinner_list[rnd_idx];

    let spinning = spinners::Spinner::new(&spinner, "Wait for authorization complete...".into());

    loop {
        sleep(StdDuration::from_secs(code.interval));
        let resp = request_token(&code.device_code).await?;
        match resp {
            RequestTokenResp::Pending => {}
            RequestTokenResp::Created { access_token } => {
                spinning.stop();
                return Ok(access_token);
            }
        }
    }
}

#[derive(Debug)]
enum RequestTokenResp {
    Pending,
    Created { access_token: AccessToken },
}

impl RequestTokenResp {
    pub fn append_access_token(self, kv: (Cow<str>, Cow<str>)) -> Self {
        match self {
            Self::Pending => Self::Pending,
            Self::Created { access_token } => match kv {
                (Cow::Borrowed("error"), message) => {
                    if message == "authorization_pending" {
                        RequestTokenResp::Pending
                    } else {
                        panic!("failed to request access token")
                    }
                }
                (Cow::Borrowed("access_token"), token) => Self::Created {
                    access_token: AccessToken {
                        access_token: token.to_string(),
                        ..access_token
                    },
                },
                (Cow::Borrowed("token_type"), token_type) => Self::Created {
                    access_token: AccessToken {
                        token_type: token_type.to_string(),
                        ..access_token
                    },
                },
                (Cow::Borrowed("scope"), scope) => Self::Created {
                    access_token: AccessToken {
                        scope: scope.to_string(),
                        ..access_token
                    },
                },
                _ => Self::Created { access_token },
            },
        }
    }
}

async fn request_token(device_code: &String) -> Result<RequestTokenResp, reqwest::Error> {
    let req_body = form_urlencoded::Serializer::new(String::new())
        .append_pair("client_id", OAUTH_CLIENT_ID)
        .append_pair("device_code", &device_code)
        .append_pair("grant_type", GRANT_TYPE)
        .finish();

    let resp = request::client()
        .post("https://github.com/login/oauth/access_token")
        .header("Content-Type", "application/x-www-form-urlencoded")
        .body(req_body)
        .send()
        .await?
        .bytes()
        .await?;

    let parsed_resp = form_urlencoded::parse(&resp);
    Ok(parsed_resp.fold(
        RequestTokenResp::Created {
            access_token: AccessToken::new(),
        },
        |acc, kv| acc.append_access_token(kv),
    ))
}

pub async fn validate_token(token: &str) -> Option<&str> {
    let resp_result = request::client()
        .get("https://api.github.com/")
        .header("Authorization", token)
        .header("Accept", "*/*")
        .header("User-Agent", "curl/7.68.1")
        .send()
        .await;
    match resp_result {
        Ok(resp) => match resp.status() {
            StatusCode::OK => Some(token),
            _ => None,
        },
        Err(_) => None,
    }
}
