use crate::{
    misc::github::{get_token, client::get_client, get_refs},
    pollapo_yml::{load_pollapo_yml, parse_dep_frag, PollapoDep, push_dep, write_pollao_yml},
};
use clap::{arg, Arg, ArgMatches, Command};
use dialoguer::{Select, theme::ColorfulTheme, console::Term};

pub fn command() -> Command<'static> {
    Command::new("add")
        .about("Add dependencies")
        .arg_required_else_help(true)
        .args(&[
            arg!(<targets> ... "Target github repository to add"),
            Arg::new("config")
                .short('c')
                .long("config")
                .default_value("pollapo.yml")
                .help("Pollapo config"),
            Arg::new("token").short('t').long("token"),
        ])
}

pub async fn run(arg_matches: &ArgMatches) {
    if arg_matches.is_present("targets") {
        let targets: Vec<_> = arg_matches.values_of("targets").unwrap().collect();

        let token: String = if arg_matches.is_present("token") {
            arg_matches.value_of("token").unwrap().to_string()
        } else {
            get_token().unwrap().to_string()
        };

        let client = get_client(&token).await.unwrap();

        let pollapo_yml_location = arg_matches.value_of("config").unwrap().to_string();
        let pollapo_yml = load_pollapo_yml(&pollapo_yml_location);

        let new_yml = targets.iter().fold(pollapo_yml, |acc, target| {
            let dep_frag = parse_dep_frag(target);
            let refs = get_refs(&client, &dep_frag.core);

            match dep_frag.rev {
                Some(rev) => {
                    let pollapo_dep = PollapoDep {
                        core: dep_frag.core,
                        rev,
                    };
                    if refs.contains(&pollapo_dep.rev) {
                        push_dep(&acc, pollapo_dep)
                    } else {
                        panic!("Revision {} not found", pollapo_dep.rev);
                    }
                }
                None => {
                    let selection = Select::with_theme(&ColorfulTheme::default()).items(&refs).default(0).interact_on_opt(&Term::stderr()).unwrap().unwrap();
                    push_dep(&acc, PollapoDep {
                        core: dep_frag.core,
                        rev: refs[selection].clone(),
                    })
                }
            }
        });
        write_pollao_yml(&pollapo_yml_location, new_yml).unwrap();
    }
}
