# Handling errors

In `Poem`, we handle errors based on the response status code. When the status code is in `400-599`, we can think that 
an error occurred while processing this request.

We can use `EndpointExt::after` to create a new endpoint type to customize the error response.

In the following example, the `after` function is used to convert the output of `index`, and a custom response is output
if there is an error.

**Note that the endpoint type generated by a `handler` macro is always `Endpoint<Output=Response>`, even if it returns 
a `Result<T>`.**

```rust
use poem::{handler, Result, Error};
use poem::http::StatusCode;

#[handler]
async fn index() -> Result<()> {
    Err(Error::new(StatusCode::BAD_REQUEST))
}

let ep = index.after(|resp| {
    if resp.status().is_server_error() {
        Response::builder()
            .status(resp.status())
            .body("custom error")
    } else {
        resp
    }
});
```

The `EndpointExt::map_to_result` function can help us convert any type of endpoint to `Endpoint<Output = Response>`, so 
that we only need to check the status code to know whether an error has occurred.

```rust
use poem::endpoint::make;
use poem::{Error, EndpointExt};
use poem::http::StatusCode;

let ep = make(|_| Ok::<(), Error>(Error::new(StatusCode::new(Status::BAD_REQUEST))))
    .map_to_response();
    
let ep = ep.after(|resp| {
    if resp.status().is_server_error() {
        Response::builder()
            .status(resp.status())
            .body("custom error")
    } else {
        resp
    }
});
```