use chrono::DateTime;
use chrono::Utc;
use serde::Deserialize;

/// A sample of transaction data for a currency.
#[derive(Clone, Debug, Deserialize)]
pub struct TransactionSample {
	pub id: u64,
	pub league_id: u16,
	pub pay_currency_id: u16,
	pub get_currency_id: u16,
	pub sample_time_utc: DateTime<Utc>,
	pub count: u16,
	pub value: f64,
	pub data_point_count: u16,
	pub includes_secondary: bool,
	pub listing_count: u16
}

/// A week's worth of price changes for a currency or item.
#[derive(Clone, Debug, Deserialize)]
pub struct WeekOfDailyPriceChanges {
	/// Represented in percentages, comparing the end of a day with the beginning.
	pub data: Vec<Option<f32>>,
	#[serde(rename = "totalChange")]
	/// Percentage comparing the end of the final day with the beginning of the first.
	pub total_change: f32,
}

/// A Path of Exile currency.
#[derive(Clone, Debug, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Currency {
	#[serde(rename = "currencyTypeName")]
	pub name: String,
	pub details_id: String,
	pub pay: Option<TransactionSample>,
	pub receive: Option<TransactionSample>,
	#[serde(rename = "paySparkLine")]
	pub pay_history: WeekOfDailyPriceChanges,
	#[serde(rename = "receiveSparkLine")]
	pub receive_history: WeekOfDailyPriceChanges,
	pub chaos_equivalent: f32,
	#[serde(rename = "lowConfidencePaySparkLine")]
	pub low_confidence_pay_history: WeekOfDailyPriceChanges,
	#[serde(rename = "lowConfidenceReceiveSparkLine")]
	pub low_confidence_receive_history: WeekOfDailyPriceChanges
}

#[derive(Clone, Debug, Deserialize)]
#[serde(rename_all = "camelCase")]
#[allow(dead_code)]
pub(crate) struct CurrencyDetail {
	pub id: u64,
	pub icon: String,
	pub name: String,
	pub trade_id: Option<String>
}

#[derive(Clone, Debug, Deserialize)]
#[allow(dead_code)]
pub(crate) struct Language {
	pub name: String,
	// TODO:  translations
}

#[derive(Clone, Debug, Deserialize)]
#[serde(rename_all = "camelCase")]
#[allow(dead_code)]
pub(crate) struct CurrencyResponse {
	pub lines: Vec<Currency>,
	pub currency_details: Vec<CurrencyDetail>,
	pub language: Language
}

#[derive(Clone, Debug, Deserialize)]
pub struct ItemModifier {
	pub text: String,
	pub optional: bool
}

#[derive(Clone, Debug, Deserialize)]
#[serde(rename_all = "camelCase")]
/// A Path of Exile item.
pub struct Item {
	pub id: u64,
	pub details_id: String,
	pub name: String,
	pub icon: Option<String>,
	#[serde(default)]
	pub stack_size: u16,
	pub art_filename: Option<String>,
	pub item_class: u8,
	#[serde(rename = "sparkline")]
	pub value_history: WeekOfDailyPriceChanges,
	#[serde(rename = "lowConfidenceSparkline")]
	pub low_confidence_value_history: WeekOfDailyPriceChanges,
	pub implicit_modifiers: Vec<ItemModifier>,
	pub explicit_modifiers: Vec<ItemModifier>,
	#[serde(rename = "flavourText")] // trollface.jpeg
	pub flavor_text: Option<String>,
	pub chaos_value: f32,
	pub exalted_value: f32,
	pub count: u16,
	#[serde(default)]
	pub listing_count: u16
}

#[derive(Clone, Debug, Deserialize)]
pub(crate) struct ItemResponse {
	pub lines: Vec<Item>,
	#[allow(dead_code)]
	pub language: Language
}

