#![deny(missing_docs)]
/*!
See the [description](https://crates.io/crates/pns)
**/

mod sys;

pub use self::sys::types::{Net, Node, State};

use self::sys::functions::*;

use std::{
    fs::{self, File},
    io::{Read, Write},
    path::Path,
};

/// A wrapper type for net to support managing multiple states for simulations.
pub struct SimulatedNet {
    net: Net,
    /// The states of the net. Cannot be changed directly.
    pub states: Vec<State>,
}

impl Clone for SimulatedNet {
    fn clone(&self) -> Self {
        let net = self.net.clone();
        let mut states = Vec::new();
        for old_state in &self.states {
            states.push(unsafe {
                let mut state = std::mem::MaybeUninit::uninit();
                pnsCloneState(state.as_mut_ptr(), old_state, &net);
                state.assume_init()
            })
        }
        Self { net, states }
    }
}

impl std::ops::Deref for SimulatedNet {
    type Target = Net;
    fn deref(&self) -> &Net {
        &self.net
    }
}

/// An iterator to iterate over the states of a simulated net.
pub struct Iter<'net> {
    net: &'net Net,
    states: std::slice::Iter<'net, State>,
}

/// A mutable iterator to iterate over the states of a simulated net.
pub struct IterMut<'net> {
    net: &'net Net,
    states: std::slice::IterMut<'net, State>,
}

/// A simulation state combined with the net it belongs to.
pub struct SimulationState<'net> {
    net: &'net Net,
    /// The real state value.
    pub state: &'net State,
}

/// A simulation state combined with the net it belongs to, ready for simualtion.
pub struct SimulationStateMut<'net> {
    net: &'net Net,
    /// The real state value.
    pub state: &'net mut State,
}

impl<'net> IntoIterator for &'net SimulatedNet {
    type Item = SimulationState<'net>;
    type IntoIter = Iter<'net>;

    fn into_iter(self) -> Self::IntoIter {
        Iter {
            net: &self.net,
            states: (&self.states).iter(),
        }
    }
}

impl<'net> IntoIterator for &'net mut SimulatedNet {
    type Item = SimulationStateMut<'net>;
    type IntoIter = IterMut<'net>;

    fn into_iter(self) -> Self::IntoIter {
        IterMut {
            net: &self.net,
            states: (&mut self.states).iter_mut(),
        }
    }
}

impl<'net> Iterator for Iter<'net> {
    type Item = SimulationState<'net>;

    fn next(&mut self) -> Option<Self::Item> {
        if let Some(state) = self.states.next() {
            Some(SimulationState {
                net: &self.net,
                state,
            })
        } else {
            None
        }
    }
}

impl<'net> Iterator for IterMut<'net> {
    type Item = SimulationStateMut<'net>;

    fn next(&mut self) -> Option<Self::Item> {
        if let Some(state) = self.states.next() {
            Some(SimulationStateMut {
                net: &self.net,
                state,
            })
        } else {
            None
        }
    }
}

/// A dynamic type, able to store both net kinds.
pub enum DynamicNet {
    /// Variant for the default `Net` type.
    Default(Net),
    /// Variant for the `SimulatedNet` type.
    Simulated(SimulatedNet),
}

/// A trait to share safe functionality for the default `Net` type and the `SimulatedNet`
pub trait SafeNet {
    /// Save the petri net to a file. Result represents success.
    fn save(&self, filename: &Path) -> bool;
    /// A slice of nodes representing the transitions of the petri net.
    fn transitions(&self) -> &[Node];
    /// Returns the index of the next reusable transition.
    fn reusable_transition(&self) -> Option<u32>;
    /// A slice of nodes representing the places of the petri net.
    fn places(&self) -> &[Node];
    /// Returns the index of the next reusable place.
    fn reusable_place(&self) -> Option<u32>;
    /// A slice of initial token counts for the places of the petri net.
    fn initial_token_counts(&self) -> &[u32];

    /// Add a new place to the petri net and get the index.
    fn add_place(&mut self) -> u32;

    /// Add a new transition to the petri net and get the index.
    fn add_transition(&mut self) -> u32;

    /// Add a new transition to the petri net, connct it to the specified places and get the index.
    fn add_connected_transition(&mut self, pids: &[u32]) -> u32;

    /// Remove a place at index `pid` from petri net.
    fn remove_place(&mut self, pid: u32);

    /// Make a connection out from the transitoin with index `tid` to place with index `pid`.
    /// Result represents success.
    fn connect_out(&mut self, tid: u32, pid: u32) -> bool;

    /// Make a connection into transition with index `tid` from place with index `pid`.
    fn disconnect_in(&mut self, tid: u32, pid: u32);

    /// Duplicate the transition and get the index of the clone.
    fn duplicate_transition(&mut self, tid: u32) -> u32;

    /// Duplicate the place and get the index of the clone.
    fn duplicate_place(&mut self, pid: u32) -> u32;

    /// Increase the initial token count in place indexed by `pid`.
    fn start(&mut self, pid: u32, count: u32) -> u32;

    /// Creates a dynamic net type from a net.
    fn dynamic(self) -> DynamicNet;
}

impl DynamicNet {
    /// Convert the dynamic net into a `Net`
    pub fn default(&self) -> Option<&Net> {
        if let DynamicNet::Default(net) = self {
            Some(net)
        } else {
            None
        }
    }
    /// Convert the dynamic net into a mutable `Net`
    pub fn default_mut(&mut self) -> Option<&mut Net> {
        if let DynamicNet::Default(net) = self {
            Some(net)
        } else {
            None
        }
    }
    /// Convert the dynamic net into a `SimulatedNet`
    pub fn simulated(&self) -> Option<&SimulatedNet> {
        if let DynamicNet::Simulated(net) = self {
            Some(net)
        } else {
            None
        }
    }
    /// Convert the dynamic net into a mutable `SimulatedNet`
    pub fn simulated_mut(&mut self) -> Option<&mut SimulatedNet> {
        if let DynamicNet::Simulated(net) = self {
            Some(net)
        } else {
            None
        }
    }
    /// Convert the dynamic net into a `SafeNet`
    pub fn safe(&self) -> &dyn SafeNet {
        match self {
            DynamicNet::Default(net) => net,
            DynamicNet::Simulated(net) => net,
        }
    }
    /// Convert the dynamic net into a mutable `SafeNet`
    pub fn safe_mut(&mut self) -> &mut dyn SafeNet {
        match self {
            DynamicNet::Default(net) => net,
            DynamicNet::Simulated(net) => net,
        }
    }
}

impl SimulatedNet {
    /// Create a new, empty petri net in simulated state.
    pub fn new() -> SimulatedNet {
        SimulatedNet {
            net: Net::new(),
            states: Vec::new(),
        }
    }
    /// Load a petri net from a file in simulated state.
    pub fn load(filename: &Path) -> Option<SimulatedNet> {
        Some(SimulatedNet {
            net: Net::load(filename)?,
            states: Vec::new(),
        })
    }
    /// Convert a `Net` to a `SimulatedNet` for advanced simulation.
    pub fn from_net(net: Net) -> SimulatedNet {
        let mut net = net;
        unsafe { pnsNet_clearEdits(&mut net) }
        SimulatedNet {
            net,
            states: Vec::new(),
        }
    }

    /// Add a new, empty state for simulation of a petri net.
    /// The state is bound to the lifetime of the simulated net.
    pub fn add_state(&mut self) {
        let state = State::new(&self.net);
        self.states.push(state);
    }

    /// Load a simulation state from a file.
    /// Using the wrong file layout will fail or create confusing state.
    #[must_use]
    pub fn load_state(&mut self, filename: &Path) -> Option<()> {
        if let Some(state) = State::load(&self.net, filename) {
            self.states.push(state);
            Some(())
        } else {
            None
        }
    }

    /// Get the simulation state of a net at the specified index.
    pub fn state<'net>(&'net self, index: usize) -> SimulationState<'net> {
        SimulationState {
            net: &self.net,
            state: &self.states[index],
        }
    }

    /// Get the simulation state of a net at the specified index.
    pub fn state_mut<'net>(&'net mut self, index: usize) -> SimulationStateMut<'net> {
        SimulationStateMut {
            net: &self.net,
            state: &mut self.states[index],
        }
    }

    /// Calculate the number of states.
    pub fn len(&self) -> usize {
        self.states.len()
    }

    /// Release the `Net` from the `SimulatedNet` again.
    /// Destroys the simulation state.
    pub fn release(self) -> Net {
        self.net
    }

    // ensure all states are updated.
    fn update_states(&mut self) {
        for state in &mut self.states {
            unsafe {
                pnsState_updateEdits(state, &self.net);
            }
        }
        unsafe { pnsNet_clearEdits(&mut self.net) }
    }
}

impl SafeNet for SimulatedNet {
    #[inline]
    fn save(&self, filename: &Path) -> bool {
        self.net.save(filename)
    }

    #[inline]
    fn transitions(&self) -> &[Node] {
        self.net.transitions()
    }
    #[inline]
    fn reusable_transition(&self) -> Option<u32> {
        self.net.reusable_transition()
    }
    #[inline]
    fn places(&self) -> &[Node] {
        self.net.places()
    }
    #[inline]
    fn reusable_place(&self) -> Option<u32> {
        self.net.reusable_place()
    }
    #[inline]
    fn initial_token_counts(&self) -> &[u32] {
        self.net.initial_token_counts()
    }

    #[inline]
    fn add_place(&mut self) -> u32 {
        let result = self.net.add_place();
        self.update_states();
        result
    }
    #[inline]
    fn add_transition(&mut self) -> u32 {
        let result = self.net.add_transition();
        self.update_states();
        result
    }
    #[inline]
    fn add_connected_transition(&mut self, pids: &[u32]) -> u32 {
        let result = self.net.add_connected_transition(pids);
        self.update_states();
        result
    }

    #[inline]
    fn remove_place(&mut self, pid: u32) {
        let result = self.net.remove_place(pid);
        self.update_states();
        result
    }

    #[inline]
    fn connect_out(&mut self, tid: u32, pid: u32) -> bool {
        let result = self.net.connect_out(tid, pid);
        self.update_states();
        result
    }
    #[inline]
    fn disconnect_in(&mut self, tid: u32, pid: u32) {
        let result = self.net.disconnect_in(tid, pid);
        self.update_states();
        result
    }

    #[inline]
    fn duplicate_transition(&mut self, tid: u32) -> u32 {
        let result = self.net.duplicate_transition(tid);
        self.update_states();
        result
    }
    #[inline]
    fn duplicate_place(&mut self, pid: u32) -> u32 {
        let result = self.net.duplicate_place(pid);
        self.update_states();
        result
    }
    #[inline]
    fn start(&mut self, pid: u32, count: u32) -> u32 {
        let result = self.net.start(pid, count);
        self.update_states();
        result
    }

    fn dynamic(self) -> DynamicNet {
        DynamicNet::Simulated(self)
    }
}

impl Node {
    /// Get a slice of indices to the next nodes.
    pub fn next(&self) -> &[u32] {
        unsafe { std::slice::from_raw_parts(self.next, self.next_count as usize) }
    }

    /// Get a slice of indices to the previous nodes.
    pub fn prev(&self) -> &[u32] {
        unsafe { std::slice::from_raw_parts(self.prev, self.prev_count as usize) }
    }
}

impl Net {
    /// Create a new, empty petri net.
    pub fn new() -> Net {
        unsafe {
            let mut net = std::mem::MaybeUninit::uninit();
            pnsCreateNet(net.as_mut_ptr());
            net.assume_init()
        }
    }
    /// Load a petri net from a file.
    pub fn load(filename: &Path) -> Option<Net> {
        let size = (fs::metadata(filename).ok()?.len() + 3) / 4;

        let mut values;
        {
            let mut file = File::open(filename).ok()?;

            values = vec![0; size as usize * 4];
            file.read_exact(&mut values).ok()?;
        }

        unsafe {
            let mut net = std::mem::MaybeUninit::uninit();

            if pnsLoadNet(
                net.as_mut_ptr(),
                size as u32,
                std::mem::transmute::<_, *mut u32>(&values[0]),
            ) {
                Some(net.assume_init())
            } else {
                None
            }
        }
    }
    /// Remove a transition at index `tid` from petri net.
    pub fn remove_transition(&mut self, tid: u32) {
        assert!(tid < self.transition_count, "Transition id out of range");
        unsafe { pnsNet_removeTransition_unsafe(self, tid) }
    }
    /// Make a connection into transition with index `tid` from place with index `pid`.
    /// Result represents success.
    pub fn connect_in(&mut self, tid: u32, pid: u32) -> bool {
        assert!(tid < self.transition_count, "Transition id out of range");
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_connectIn_unsafe(self, tid, pid) }
    }
    /// Remove the connection out from transition with index `tid` to place with index `pid`.
    pub fn disconnect_out(&mut self, tid: u32, pid: u32) {
        assert!(tid < self.transition_count, "Transition id out of range");
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_disconnectOut_unsafe(self, tid, pid) }
    }
}

impl SafeNet for Net {
    fn save(&self, filename: &Path) -> bool {
        let mut file = File::create(filename).unwrap();

        let data = unsafe {
            let count = pnsNet_serializeSize(self);
            let mut data = vec![0u32; count as usize];
            pnsNet_serialize(self, &mut data[0]);
            data
        };

        for value in data {
            if file.write_all(&value.to_ne_bytes()).is_err() {
                return false;
            }
        }

        true
    }

    fn transitions(&self) -> &[Node] {
        unsafe { std::slice::from_raw_parts(self.transitions, self.transition_count as usize) }
    }
    fn reusable_transition(&self) -> Option<u32> {
        if let Some(list) = unsafe { self.reusable_transitions.as_ref() } {
            Some(list.index)
        } else {
            None
        }
    }
    fn places(&self) -> &[Node] {
        unsafe { std::slice::from_raw_parts(self.places, self.place_count as usize) }
    }
    fn reusable_place(&self) -> Option<u32> {
        if let Some(list) = unsafe { self.reusable_places.as_ref() } {
            Some(list.index)
        } else {
            None
        }
    }
    fn initial_token_counts(&self) -> &[u32] {
        unsafe { std::slice::from_raw_parts(self.initial_token_counts, self.place_count as usize) }
    }

    fn add_place(&mut self) -> u32 {
        unsafe { pnsNet_addPlace(self) }
    }
    fn add_transition(&mut self) -> u32 {
        unsafe { pnsNet_addTransition(self) }
    }
    fn add_connected_transition(&mut self, pids: &[u32]) -> u32 {
        for pid in pids {
            assert!(*pid < self.place_count, "Place id out of range");
        }
        unsafe { pnsNet_addConnectedTransition(self, pids.len() as u32, pids.as_ptr()) }
    }

    fn remove_place(&mut self, pid: u32) {
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_removePlace(self, pid) }
    }

    fn connect_out(&mut self, tid: u32, pid: u32) -> bool {
        assert!(tid < self.transition_count, "Transition id out of range");
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_connectOut(self, tid, pid) }
    }
    fn disconnect_in(&mut self, tid: u32, pid: u32) {
        assert!(tid < self.transition_count, "Transition id out of range");
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_disconnectIn(self, tid, pid) }
    }

    fn duplicate_transition(&mut self, tid: u32) -> u32 {
        assert!(tid < self.transition_count, "Transition id out of range");
        unsafe { pnsNet_duplicateTransition(self, tid) }
    }
    fn duplicate_place(&mut self, pid: u32) -> u32 {
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_duplicatePlace(self, pid) }
    }
    fn start(&mut self, pid: u32, count: u32) -> u32 {
        assert!(pid < self.place_count, "Place id out of range");
        unsafe { pnsNet_start(self, pid, count) }
    }

    fn dynamic(self) -> DynamicNet {
        DynamicNet::Default(self)
    }
}

impl Clone for Net {
    fn clone(&self) -> Self {
        unsafe {
            let mut net = std::mem::MaybeUninit::uninit();
            pnsCloneNet(net.as_mut_ptr(), self);
            net.assume_init()
        }
    }
}

impl Drop for Net {
    fn drop(&mut self) {
        unsafe {
            pnsDestroyNet(self);
        }
    }
}

impl State {
    fn new(net: &Net) -> State {
        unsafe {
            let mut state = std::mem::MaybeUninit::uninit();
            pnsCreateState(state.as_mut_ptr(), net);
            state.assume_init()
        }
    }

    fn load(net: &Net, filename: &Path) -> Option<State> {
        let size = (fs::metadata(filename).ok()?.len() + 3) / 4;

        let mut values;
        {
            let mut file = File::open(filename).ok()?;

            values = vec![0; size as usize * 4];
            file.read_exact(&mut values).ok()?;
        }

        unsafe {
            let mut state = std::mem::MaybeUninit::uninit();
            if pnsLoadState(
                state.as_mut_ptr(),
                net,
                size as u32,
                std::mem::transmute::<_, *mut u32>(&mut values[0]),
            ) {
                Some(state.assume_init())
            } else {
                None
            }
        }
    }

    fn changed_transitions(
        &mut self,
        call: unsafe extern "C" fn(state: *mut State, *mut u32, *mut u32),
    ) -> Box<[u32]> {
        unsafe {
            let mut count = std::mem::MaybeUninit::uninit();
            call(self, count.as_mut_ptr(), std::ptr::null_mut());
            let mut count = count.assume_init();
            let mut transitions = Vec::with_capacity(count as usize);
            transitions.set_len(count as usize);
            call(self, &mut count, transitions.as_mut_ptr());
            transitions.into_boxed_slice()
        }
    }

    /// Get a list of newly fireable transitions since the last call of this method.
    /// Automatically clears that list.
    pub fn added_transitions(&mut self) -> Box<[u32]> {
        self.changed_transitions(pnsState_addedTransitions)
    }

    /// Get a list of newly fireable transitions when playing backwards since the last call of this method.
    /// Automatically clears that list.
    pub fn added_transitions_backwards(&mut self) -> Box<[u32]> {
        self.changed_transitions(pnsState_addedTransitions_backwards)
    }

    /// Get a list of nomore fireable transitions since the last call of this method.
    /// Automatically clears that list.
    pub fn removed_transitions(&mut self) -> Box<[u32]> {
        self.changed_transitions(pnsState_removedTransitions)
    }

    /// Get a list of nomore fireable transitions when playing backwards since the last call of this method.
    /// Automatically clears that list.
    pub fn removed_transitions_backwards(&mut self) -> Box<[u32]> {
        self.changed_transitions(pnsState_removedTransitions_backwards)
    }
}

impl Drop for State {
    fn drop(&mut self) {
        unsafe {
            pnsDestroyState(self);
        }
    }
}

impl<'net> SimulationState<'net> {
    /// Save the current state of the simulation.
    pub fn save(&self, filename: &Path) -> bool {
        let mut file = File::create(filename).unwrap();
        for count in self.call_counts() {
            if file.write_all(&count.to_ne_bytes()).is_err() {
                return false;
            }
        }

        true
    }

    /// List the current call counts of all transitions.
    /// Counts are counted down again when playing backwards.
    pub fn call_counts(&self) -> &[u32] {
        unsafe {
            std::slice::from_raw_parts(self.state.call_counts, self.net.transition_count as usize)
        }
    }

    /// List the current token counts of all transitions.
    pub fn token_counts(&self) -> &[u32] {
        unsafe {
            std::slice::from_raw_parts(self.state.token_counts, self.net.place_count as usize)
        }
    }
}

impl<'net> SimulationStateMut<'net> {
    /// Save the current state of the simulation.
    pub fn save(&self, filename: &Path) -> bool {
        let mut file = File::create(filename).unwrap();
        for count in self.call_counts() {
            if file.write_all(&count.to_ne_bytes()).is_err() {
                return false;
            }
        }

        true
    }

    /// List the current call counts of all transitions.
    /// Counts are counted down again when playing backwards.
    pub fn call_counts(&self) -> &[u32] {
        unsafe {
            std::slice::from_raw_parts(self.state.call_counts, self.net.transition_count as usize)
        }
    }

    /// List the current token counts of all transitions.
    pub fn token_counts(&self) -> &[u32] {
        unsafe {
            std::slice::from_raw_parts(self.state.token_counts, self.net.place_count as usize)
        }
    }

    /// Lists fireable transitions without being able to fire them.
    pub fn fireable(&mut self) -> Box<[u32]> {
        unsafe {
            let mut count = std::mem::MaybeUninit::uninit();
            pnsState_transitions(self.state, count.as_mut_ptr(), std::ptr::null_mut());
            let mut count = count.assume_init();
            let mut transitions = Vec::with_capacity(count as usize);
            transitions.set_len(count as usize);
            let mut transitions = transitions.into_boxed_slice();
            pnsState_transitions(self.state, &mut count, transitions.as_mut_ptr());
            transitions
        }
    }

    /// Lists unfireable transitions without being able to unfire them.
    pub fn unfireable(&mut self) -> Box<[u32]> {
        unsafe {
            let mut count = std::mem::MaybeUninit::uninit();
            pnsState_transitions_backwards(self.state, count.as_mut_ptr(), std::ptr::null_mut());
            let mut count = count.assume_init();
            let mut transitions = Vec::with_capacity(count as usize);
            transitions.set_len(count as usize);
            let mut transitions = transitions.into_boxed_slice();
            pnsState_transitions_backwards(self.state, &mut count, transitions.as_mut_ptr());
            transitions
        }
    }

    /// Generate a `FireState` to fire a transition later.
    pub fn fire<'state>(&'state mut self) -> FireState<'net, 'state> {
        FireState::new(self)
    }

    /// Generate a `FireState` to unfire a transition later.
    pub fn unfire<'state>(&'state mut self) -> FireState<'net, 'state> {
        FireState::new_backwards(self)
    }

    /// Just fire the transition `tid`.
    /// Caution: Be sure to fire only, when you know a transition to be fireable.
    pub unsafe fn fire_unchecked(&mut self, tid: u32) {
        pnsState_fire(self.state, self.net, tid);
    }

    /// Just unfire the transition `tid`.
    /// Caution: Be sure to unfire only, when you know a transition to be unfireable.
    pub unsafe fn unfire_unchecked(&mut self, tid: u32) {
        pnsState_fire_backwards(self.state, self.net, tid);
    }
}

/// A temporary object for selecting a transition to fire.
pub struct FireState<'net, 'state> {
    state: &'state mut SimulationStateMut<'net>,
    /// A slice of indices of the current available transitions.
    pub transitions: Box<[u32]>,
    backwards: bool,
}

impl<'net, 'state> FireState<'net, 'state> {
    fn new(state: &'state mut SimulationStateMut<'net>) -> Self {
        FireState {
            transitions: state.fireable(),
            backwards: false,
            state,
        }
    }

    fn new_backwards(state: &'state mut SimulationStateMut<'net>) -> Self {
        FireState {
            transitions: state.unfireable(),
            backwards: true,
            state,
        }
    }

    /// Finish the fire by specifying the index of the transition index inside the slice of available indices.
    /// Not using the index of the transition directly may seem weird, but this way it's easy and efficient to ensure, no invalid transition will be called.
    /// And nomrally you know the index anyway and you save the work to get the index manually.
    /// As a result you get back your original state.
    pub fn finish(self, id: u32) {
        if self.backwards {
            unsafe {
                pnsState_fire_backwards(
                    self.state.state,
                    self.state.net,
                    self.transitions[id as usize],
                );
            }
        } else {
            unsafe {
                pnsState_fire(
                    self.state.state,
                    self.state.net,
                    self.transitions[id as usize],
                );
            }
        }
    }
}
