#include <stdbool.h>
#include <stdint.h>

#ifndef PNS_H_
#define PNS_H_

/* TYPES */

typedef struct PnsNode {
    uint32_t next_count;
    uint32_t next_size;
    uint32_t* next;
    uint32_t prev_count;
    uint32_t prev_size;
    uint32_t* prev;
} PnsNode;

typedef struct PnsIndexList {
    uint32_t index;
    struct PnsIndexList* next;
} PnsIndexList;

typedef struct PnsNet {
    uint32_t transition_count;
    uint32_t transitions_size;
    PnsNode* transitions;
    PnsIndexList* reusable_transitions;
    uint32_t place_count;
    uint32_t places_size;
    PnsNode* places;
    uint32_t* initial_token_counts;
    PnsIndexList* reusable_places;
    PnsIndexList* dirt;
    PnsIndexList* reverseDirt;
} PnsNet;

typedef struct PnsFireChanges {
    uint32_t count, added_count, removed_count;
    PnsIndexList* active, * added, * removed;
} PnsFireChanges;

typedef struct PnsState {
    uint32_t places_size;
    uint32_t* token_counts;
    uint32_t transitions_size;
    uint32_t* call_counts;
    PnsFireChanges fire;
    PnsFireChanges unfire;
} PnsState;

/* FUNCTIONS */

/** Net **/

/*** default ***/

/// initialize a new `Net`
void pnsCreateNet(PnsNet* net);
/// make a (deep) copy of a `Net`
void pnsCloneNet(PnsNet* net_clone, const PnsNet* net);
/// load a new `Net` from a u32 array; return value indicates success
bool pnsLoadNet(PnsNet* net, uint32_t count, const uint32_t* values);
/// destructor for `Net`
void pnsDestroyNet(PnsNet* net);
/// calculate the number of required u32 values for saving
uint32_t pnsNet_serializeSize(const PnsNet* net);
/// save `Net` to u32 array
void pnsNet_serialize(const PnsNet* net, uint32_t* values);

/*** edit ***/

/// add new place to a `Net` and return id
uint32_t pnsNet_addPlace(PnsNet* net);
/// add new transition to a `Net` and return its id
/// Caution: rather use `addConnectedTransition` when you want to extend the net safely (see safety.md)
uint32_t pnsNet_addTransition(PnsNet* net);
/// adds a new transition to a `Net` and connects it to specified places; useful for safety reasons (see safety.md)
uint32_t pnsNet_addConnectedTransition(PnsNet* net, uint32_t place_count, uint32_t* pids);
/// removes place `pid` from `net`
void pnsNet_removePlace(PnsNet* net, uint32_t pid);
/// removes transition `tid` from `net`
void pnsNet_removeTransition_unsafe(PnsNet* net, uint32_t tid);
/// create a new input connection of transition `tid` from place `pid`
bool pnsNet_connectIn_unsafe(PnsNet* net, uint32_t tid, uint32_t pid);
/// create a new output connection of transition `tid` to place `pid`
bool pnsNet_connectOut(PnsNet* net, uint32_t tid, uint32_t pid);
/// remove the input connection of transition `tid` from place `pid`
void pnsNet_disconnectIn(PnsNet* net, uint32_t tid, uint32_t pid);
/// remove the output connection of transition `tid` to place `pid`
void pnsNet_disconnectOut_unsafe(PnsNet* net, uint32_t tid, uint32_t pid);
/// duplicates a place, sharing all input and output places
uint32_t pnsNet_duplicateTransition(PnsNet* net, uint32_t tid);
/// duplicates a place, sharing all input and output transitions
uint32_t pnsNet_duplicatePlace(PnsNet* net, uint32_t pid);
/// increases the initial token count of the place specified by `pid` by `count`
uint32_t pnsNet_start(PnsNet* net, uint32_t pid, uint32_t count);
/// clears recent edits to `net`, so states won't check old changes anymore
void pnsNet_clearEdits(PnsNet* net);

/** State **/

/*** default ***/

/// initialize a new `State`
void pnsCreateState(PnsState* state, const PnsNet* net);
/// make a (deep) copy of a `State` (sharing the same `Net`)
void pnsCloneState(PnsState* state_clone, const PnsState* state, const PnsNet* net);
/// load a new `State` from a u32 array; return value indicates success
bool pnsLoadState(PnsState* state, const PnsNet* net, uint32_t count, const uint32_t* values);
/// destructor for `State`
void pnsDestroyState(PnsState* state);

/*** simulate ***/

/// get a list of currently available transitions
/// when `transitions` is `NULL`, `count will be set to the transition count
/// else `transitions` will be filled with up to `count` elements
/// `count` will be set to the indices in `transitions` and might be lower than specified
void pnsState_transitions(PnsState* state, uint32_t* count, uint32_t* transitions);
void pnsState_transitions_backwards(PnsState* state, uint32_t* count, uint32_t* transitions);
/// get a list of newly added transitions in specified direction
/// all written transitions will be synchronized and won't be processed until another change
void pnsState_addedTransitions(PnsState* state, uint32_t* count, uint32_t* transitions);
void pnsState_addedTransitions_backwards(PnsState* state, uint32_t* count, uint32_t* transitions);
/// get a list of newly removed transitions in specified direction
/// all written transitions will be synchronized and won't be processed until another change
void pnsState_removedTransitions(PnsState* state, uint32_t* count, uint32_t* transitions);
void pnsState_removedTransitions_backwards(PnsState* state, uint32_t* count, uint32_t* transitions);
/// this will remove all transition changes
/// Caution: only use after querying the current transitions directly and manually synchronizing your local state
void pnsState_cleanChanges(PnsState* state);
void pnsState_cleanChanges_backwards(PnsState* state);
/// fire the transition
/// Caution: ensure to only call valid transitions
/// (queried using the methods transitions or managed yourself using addedTransitions and removedTransitions)
void pnsState_fire(PnsState* state, const PnsNet* net, uint32_t tid);
void pnsState_fire_backwards(PnsState* state, const PnsNet* net, uint32_t tid);

/*** update ***/

/// updates petri net where dirty
void pnsState_updateEdits(PnsState* state, const PnsNet* net);

#endif
