// Copyright Claudio Mattera 2021.
// Distributed under the MIT License.
// See accompanying file License.txt, or online at
// https://opensource.org/licenses/MIT

#[cfg(test)]
mod tests {
    use anyhow::Result;

    use png2wasm4src::{convert_png_to_rust_variables, Flags};

    const ONE_COLOR_BYTES: &[u8] = include_bytes!("one-color.png");
    const TWO_COLORS_BYTES: &[u8] = include_bytes!("two-colors.png");
    const THREE_COLORS_BYTES: &[u8] = include_bytes!("three-colors.png");
    const FOUR_COLORS_BYTES: &[u8] = include_bytes!("four-colors.png");

    // https://publicdomainvectors.org/en/free-clipart/Vector-illustration-of-side-view-of-green-car-pixel-art/27161.html
    const CAR_BYTES: &[u8] = include_bytes!("car.png");

    #[test]
    fn encode_one_color() -> Result<()> {
        let name = "one-color";

        let rust_variables = convert_png_to_rust_variables(name, ONE_COLOR_BYTES)?;

        let expected_width: u32 = 8;
        let expected_height: u32 = 8;
        let expected_flags: Flags = Flags::OneBitPerPixel;
        let expected_data: [u8; 8] = [0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff];

        assert_eq!(rust_variables.width(), expected_width);
        assert_eq!(rust_variables.height(), expected_height);
        assert_eq!(rust_variables.flags(), expected_flags);
        assert_eq!(rust_variables.data(), expected_data);

        Ok(())
    }

    #[test]
    fn encode_one_color_string() -> Result<()> {
        let name = "one-color";

        let rust_variables = convert_png_to_rust_variables(name, ONE_COLOR_BYTES)?;
        let rust_code = rust_variables.to_string();

        let expected = "const ONE_COLOR_WIDTH: u32 = 8;
const ONE_COLOR_HEIGHT: u32 = 8;
const ONE_COLOR_FLAGS: u32 = 0; // BLIT_1BPP
const ONE_COLOR: [u8; 8] = [0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff];\n";

        assert_eq!(rust_code, expected);

        Ok(())
    }

    #[test]
    fn encode_two_colors() -> Result<()> {
        let name = "two-colors";

        let rust_variables = convert_png_to_rust_variables(name, TWO_COLORS_BYTES)?;

        let expected_width: u32 = 8;
        let expected_height: u32 = 8;
        let expected_flags: Flags = Flags::OneBitPerPixel;
        let expected_data: [u8; 8] = [0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0];

        assert_eq!(rust_variables.width(), expected_width);
        assert_eq!(rust_variables.height(), expected_height);
        assert_eq!(rust_variables.flags(), expected_flags);
        assert_eq!(rust_variables.data(), expected_data);

        Ok(())
    }

    #[test]
    fn encode_two_colors_string() -> Result<()> {
        let name = "two-colors";

        let rust_variables = convert_png_to_rust_variables(name, TWO_COLORS_BYTES)?;
        let rust_code = rust_variables.to_string();

        let expected = "const TWO_COLORS_WIDTH: u32 = 8;
const TWO_COLORS_HEIGHT: u32 = 8;
const TWO_COLORS_FLAGS: u32 = 0; // BLIT_1BPP
const TWO_COLORS: [u8; 8] = [0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0];\n";

        assert_eq!(rust_code, expected);

        Ok(())
    }

    #[test]
    fn encode_three_colors() -> Result<()> {
        let name = "three-colors";

        let rust_variables = convert_png_to_rust_variables(name, THREE_COLORS_BYTES)?;

        let expected_width: u32 = 4;
        let expected_height: u32 = 4;
        let expected_flags: Flags = Flags::TwoBitsPerPixel;
        let expected_data: [u8; 4] = [0x5a, 0x5a, 0xfa, 0xfa];

        assert_eq!(rust_variables.width(), expected_width);
        assert_eq!(rust_variables.height(), expected_height);
        assert_eq!(rust_variables.flags(), expected_flags);
        assert_eq!(rust_variables.data(), expected_data);

        Ok(())
    }

    #[test]
    fn encode_three_colors_string() -> Result<()> {
        let name = "three-colors";

        let rust_variables = convert_png_to_rust_variables(name, THREE_COLORS_BYTES)?;
        let rust_code = rust_variables.to_string();

        let expected = "const THREE_COLORS_WIDTH: u32 = 4;
const THREE_COLORS_HEIGHT: u32 = 4;
const THREE_COLORS_FLAGS: u32 = 1; // BLIT_2BPP
const THREE_COLORS: [u8; 4] = [0x5a, 0x5a, 0xfa, 0xfa];\n";

        assert_eq!(rust_code, expected);

        Ok(())
    }

    #[test]
    fn encode_four_colors() -> Result<()> {
        let name = "four-colors";

        let rust_variables = convert_png_to_rust_variables(name, FOUR_COLORS_BYTES)?;

        let expected_width: u32 = 4;
        let expected_height: u32 = 4;
        let expected_flags: Flags = Flags::TwoBitsPerPixel;
        let expected_data: [u8; 4] = [0x5a, 0x5a, 0xf0, 0xf0];

        assert_eq!(rust_variables.width(), expected_width);
        assert_eq!(rust_variables.height(), expected_height);
        assert_eq!(rust_variables.flags(), expected_flags);
        assert_eq!(rust_variables.data(), expected_data);

        Ok(())
    }

    #[test]
    fn encode_four_colors_string() -> Result<()> {
        let name = "four-colors";

        let rust_variables = convert_png_to_rust_variables(name, FOUR_COLORS_BYTES)?;
        let rust_code = rust_variables.to_string();

        let expected = "const FOUR_COLORS_WIDTH: u32 = 4;
const FOUR_COLORS_HEIGHT: u32 = 4;
const FOUR_COLORS_FLAGS: u32 = 1; // BLIT_2BPP
const FOUR_COLORS: [u8; 4] = [0x5a, 0x5a, 0xf0, 0xf0];\n";

        assert_eq!(rust_code, expected);

        Ok(())
    }

    #[test]
    fn encode_car() -> Result<()> {
        let name = "car";

        let rust_variables = convert_png_to_rust_variables(name, CAR_BYTES)?;

        let expected_width: u32 = 52;
        let expected_height: u32 = 23;
        let expected_flags: Flags = Flags::TwoBitsPerPixel;
        let expected_data: [u8; 299] = [
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x15, 0x5a, 0xaa, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x05, 0x6a, 0xaa, 0xaa, 0xa9, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x1f, 0x6a, 0xaa, 0xaa, 0xaa, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
            0x7f, 0x6a, 0xaa, 0xaa, 0xaa, 0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x5f, 0xff,
            0x6a, 0xaa, 0xaa, 0xaa, 0x7f, 0x55, 0x54, 0x00, 0x00, 0x15, 0xaa, 0x9f, 0xff, 0x6a,
            0xff, 0xff, 0xff, 0x7f, 0x6a, 0xa9, 0x00, 0x01, 0x6a, 0xaa, 0x7f, 0xff, 0xdf, 0xd5,
            0x55, 0x55, 0x5f, 0xda, 0xaa, 0x40, 0x16, 0xab, 0xfa, 0x7f, 0xff, 0xf5, 0x7f, 0xf7,
            0xff, 0xd7, 0xf6, 0xaa, 0x90, 0x1a, 0xba, 0xaa, 0x7f, 0xff, 0xd7, 0xff, 0xf7, 0xff,
            0xfd, 0xf6, 0xaa, 0xa4, 0x6b, 0xaa, 0xaa, 0x7f, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff,
            0x7d, 0xaa, 0xb4, 0x6a, 0xaa, 0xaa, 0x9f, 0xfd, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0x7d,
            0xaa, 0xf4, 0x6a, 0xaa, 0xaa, 0xa7, 0xd7, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0x9d, 0xff,
            0x59, 0x6a, 0xaa, 0xaa, 0xbd, 0x59, 0xff, 0xff, 0xdf, 0xff, 0xf6, 0xa5, 0x55, 0xa9,
            0x5a, 0xaa, 0xbf, 0xff, 0xd9, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5a, 0xaa, 0xa9, 0x67,
            0xff, 0xfd, 0x55, 0x66, 0xaa, 0xaa, 0x6a, 0xaa, 0xa9, 0xaa, 0xaa, 0xa9, 0x69, 0x55,
            0x56, 0xaa, 0x6a, 0xaa, 0x5a, 0x6a, 0xaa, 0x59, 0xaa, 0xaa, 0xa9, 0x6a, 0xaa, 0xaa,
            0xa9, 0xaa, 0xaa, 0xa9, 0xaa, 0xaa, 0xaa, 0x6a, 0xaa, 0xa4, 0x16, 0xa5, 0x55, 0x55,
            0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x06, 0x55, 0xf5, 0x59, 0xaa,
            0xaa, 0xa9, 0xaa, 0xaa, 0xa5, 0x5f, 0x55, 0x40, 0x01, 0x15, 0xf5, 0x45, 0x55, 0x55,
            0x55, 0x55, 0x55, 0x51, 0x5f, 0x54, 0x00, 0x00, 0x05, 0x55, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x55, 0x50, 0x00, 0x00, 0x01, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x15, 0x40, 0x00,
        ];

        assert_eq!(rust_variables.width(), expected_width);
        assert_eq!(rust_variables.height(), expected_height);
        assert_eq!(rust_variables.flags(), expected_flags);
        assert_eq!(rust_variables.data(), expected_data);

        Ok(())
    }

    #[test]
    fn encode_car_string() -> Result<()> {
        let name = "car";

        let rust_variables = convert_png_to_rust_variables(name, CAR_BYTES)?;
        let rust_code = rust_variables.to_string();

        let expected = "const CAR_WIDTH: u32 = 52;
const CAR_HEIGHT: u32 = 23;
const CAR_FLAGS: u32 = 1; // BLIT_2BPP
const CAR: [u8; 299] = [0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x5a, 0xaa, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x6a, 0xaa, 0xaa, 0xa9, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x6a, 0xaa, 0xaa, 0xaa, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x7f, 0x6a, 0xaa, 0xaa, 0xaa, 0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x5f, 0xff, 0x6a, 0xaa, 0xaa, 0xaa, 0x7f, 0x55, 0x54, 0x00, 0x00, 0x15, 0xaa, 0x9f, 0xff, 0x6a, 0xff, 0xff, 0xff, 0x7f, 0x6a, 0xa9, 0x00, 0x01, 0x6a, 0xaa, 0x7f, 0xff, 0xdf, 0xd5, 0x55, 0x55, 0x5f, 0xda, 0xaa, 0x40, 0x16, 0xab, 0xfa, 0x7f, 0xff, 0xf5, 0x7f, 0xf7, 0xff, 0xd7, 0xf6, 0xaa, 0x90, 0x1a, 0xba, 0xaa, 0x7f, 0xff, 0xd7, 0xff, 0xf7, 0xff, 0xfd, 0xf6, 0xaa, 0xa4, 0x6b, 0xaa, 0xaa, 0x7f, 0xff, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0x7d, 0xaa, 0xb4, 0x6a, 0xaa, 0xaa, 0x9f, 0xfd, 0x7f, 0xff, 0xdf, 0xff, 0xff, 0x7d, 0xaa, 0xf4, 0x6a, 0xaa, 0xaa, 0xa7, 0xd7, 0xff, 0xff, 0xdf, 0xff, 0xfd, 0x9d, 0xff, 0x59, 0x6a, 0xaa, 0xaa, 0xbd, 0x59, 0xff, 0xff, 0xdf, 0xff, 0xf6, 0xa5, 0x55, 0xa9, 0x5a, 0xaa, 0xbf, 0xff, 0xd9, 0x55, 0x55, 0x55, 0x55, 0x55, 0x5a, 0xaa, 0xa9, 0x67, 0xff, 0xfd, 0x55, 0x66, 0xaa, 0xaa, 0x6a, 0xaa, 0xa9, 0xaa, 0xaa, 0xa9, 0x69, 0x55, 0x56, 0xaa, 0x6a, 0xaa, 0x5a, 0x6a, 0xaa, 0x59, 0xaa, 0xaa, 0xa9, 0x6a, 0xaa, 0xaa, 0xa9, 0xaa, 0xaa, 0xa9, 0xaa, 0xaa, 0xaa, 0x6a, 0xaa, 0xa4, 0x16, 0xa5, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x50, 0x06, 0x55, 0xf5, 0x59, 0xaa, 0xaa, 0xa9, 0xaa, 0xaa, 0xa5, 0x5f, 0x55, 0x40, 0x01, 0x15, 0xf5, 0x45, 0x55, 0x55, 0x55, 0x55, 0x55, 0x51, 0x5f, 0x54, 0x00, 0x00, 0x05, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x50, 0x00, 0x00, 0x01, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x40, 0x00];
";

        assert_eq!(rust_code, expected);

        Ok(())
    }
}
