use crate::renderer::NetRenderer;

use femtovg::Renderer;
use gapp_winit::WindowInput;
use pn_editor_core::{Editor, ViewMode};
use winit::event::{
    ElementState, KeyboardInput, ModifiersState, MouseButton, VirtualKeyCode, WindowEvent,
};

pub struct InputContext {
    modifiers: ModifiersState,
}

impl InputContext {
    pub fn new() -> Self {
        Self {
            modifiers: ModifiersState::empty(),
        }
    }
}

impl<R: Renderer> WindowInput<InputContext, NetRenderer<R>> for Editor {
    fn input<'a>(&mut self, event: &WindowEvent<'a>, context: &mut InputContext) -> bool {
        match event {
            &WindowEvent::ModifiersChanged(modifiers) => context.modifiers = modifiers,

            WindowEvent::Resized(physical_size) => {
                self.resize(physical_size.width, physical_size.height)
            }

            WindowEvent::CursorMoved { position, .. } => {
                self.set_cursor(position.x as f32, position.y as f32)
            }

            WindowEvent::MouseWheel { delta, .. } => match delta {
                winit::event::MouseScrollDelta::LineDelta(_, roll) => self.zoom(
                    2.0f32.powf(roll / -0x10 as f32),
                    !context.modifiers.shift(),
                    context.modifiers.ctrl(),
                ),
                _ => (),
            },
            WindowEvent::MouseInput { button, state, .. } => {
                use MouseButton::*;
                let primary = match button {
                    Left => true,
                    Right => false,
                    _ => return false,
                };
                use ElementState::*;
                match state {
                    Pressed => {
                        if context.modifiers.alt() {
                            self.fire_node(primary)
                        } else {
                            self.grab(
                                primary,
                                !context.modifiers.shift(),
                                context.modifiers.ctrl(),
                            )
                        }
                    }
                    Released => self.release(
                        primary,
                        context.modifiers.ctrl(),
                        !context.modifiers.shift(),
                        !context.modifiers.alt(),
                    ),
                }
            }
            WindowEvent::ReceivedCharacter(c) => self.text_input(*c, context.modifiers.ctrl()),
            WindowEvent::KeyboardInput {
                input:
                    KeyboardInput {
                        virtual_keycode: Some(keycode),
                        state,
                        ..
                    },
                ..
            } => {
                use VirtualKeyCode::*;
                if *state == ElementState::Pressed {
                    match keycode {
                        Escape => self.escape(),
                        Return => self.finish(context.modifiers.shift(), context.modifiers.ctrl()),
                        Left => self.move_text_cursor(false, context.modifiers.ctrl()),
                        Right => self.move_text_cursor(true, context.modifiers.ctrl()),
                        Up => self.move_line(false),
                        Down => self.move_line(true),
                        PageUp => self.move_page(false),
                        PageDown => self.move_page(true),
                        Home => self.move_text_cursor_border(false, context.modifiers.ctrl()),
                        End => self.move_text_cursor_border(true, context.modifiers.ctrl()),
                        Tab => self.cycle(!context.modifiers.ctrl()),
                        Delete if context.modifiers.alt() => self.remove_nodes(),
                        code if context.modifiers.alt() => match code {
                            S => self.save_progress(context.modifiers.shift()),
                            O => self.load_progress(),
                            _ => (),
                        },
                        code if context.modifiers.ctrl() => match code {
                            NumpadAdd | Plus => self.start_nodes(),
                            S => self.save(context.modifiers.shift()),
                            O => self.load(!context.modifiers.shift()),
                            G => self.toggle_snapping(),
                            E => self.toggle_editor(),
                            F => self.start_search_mode(),
                            N => self.load_new(),
                            P => self.switch_mode(),
                            A => self.add_state(),
                            C => self.copy_selected(),
                            X => self.cut_selected(),
                            V => self.paste(context.modifiers.shift()),
                            Key1 => self.set_view_mode(ViewMode::Default),
                            Key2 => self.set_view_mode(ViewMode::State),
                            Key3 => self.set_view_mode(ViewMode::Actions),
                            _ => (),
                        },
                        _ => (),
                    }
                }
            }
            WindowEvent::CloseRequested => {
                if self.ensure_saved(false) {
                    return true;
                }
            }
            _ => (),
        }
        false
    }

    fn resize(&self, renderer: &mut NetRenderer<R>, w: u32, h: u32) {
        renderer.canvas.set_size(w, h, 1.0);
    }
}
