use femtovg::{Align, Baseline, Canvas, Color, FontId, Paint, Path, Renderer};
use nalgebra::Vector2;
use pn_editor_core::{ArrowKind, Renderer as GraphRenderer};
use std::ops::Range;
use text_editing::TextLine;

pub struct NetRenderer<R: Renderer> {
    text_size: f32,
    text_width: f32,
    pub font: FontId,
    pub canvas: Canvas<R>,
}

impl<R: Renderer> NetRenderer<R> {
    pub fn new(canvas: Canvas<R>, font: FontId) -> Self {
        Self {
            canvas,
            text_size: 16.0,
            text_width: 128.0,
            font,
        }
    }
}

impl<R: Renderer> NetRenderer<R> {
    fn draw_cursor_range(
        &mut self,
        x: f32,
        y: f32,
        zoom: f32,
        text: &str,
        text_cursor: usize,
        range: Range<usize>,
        text_paint: Paint,
    ) {
        if range.contains(&text_cursor) || (text.len() == range.end && text_cursor == range.end) {
            let text_cursor = text_cursor - range.start;
            self.draw_cursor(x, y, zoom, &text[range], text_cursor, text_paint);
        }
    }

    fn draw_cursor(
        &mut self,
        mut x: f32,
        mut y: f32,
        zoom: f32,
        text: &str,
        text_cursor: usize,
        text_paint: Paint,
    ) {
        if let (Ok(metrics), Ok(full_metrics)) = (
            self.canvas
                .measure_text(x, y, &text[0..text_cursor], text_paint),
            self.canvas.measure_text(x, y, text, text_paint),
        ) {
            x += metrics.width();
            match text_paint.text_align() {
                Align::Left => (),
                Align::Center => x -= full_metrics.width() / 2.0,
                Align::Right => x -= full_metrics.width(),
            }
            let h = self.text_size / 2.0 / zoom;
            match text_paint.text_baseline() {
                Baseline::Top => y += h,
                Baseline::Middle => (),
                Baseline::Alphabetic => (),
                Baseline::Bottom => y -= h,
            }
            let mut path = Path::new();
            path.move_to(x, y - h);
            path.line_to(x, y + h);
            self.canvas.stroke_path(&mut path, text_paint);
        }
    }
}

impl<R: Renderer> GraphRenderer for NetRenderer<R> {
    fn draw_simulated(&mut self, simulated: bool) {
        let sh = 0.5;
        let color = if simulated {
            Color::rgbf(sh, sh, sh + 1.0 / 8.0)
        } else {
            Color::rgbf(sh, sh, sh - 1.0 / 8.0)
        };

        self.canvas.clear_rect(
            0,
            0,
            self.canvas.width() as u32,
            self.canvas.height() as u32,
            color,
        );
    }

    fn draw_place(
        &mut self,
        x: f32,
        y: f32,
        radius: f32,
        zoom: f32,
        name: &TextLine,
        text_cursor: usize,
        show_cursor: bool,
        count: u32,
        marked: bool,
    ) {
        let x = x / zoom;
        let y = y / zoom;

        let mut path = Path::new();
        let (start_color, end_color) = if count == 0 {
            (Color::rgb(0xE0, 0x40, 0x40), Color::rgb(0xA0, 0x20, 0xA0))
        } else {
            (Color::rgb(0x20, 0x20, 0xE0), Color::rgb(0x20, 0xE0, 0xE0))
        };
        path.circle(x, y, radius / zoom);
        let fill_paint = Paint::radial_gradient(
            x - radius / zoom / 16.0,
            y + radius / zoom / 16.0,
            radius / zoom * 0.5,
            radius / zoom,
            start_color,
            end_color,
        );
        self.canvas.fill_path(&mut path, fill_paint);
        let (stroke_color, stroke_width) = if marked {
            (Color::rgb(0xC0, 0xC0, 0), 4.0)
        } else {
            (Color::black(), 2.0)
        };
        let mut stroke_paint = Paint::color(stroke_color);
        stroke_paint.set_line_width(stroke_width);
        self.canvas.stroke_path(&mut path, stroke_paint);

        let mut text_paint = Paint::color(Color::white());
        text_paint.set_font(&[self.font]);
        text_paint.set_font_size(self.text_size / zoom);
        text_paint.set_text_align(Align::Center);
        text_paint.set_text_baseline(Baseline::Middle);

        let _ = self
            .canvas
            .fill_text(x, y, format!("{}", count), text_paint);

        let text = name.as_str();

        if let Ok(split_text) = self
            .canvas
            .break_text_vec(self.text_width / zoom, text, text_paint)
        {
            for (i, range) in split_text.into_iter().enumerate() {
                let (x, y) = (x, y + self.text_size * (i as f32 + 1.5) / zoom);

                let _ = self
                    .canvas
                    .fill_text(x, y, &text[range.clone()], text_paint);

                if show_cursor {
                    self.draw_cursor_range(
                        x,
                        y,
                        zoom,
                        text,
                        name.string_index(text_cursor),
                        range,
                        text_paint,
                    );
                }
            }
        }
    }

    fn draw_transition(
        &mut self,
        x: f32,
        y: f32,
        width: f32,
        height: f32,
        zoom: f32,
        name: &TextLine,
        text_cursor: usize,
        show_cursor: bool,
        marked: bool,
        forward: bool,
        backwards: bool,
    ) {
        let x = x / zoom;
        let y = y / zoom;

        let mut path = Path::new();
        let (start_color, end_color) = match (forward, backwards) {
            (true, true) => (Color::rgb(0x60, 0xC0, 0x80), Color::rgb(0x40, 0xA0, 0x60)),
            (true, false) => (Color::rgb(0xC0, 0xA0, 0x40), Color::rgb(0xE0, 0xC0, 0x60)),
            (false, true) => (Color::rgb(0x40, 0x80, 0xE0), Color::rgb(0x40, 0x60, 0xA0)),
            (false, false) => (Color::rgb(0x80, 0xE0, 0xE0), Color::rgb(0xE0, 0xA0, 0xA0)),
        };
        let xmin = x - width / 2.0 / zoom;
        let ymin = y - height / 2.0 / zoom;
        let xmax = x + width / 2.0 / zoom;
        let ymax = y + height / 2.0 / zoom;

        path.rounded_rect(xmin, ymin, width / zoom, height / zoom, 4.0);
        let fill_paint = Paint::linear_gradient(xmin, ymin, xmax, ymax, start_color, end_color);
        self.canvas.fill_path(&mut path, fill_paint);
        let (stroke_color, stroke_width) = if marked {
            (Color::rgb(0xC0, 0xC0, 0), 4.0)
        } else {
            (Color::black(), 2.0)
        };
        let mut stroke_paint = Paint::color(stroke_color);
        stroke_paint.set_line_width(stroke_width);
        self.canvas.stroke_path(&mut path, stroke_paint);

        let mut text_paint = Paint::color(Color::black());
        text_paint.set_font(&[self.font]);
        text_paint.set_font_size(self.text_size / zoom);
        text_paint.set_text_align(Align::Center);
        text_paint.set_text_baseline(Baseline::Middle);

        let text = name.as_str();

        if let Ok(split_text) = self.canvas.break_text_vec(width / zoom, text, text_paint) {
            let count = split_text.len();
            for (i, range) in split_text.into_iter().enumerate() {
                let (x, y) = (
                    x,
                    y + self.text_size * (i as f32 - (count - 1) as f32 / 2.0) / zoom,
                );

                let _ = self
                    .canvas
                    .fill_text(x, y, &text[range.clone()], text_paint);

                if show_cursor {
                    self.draw_cursor_range(
                        x,
                        y,
                        zoom,
                        text,
                        name.string_index(text_cursor),
                        range,
                        text_paint,
                    );
                }
            }
        }
    }

    fn draw_line(&mut self, from_x: f32, from_y: f32, to_x: f32, to_y: f32, zoom: f32) {
        let from_x = from_x / zoom;
        let from_y = from_y / zoom;
        let to_x = to_x / zoom;
        let to_y = to_y / zoom;
        let mut path = Path::new();
        path.move_to(from_x, from_y);
        path.line_to(to_x, to_y);
        let mut paint = Paint::color(Color::rgbaf(1.0, 0.5, 0.5, 0.5));
        paint.set_line_width(2.0);
        self.canvas.stroke_path(&mut path, paint);
    }

    fn draw_arrow(
        &mut self,
        from_x: f32,
        from_y: f32,
        to_x: f32,
        to_y: f32,
        radius: f32,
        width: f32,
        height: f32,
        zoom: f32,
        arrow_kind: ArrowKind,
    ) {
        use ArrowKind::*;
        let (from_point, to_point) = match arrow_kind {
            PointToPlace | PointToTransition => (true, false),
            PlaceToPoint | TransitionToPoint => (false, true),
            PlaceToTransition | TransitionToPlace => (false, false),
        };

        let from = Vector2::new(from_x, from_y) / zoom;
        let to = Vector2::new(to_x, to_y) / zoom;
        let dv = to - from;
        let poffset = dv.normalize() * radius / zoom;
        let (toffset, tdirection) =
            if dv.x.abs() + height / 2.0 / zoom > dv.y.abs() + width / 2.0 / zoom {
                let y = dv.y / (dv.x.abs() + (height - width) / 2.0 / zoom);
                (
                    Vector2::new(if dv.x > 0.0 { width } else { -width }, y * height) / zoom / 2.0,
                    Vector2::new(if dv.x > 0.0 { 1.0 } else { -1.0 }, y * y * y).normalize(),
                )
            } else {
                let x = dv.x / (dv.y.abs() + (width - height) / 2.0 / zoom);
                (
                    Vector2::new(x * width, if dv.y > 0.0 { height } else { -height }) / zoom / 2.0,
                    Vector2::new(x * x * x, if dv.y > 0.0 { 1.0 } else { -1.0 }).normalize(),
                )
            };
        let (from_offset, from_direction, to_offset, to_direction) = match arrow_kind {
            PlaceToPoint | PlaceToTransition | PointToTransition => {
                (poffset, poffset.normalize(), toffset, tdirection)
            }
            PointToPlace | TransitionToPlace | TransitionToPoint => {
                (toffset, tdirection, poffset, poffset.normalize())
            }
        };

        let from = if from_point { from } else { from + from_offset };
        let to = if to_point { to } else { to - to_offset };
        let mag = (to - from).norm();
        let from2 = from + from_direction * mag / 3.0;
        let to2 = to - to_direction * mag / 3.0;

        let last_point = match (from_point, to_point) {
            (true, true) => from,
            (false, true) => from2,
            (_, false) => to2,
        };

        let dir = (to - last_point).normalize();
        let tip = to;
        let to = tip - dir * 8.0;

        let mut path = Path::new();
        path.move_to(from.x, from.y);
        match (from_point, to_point) {
            (true, true) => path.line_to(to.x, to.y),
            (true, false) => path.quad_to(to2.x, to2.y, to.x, to.y),
            (false, true) => path.quad_to(from2.x, from2.y, to.x, to.y),
            (false, false) => path.bezier_to(from2.x, from2.y, to2.x, to2.y, to.x, to.y),
        }
        let mut paint = Paint::color(Color::black());
        paint.set_line_width(4.0);
        self.canvas.stroke_path(&mut path, paint);

        let mut path = Path::new();
        let orth = Vector2::new(dir.y, -dir.x);
        let left = tip - dir * 16.0 + orth * 8.0;
        let right = tip - dir * 16.0 - orth * 8.0;

        path.move_to(tip.x, tip.y);
        path.line_to(left.x, left.y);
        path.line_to(right.x, right.y);
        path.line_to(tip.x, tip.y);
        self.canvas.fill_path(&mut path, paint);
    }

    fn draw_field(&mut self, text: &TextLine, text_cursor: usize) {
        let xmin = 32.0;
        let xmax = self.canvas.width() - 32.0;
        let ymin = 32.0;
        let ymax = 64.0;

        let mut path = Path::new();
        path.rounded_rect(xmin, ymin, xmax - xmin, ymax - ymin, 16.0);

        self.canvas.fill_path(
            &mut path,
            Paint::linear_gradient(
                xmin,
                ymin,
                xmax,
                ymax,
                Color::rgb(0x80, 0x80, 0x80),
                Color::rgb(0x40, 0xFF, 0xC0),
            ),
        );
        let mut stroke_paint = Paint::color(Color::black());
        stroke_paint.set_line_width(2.0);
        self.canvas.stroke_path(&mut path, stroke_paint);

        let mut text_paint = Paint::color(Color::black());
        text_paint.set_font(&[self.font]);
        text_paint.set_font_size(24.0);
        text_paint.set_text_align(Align::Left);
        text_paint.set_text_baseline(Baseline::Middle);

        let (x, y) = (xmin + 16.0, (ymax + ymin) / 2.0);
        let _ = self.canvas.fill_text(x, y, &text.as_str(), text_paint);
        self.draw_cursor(
            x,
            y,
            1.0,
            text.as_str(),
            text.string_index(text_cursor),
            text_paint,
        );
    }

    fn draw_select_box(&mut self, x1: f32, y1: f32, x2: f32, y2: f32, zoom: f32) {
        let mut path = Path::new();
        path.rect(x1 / zoom, y1 / zoom, (x2 - x1) / zoom, (y2 - y1) / zoom);

        let mut paint = Paint::color(Color::rgbaf(0.0, 0.0, 0.0, 0.5));
        paint.set_line_width(2.0);

        self.canvas.stroke_path(&mut path, paint);
    }

    fn draw_text_box(
        &mut self,
        text: &Vec<TextLine>,
        text_cursor: usize,
        text_line: usize,
        text_offset: usize,
    ) {
        let mut path = Path::new();
        path.rounded_rect(
            16.0 + self.canvas.width() / 2.0,
            8.0,
            self.canvas.width() / 2.0 - 32.0,
            self.canvas.height() - 32.0,
            32.0,
        );
        self.canvas
            .fill_path(&mut path, Paint::color(Color::rgba(0xFF, 0xFF, 0xFF, 0xc0)));

        let mut stroke_paint = Paint::color(Color::rgba(0, 0, 0, 0x80));
        stroke_paint.set_line_width(4.0);
        self.canvas.stroke_path(&mut path, stroke_paint);

        let mut text_paint = Paint::color(Color::black());
        text_paint.set_font(&[self.font]);
        text_paint.set_font_size(16.0);
        text_paint.set_text_align(Align::Left);
        text_paint.set_text_baseline(Baseline::Top);

        let (x, y) = (32.0 + self.canvas.width() / 2.0, 32.0);
        let w = self.canvas.width() / 2.0 - 64.0;

        let line_count = (self.canvas.height() / self.text_size * 4.0) as usize - 5;

        let mut current_line = 0;
        for (i, line) in text[text_offset..].iter().enumerate() {
            if let Ok(split_text) = self.canvas.break_text_vec(w, line.as_str(), text_paint) {
                if split_text.is_empty() {
                    if i == text_line - text_offset {
                        let y = y + self.text_size * current_line as f32;
                        self.draw_cursor(x, y, 1.0, "", text_cursor, text_paint);
                    }
                    current_line += 1;
                } else {
                    for range in split_text {
                        let y = y + self.text_size * current_line as f32;

                        let _ =
                            self.canvas
                                .fill_text(x, y, &line.as_str()[range.clone()], text_paint);

                        if text_line - text_offset == i {
                            self.draw_cursor_range(
                                x,
                                y,
                                1.0,
                                line.as_str(),
                                line.string_index(text_cursor),
                                range,
                                text_paint,
                            );
                        }
                        current_line += 1;
                    }
                }

                if current_line >= line_count {
                    break;
                }
            }
        }
    }

    fn finish(&mut self) {
        self.canvas.flush();
    }
}
