use gapp::Update;

use crate::edit_list::NodeKind;
use crate::feature_types::{Grab, GrabKind, MoveType};

use super::Editor;

impl Update for Editor {
    fn update(&mut self, _timestep: f32) {
        if let Some(Grab {
            kind, moved: true, ..
        }) = self.grab
        {
            if let GrabKind::Move(MoveType::Nodes) | GrabKind::Connect | GrabKind::Select = kind {
                const WS: f32 = 6.0;
                let movefac = self.zoom * 0x400 as f32;
                let window_left = (self.mouse_pos.x - self.window_size.x / WS) / self.window_size.x;
                if window_left < 0.0 {
                    self.view_offset.x -= window_left.powf(2.0) * movefac;
                }
                let window_right = (1.0 - self.mouse_pos.x + self.window_size.x * (WS - 1.0) / WS)
                    / self.window_size.x;
                if window_right < 0.0 {
                    self.view_offset.x += window_right.powf(2.0) * movefac;
                }
                let window_up = (self.mouse_pos.y - self.window_size.y / WS) / self.window_size.y;
                if window_up < 0.0 {
                    self.view_offset.y -= window_up.powf(2.0) * movefac;
                }
                let window_down = (1.0 - self.mouse_pos.y + self.window_size.y * (WS - 1.0) / WS)
                    / self.window_size.y;
                if window_down < 0.0 {
                    self.view_offset.y += window_down.powf(2.0) * movefac;
                }
            }
        }

        let relative_mouse_pos = self.relative_mouse_pos();
        if let Some(Grab {
            pos: grab_pos,
            kind: GrabKind::Move(move_type),
            ..
        }) = &mut self.grab
        {
            let movevec = relative_mouse_pos - *grab_pos;

            match move_type {
                MoveType::Nodes => {
                    for (pid, subindices) in self.selected.pids.iter() {
                        let positions = &mut self.place_nodes[*pid as usize].positions;
                        for &subindex in subindices {
                            positions[subindex] += movevec;
                        }
                    }
                    for (tid, subindices) in self.selected.tids.iter() {
                        let positions = &mut self.transition_nodes[*tid as usize].positions;
                        for &subindex in subindices {
                            positions[subindex] += movevec;
                        }
                    }
                    *grab_pos += movevec;
                }
                MoveType::View => self.view_offset -= movevec,
            }
        }

        if let Some(string_searcher) = &self.string_searcher {
            if let Some(index) = string_searcher.index {
                use NodeKind::*;
                if let Some(goal_pos) = match index.kind {
                    Place => {
                        let id = index.index;
                        let pid = *string_searcher.list.pids.iter().nth(id as usize).unwrap();
                        let node = &self.place_nodes[pid as usize];
                        node.positions.iter().fold(None, |result, pos| {
                            let new_distance = (pos - self.view_offset).norm_squared();
                            if let Some((position, distance)) = result {
                                if new_distance >= distance {
                                    return Some((position, distance));
                                }
                            }
                            Some((pos, new_distance))
                        })
                    }
                    Transition => {
                        let id = index.index;
                        let tid = *string_searcher.list.tids.iter().nth(id as usize).unwrap();
                        let node = &self.transition_nodes[tid as usize];
                        node.positions.iter().fold(None, |result, pos| {
                            let new_distance = (pos - self.view_offset).norm_squared();
                            if let Some((position, distance)) = result {
                                if new_distance >= distance {
                                    return Some((position, distance));
                                }
                            }
                            Some((pos, new_distance))
                        })
                    }
                } {
                    self.view_offset += (goal_pos.0 - self.view_offset) / 4.0;
                }
            }
        }
    }
}
