use gapp::Render;
use nalgebra::Vector2;

use crate::feature_types::{Grab, GrabKind, ViewMode};
use crate::renderer::{ArrowKind, Renderer};

use super::Editor;

impl Editor {
    // helpers

    fn is_inside(&self, pos: Vector2<f32>, size: Vector2<f32>) -> bool {
        let pos = pos - self.view_offset;
        pos.x + size.x / 2.0 > -self.window_size.x * self.zoom / 2.0
            && pos.x - size.x / 2.0 < self.window_size.x * self.zoom / 2.0
            && pos.y + size.y / 2.0 > -self.window_size.y * self.zoom / 2.0
            && pos.y - size.y / 2.0 < self.window_size.y * self.zoom / 2.0
    }

    fn token_count(&self, index: usize) -> u32 {
        if let Some(net) = self.net.simulated() {
            net.state(self.simulation).token_counts()[index]
                + self.net.safe().initial_token_counts()[index]
        } else {
            self.net.safe().initial_token_counts()[index]
        }
    }

    // rendering

    fn draw_arcs<R: Renderer>(&self, renderer: &mut R) {
        fn nearest_position_pair(
            positions: &Vec<Vector2<f32>>,
            other_positions: &Vec<Vector2<f32>>,
        ) -> Option<(Vector2<f32>, Vector2<f32>)> {
            use itertools::iproduct;
            if let Some((pos, other_pos, _)) =
                iproduct!(positions, other_positions).fold(None, |result, (pos, other_pos)| {
                    let new_distance = (other_pos - pos).norm_squared();
                    if let Some((_, _, distance)) = result {
                        if new_distance >= distance {
                            return result;
                        }
                    }
                    Some((pos, other_pos, new_distance))
                })
            {
                Some((*pos, *other_pos))
            } else {
                None
            }
        }

        use ViewMode::*;
        match self.view_mode {
            Actions => {
                for place in self.net.safe().places() {
                    if place.next_count == 0 || place.prev_count == 0 {
                        continue;
                    }
                    let mut next_offset = Vector2::new(0.0, 0.0);
                    let mut prev_offset = Vector2::new(0.0, 0.0);

                    for tid in place.next() {
                        let positions = &self.transition_nodes[*tid as usize].positions;
                        if !positions.is_empty() {
                            let mut current_offset = Vector2::new(0.0, 0.0);
                            for next_transform in positions {
                                current_offset += next_transform;
                            }
                            next_offset += current_offset / positions.len() as f32;
                        }
                    }

                    for tid in place.prev() {
                        let positions = &self.transition_nodes[*tid as usize].positions;
                        if !positions.is_empty() {
                            let mut current_offset = Vector2::new(0.0, 0.0);
                            for prev_transform in positions {
                                current_offset += prev_transform;
                            }
                            prev_offset += current_offset / positions.len() as f32;
                        }
                    }

                    let offset = (next_offset / place.next_count as f32
                        + prev_offset / place.prev_count as f32)
                        / 2.0;

                    for tid in place.next() {
                        for next_transform in &self.transition_nodes[*tid as usize].positions {
                            next_offset += next_transform;
                            let from =
                                offset - self.view_offset + self.window_size / 2.0 * self.zoom;
                            let to = next_transform - self.view_offset
                                + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::PointToTransition,
                            );
                        }
                    }

                    for tid in place.prev() {
                        for prev_transform in &self.transition_nodes[*tid as usize].positions {
                            prev_offset += prev_transform - offset;
                            let from = prev_transform - self.view_offset
                                + self.window_size / 2.0 * self.zoom;
                            let to = offset - self.view_offset + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::TransitionToPoint,
                            );
                        }
                    }
                }
            }

            State => {
                for transition in self.net.safe().transitions() {
                    if transition.next_count == 0 || transition.prev_count == 0 {
                        continue;
                    }
                    let mut next_offset = Vector2::new(0.0, 0.0);
                    let mut prev_offset = Vector2::new(0.0, 0.0);

                    for pid in transition.next() {
                        let positions = &self.place_nodes[*pid as usize].positions;
                        if !positions.is_empty() {
                            let mut current_offset = Vector2::new(0.0, 0.0);
                            for next_transform in positions {
                                current_offset += next_transform;
                            }
                            next_offset += current_offset / positions.len() as f32;
                        }
                    }

                    for pid in transition.prev() {
                        let positions = &self.place_nodes[*pid as usize].positions;
                        if !positions.is_empty() {
                            let mut current_offset = Vector2::new(0.0, 0.0);
                            for prev_transform in positions {
                                current_offset += prev_transform;
                            }
                            prev_offset += current_offset / positions.len() as f32;
                        }
                    }

                    let offset = (next_offset / transition.next_count as f32
                        + prev_offset / transition.prev_count as f32)
                        / 2.0;

                    for pid in transition.next() {
                        for next_transform in &self.place_nodes[*pid as usize].positions {
                            next_offset += next_transform - offset;
                            let from =
                                offset - self.view_offset + self.window_size / 2.0 * self.zoom;
                            let to = next_transform - self.view_offset
                                + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::PointToPlace,
                            );
                        }
                    }

                    for pid in transition.prev() {
                        for prev_transform in &self.place_nodes[*pid as usize].positions {
                            prev_offset += prev_transform - offset;
                            let from = prev_transform - self.view_offset
                                + self.window_size / 2.0 * self.zoom;
                            let to = offset - self.view_offset + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::PlaceToPoint,
                            );
                        }
                    }
                }
            }

            Default => {
                for (node, place) in self.place_nodes.iter().zip(self.net.safe().places()) {
                    for tid in place.next() {
                        if let Some((pos, next_pos)) = nearest_position_pair(
                            &node.positions,
                            &self.transition_nodes[*tid as usize].positions,
                        ) {
                            let from = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                            let to =
                                next_pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::PlaceToTransition,
                            );
                        }
                    }

                    for tid in place.prev() {
                        if let Some((pos, prev_pos)) = nearest_position_pair(
                            &node.positions,
                            &self.transition_nodes[*tid as usize].positions,
                        ) {
                            let from =
                                prev_pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                            let to = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                            renderer.draw_arrow(
                                from.x,
                                from.y,
                                to.x,
                                to.y,
                                self.node_settings.radius,
                                self.node_settings.width,
                                self.node_settings.height,
                                self.zoom,
                                ArrowKind::TransitionToPlace,
                            );
                        }
                    }
                }
            }
        }
    }
}

impl<R: Renderer> Render<R> for Editor {
    fn render(&self, renderer: &mut R) {
        renderer.draw_simulated(self.net.simulated().is_some());
        if self.view_mode != ViewMode::Actions {
            for (&pid, subindices) in self.selected.pids.iter() {
                let node = &self.place_nodes[pid as usize];
                if !subindices.is_empty() {
                    let mut center = Vector2::new(0.0, 0.0);
                    for subindex in subindices {
                        center += node.positions[*subindex];
                    }
                    center /= subindices.len() as f32;

                    for pos in &node.positions {
                        let from = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        let to = center - self.view_offset + self.window_size / 2.0 * self.zoom;
                        renderer.draw_line(from.x, from.y, to.x, to.y, self.zoom);
                    }
                }
            }
        }

        if self.view_mode != ViewMode::State {
            for &tid in self.selected.tids.keys() {
                let node = &self.transition_nodes[tid as usize];
                if node.positions.len() > 1 {
                    let mut center = Vector2::new(0.0, 0.0);
                    for pos in &node.positions {
                        center += pos;
                    }
                    center /= node.positions.len() as f32;

                    for pos in &node.positions {
                        let from = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        let to = center - self.view_offset + self.window_size / 2.0 * self.zoom;
                        renderer.draw_line(from.x, from.y, to.x, to.y, self.zoom);
                    }
                }
            }
        }

        self.draw_arcs(renderer);

        if self.view_mode != ViewMode::Actions {
            for (index, node) in self.place_nodes.iter().enumerate() {
                let show_cursor = self.selected.pids.contains_key(&(index as u32))
                    && (!self.text_content_shown || self.selected.tids.len() != 1);

                for (subindex, pos) in node.positions.iter().enumerate() {
                    let count = self.token_count(index);
                    if self.is_inside(
                        *pos,
                        Vector2::new(
                            self.node_settings.radius * 2.0,
                            self.node_settings.radius * 2.0,
                        ),
                    ) {
                        let pos = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        renderer.draw_place(
                            pos.x,
                            pos.y,
                            self.node_settings.radius,
                            self.zoom,
                            &node.name,
                            node.name_cursor,
                            show_cursor,
                            count,
                            self.selected.pids.contains(&(index as u32), &subindex),
                        );
                    }
                }
            }

            if let Some(Grab {
                kind: GrabKind::Connect,
                ..
            }) = self.grab
            {
                for (pid, subindices) in self.selected.pids.iter() {
                    let node = &self.place_nodes[*pid as usize];
                    for &subindex in subindices {
                        let pos = &node.positions[subindex];
                        let from = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        let to = self.mouse_pos * self.zoom;
                        renderer.draw_arrow(
                            from.x,
                            from.y,
                            to.x,
                            to.y,
                            self.node_settings.radius,
                            self.node_settings.width,
                            self.node_settings.height,
                            self.zoom,
                            ArrowKind::PlaceToPoint,
                        );
                    }
                }
            }
        }

        if self.view_mode != ViewMode::State {
            for (index, node) in self.transition_nodes.iter().enumerate() {
                let (forward, backwards) = if let Some(_) = self.net.simulated() {
                    let info = &self.state_infos[self.simulation];
                    info.callables[index]
                } else {
                    (false, false)
                };
                let show_cursor = self.selected.tids.contains_key(&(index as u32))
                    && (!self.text_content_shown || self.selected.tids.len() != 1);
                for (subindex, pos) in node.positions.iter().enumerate() {
                    if self.is_inside(
                        *pos,
                        Vector2::new(self.node_settings.width, self.node_settings.height),
                    ) {
                        let pos = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        renderer.draw_transition(
                            pos.x,
                            pos.y,
                            self.node_settings.width,
                            self.node_settings.height,
                            self.zoom,
                            &node.name,
                            node.name_cursor,
                            show_cursor,
                            self.selected.tids.contains(&(index as u32), &subindex),
                            forward,
                            backwards,
                        );
                    }
                }
            }

            if let Some(Grab {
                kind: GrabKind::Connect,
                ..
            }) = self.grab
            {
                for (tid, subindices) in self.selected.tids.iter() {
                    let node = &self.transition_nodes[*tid as usize];
                    for &subindex in subindices {
                        let pos = &node.positions[subindex];
                        let from = pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                        let to = self.mouse_pos * self.zoom;
                        renderer.draw_arrow(
                            from.x,
                            from.y,
                            to.x,
                            to.y,
                            self.node_settings.radius,
                            self.node_settings.width,
                            self.node_settings.height,
                            self.zoom,
                            ArrowKind::TransitionToPoint,
                        );
                    }
                }
            }
        }

        if let Some(Grab {
            pos: grab_pos,
            kind: grab_kind,
            ..
        }) = self.grab
        {
            if grab_kind == GrabKind::Select {
                let local_grab_pos =
                    grab_pos - self.view_offset + self.window_size / 2.0 * self.zoom;
                let real_mouse_pos = self.mouse_pos * self.zoom;
                renderer.draw_select_box(
                    local_grab_pos.x,
                    local_grab_pos.y,
                    real_mouse_pos.x,
                    real_mouse_pos.y,
                    self.zoom,
                );
            }
        }

        if let Some(string_searcher) = &self.string_searcher {
            renderer.draw_field(&string_searcher.name, string_searcher.name_cursor);
        }

        if self.text_content_shown && self.selected.tids.len() == 1 {
            let node = &self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
            renderer.draw_text_box(
                &node.content,
                node.content_cursor,
                node.content_line,
                node.content_offset,
            );
        }

        renderer.finish();
    }
}
