use copypasta::ClipboardProvider;
use nalgebra::Vector2;
use pns::{DynamicNet, Net, SafeNet, SimulatedNet};
use rfd::{FileDialog, MessageButtons, MessageDialog, MessageLevel};
use text_editing::TextLine;

use std::{
    collections::{HashMap, HashSet},
    fs::File,
    io::{BufRead, BufReader, Write},
};

fn folder_name(path: &str) -> String {
    if let Some(pos) = path.rfind('/') {
        path[(pos + 1)..].to_string()
    } else {
        path.to_string()
    }
}

fn snap(pos: &mut Vector2<f32>, snap_size: f32) {
    pos.x = (pos.x / snap_size).round() * snap_size;
    pos.y = (pos.y / snap_size).round() * snap_size;
}

use crate::edit_list::{EditList, NodeId, NodeKind, SelectionList};
use crate::feature_types::{DuplicationType, Grab, GrabKind, MoveType, StringSearcher, ViewMode};
use crate::node::Node;
use crate::state_info::StateInfo;

use super::Editor;
impl Editor {
    // helpers

    fn snapped_relative_mouse_pos(&self) -> Vector2<f32> {
        let mut new_pos = self.relative_mouse_pos();
        if let Some(snap_size) = self.snapping {
            snap(&mut new_pos, snap_size);
        }
        new_pos
    }

    fn add_transition(
        &mut self,
        tid: u32,
        positions: &[Vector2<f32>],
        name: Option<String>,
    ) -> Vec<NodeId> {
        self.changes = true;
        let name = if let Some(name) = name {
            name
        } else {
            if tid == 0 {
                "Intro".to_string()
            } else {
                format!("Act {}", tid)
            }
        };
        let new_node = Node::new(positions, name);
        if tid < self.transition_nodes.len() as u32 {
            self.transition_nodes[tid as usize] = new_node;
        } else {
            self.transition_nodes.push(new_node);
            if let DynamicNet::Simulated(net) = &mut self.net {
                for (info, state) in &mut self.state_infos.iter_mut().zip(&mut net.states) {
                    info.callables.push((false, false));
                    info.update_state(state);
                }
            }
        }
        (0..positions.len())
            .map(|subindex| NodeId {
                kind: NodeKind::Transition,
                index: tid,
                subindex,
            })
            .collect()
    }

    fn add_transition_linked(&mut self, tid: u32, positions: &[Vector2<f32>]) -> Vec<NodeId> {
        self.changes = true;
        let node = &mut self.transition_nodes[tid as usize];
        let old_len = node.positions.len();
        node.positions.extend(positions);
        (0..positions.len())
            .map(|subindex| NodeId {
                kind: NodeKind::Transition,
                index: tid,
                subindex: subindex + old_len,
            })
            .collect()
    }

    fn add_place(
        &mut self,
        pid: u32,
        positions: &[Vector2<f32>],
        name: Option<String>,
    ) -> Vec<NodeId> {
        self.changes = true;
        let new_node = Node::new(
            positions,
            if let Some(name) = name {
                name
            } else {
                String::new()
            },
        );
        if pid < self.place_nodes.len() as u32 {
            self.place_nodes[pid as usize] = new_node;
        } else {
            self.place_nodes.push(new_node);
        }
        (0..positions.len())
            .map(|subindex| NodeId {
                kind: NodeKind::Place,
                index: pid,
                subindex,
            })
            .collect()
    }

    fn add_place_linked(&mut self, pid: u32, positions: &[Vector2<f32>]) -> Vec<NodeId> {
        self.changes = true;
        let node = &mut self.place_nodes[pid as usize];
        let old_len = node.positions.len();
        node.positions.extend(positions);
        (0..positions.len())
            .map(|subindex| NodeId {
                kind: NodeKind::Place,
                index: pid,
                subindex: subindex + old_len,
            })
            .collect()
    }

    fn click_info(&mut self, click_pos: Vector2<f32>) -> Option<NodeId> {
        if self.view_mode != ViewMode::State {
            for (id, node) in self.transition_nodes.iter().enumerate().rev() {
                for (subindex, pos) in node.positions.iter().enumerate() {
                    if pos.x - self.node_settings.width / 2.0 < click_pos.x
                        && click_pos.x < pos.x + self.node_settings.width / 2.0
                        && pos.y - self.node_settings.height / 2.0 < click_pos.y
                        && click_pos.y < pos.y + self.node_settings.height / 2.0
                    {
                        return Some(NodeId {
                            kind: NodeKind::Transition,
                            index: id as u32,
                            subindex,
                        });
                    }
                }
            }
        }

        if self.view_mode != ViewMode::Actions {
            for (id, node) in self.place_nodes.iter().enumerate().rev() {
                for (subindex, pos) in node.positions.iter().enumerate() {
                    if (pos - click_pos).norm() < self.node_settings.radius {
                        return Some(NodeId {
                            kind: NodeKind::Place,
                            index: id as u32,
                            subindex,
                        });
                    }
                }
            }
        }
        None
    }

    fn box_info(&mut self) -> Vec<NodeId> {
        let mut result = Vec::new();
        if let Some(Grab { pos: grab_pos, .. }) = self.grab {
            let pos1 = grab_pos;
            let pos2 = self.relative_mouse_pos();
            let right = pos1.x.max(pos2.x);
            let left = pos1.x.min(pos2.x);
            let top = pos1.y.max(pos2.y);
            let bottom = pos1.y.min(pos2.y);

            if self.view_mode != ViewMode::State {
                for (id, node) in self.transition_nodes.iter().enumerate().rev() {
                    for (subindex, pos) in node.positions.iter().enumerate() {
                        if pos.x - self.node_settings.width / 2.0 < right
                            && left < pos.x + self.node_settings.width / 2.0
                            && pos.y - self.node_settings.height / 2.0 < top
                            && bottom < pos.y + self.node_settings.height / 2.0
                        {
                            result.push(NodeId {
                                kind: NodeKind::Transition,
                                index: id as u32,
                                subindex,
                            });
                        }
                    }
                }
            }

            if self.view_mode != ViewMode::Actions {
                for (id, node) in self.place_nodes.iter().enumerate().rev() {
                    for (subindex, pos) in node.positions.iter().enumerate() {
                        if pos.x - self.node_settings.radius < right
                            && left < pos.x + self.node_settings.radius
                            && pos.y - self.node_settings.radius < top
                            && bottom < pos.y + self.node_settings.radius
                        {
                            result.push(NodeId {
                                kind: NodeKind::Place,
                                index: id as u32,
                                subindex,
                            });
                        }
                    }
                }
            }
        }
        result
    }

    /// Switches simulation mode.
    pub fn switch_mode(&mut self) {
        use DynamicNet::*;
        if !self.ensure_saved(true) {
            return;
        }
        replace_with::replace_with_or_abort(&mut self.net, |old_net| match old_net {
            Default(net) => Simulated(SimulatedNet::from_net(net)),
            Simulated(net) => Default(net.release()),
        });
        if let Simulated(net) = &mut self.net {
            net.add_state();
            let mut info = StateInfo {
                state_path: None,
                changes: false,
                callables: (0..net.transition_count)
                    .map(|_index| (false, false))
                    .collect(),
            };
            self.simulation = 0;
            info.update_state(&mut net.states[0]);
            self.state_infos.push(info);
        } else {
            self.state_infos = Vec::new();
        }
    }

    fn switch_simulation(&mut self) {
        if let Some(net) = self.net.simulated() {
            self.simulation += 1;
            if self.simulation == net.len() {
                self.simulation = 0;
            }
        }
    }

    // edit

    /// Adds a token to nodes if they are places.
    pub fn start_nodes(&mut self) {
        self.changes = true;
        for pid in self.selected.pids.keys() {
            let net = self.net.safe_mut();
            net.start(*pid, 1);
        }
    }

    /// Removes all selected nodes.
    pub fn remove_nodes(&mut self) {
        self.changes = true;
        if self.view_mode != ViewMode::Actions {
            for (pid, subindices) in self.selected.pids.iter() {
                let positions = &mut self.place_nodes[*pid as usize].positions;
                for index in subindices.iter().rev() {
                    positions.remove(*index);
                }
                if positions.is_empty() {
                    self.net.safe_mut().remove_place(*pid);
                }
            }
        }
        if self.view_mode != ViewMode::State {
            if let Some(net) = self.net.default_mut() {
                for (tid, subindices) in self.selected.tids.iter() {
                    let positions = &mut self.transition_nodes[*tid as usize].positions;
                    for index in subindices.iter().rev() {
                        positions.remove(*index);
                    }
                    if positions.is_empty() {
                        net.remove_transition(*tid);
                    }
                }
            }
        }
        self.selected.clear()
    }

    /// Should be called when the window is resized.
    pub fn resize(&mut self, width: u32, height: u32) {
        self.window_size = Vector2::new(width as f32, height as f32);
    }

    /// Cycle through states.
    pub fn cycle(&mut self, forward: bool) {
        if let Some(string_searcher) = &mut self.string_searcher {
            if let Some(index) = &mut string_searcher.index {
                use NodeKind::*;
                match index.kind {
                    Place => {
                        if forward {
                            if !string_searcher.list.tids.is_empty() {
                                index.kind = Transition;
                                index.index = 0;
                            }
                        } else {
                            index.index += 1;
                            if string_searcher.list.pids.len() as u32 == index.index {
                                if !string_searcher.list.tids.is_empty() {
                                    index.kind = Transition;
                                }
                                index.index = 0;
                            }
                        }
                    }
                    Transition => {
                        if forward {
                            if !string_searcher.list.pids.is_empty() {
                                index.kind = Place;
                                index.index = 0;
                            }
                        } else {
                            index.index += 1;
                            if string_searcher.list.tids.len() as u32 == index.index {
                                if !string_searcher.list.pids.is_empty() {
                                    index.kind = Place;
                                }
                                index.index = 0;
                            }
                        }
                    }
                }
            }
        } else {
            self.switch_simulation()
        }
    }

    /// Adds a new simulation state.
    pub fn add_state(&mut self) {
        if let DynamicNet::Simulated(net) = &mut self.net {
            net.add_state();
            let new_index = net.states.len() - 1;
            let mut info = StateInfo {
                state_path: None,
                changes: false,
                callables: (0..net.transition_count)
                    .map(|_index| (false, false))
                    .collect(),
            };
            self.simulation = new_index;
            info.update_state(&mut net.state_mut(new_index).state);
            self.state_infos.push(info);
        }
    }

    /// Toggles snapping.
    pub fn toggle_snapping(&mut self) {
        if self.snapping.is_some() {
            self.snapping = None;
        } else {
            self.snapping = Some(16.0)
        }
    }

    /// Toggles the editor.
    pub fn toggle_editor(&mut self) {
        self.text_content_shown = !self.text_content_shown;
        self.string_searcher = None;
    }

    /// Enables the search bar.
    pub fn start_search_mode(&mut self) {
        if self.string_searcher.is_none() {
            self.text_content_shown = false;
            self.string_searcher = Some(StringSearcher {
                name: TextLine::new(),
                name_cursor: 0,
                index: None,
                list: EditList::new(),
            });
        }
    }

    /// Loads in an empty net.
    pub fn load_new(&mut self) {
        if self.ensure_saved(false) {
            self.net_path = None;
            self.reset(Net::new(), Vec::new(), Vec::new())
        }
    }

    /// Copy selected elements to clipboard.
    pub fn copy_selected(&mut self) {
        if let Some(data) = self.copy_data() {
            let _result = self.clipboard.set_contents(data);
        }
    }

    /// Cut selected elements to clipboard.
    pub fn cut_selected(&mut self) {
        self.copy_selected();
        self.remove_nodes()
    }

    /// Pastes contents from clipboard.
    pub fn paste(&mut self, clear_old: bool) {
        if let Ok(content) = self.clipboard.get_contents() {
            self.paste_data(&content, clear_old);
        }
    }

    /// Escapes from submenus.
    pub fn escape(&mut self) {
        self.string_searcher = None;
        self.text_content_shown = false;
    }

    /// Changes the view mode.
    /// See `ViewMode` for more information about the view modes.
    pub fn set_view_mode(&mut self, view_mode: ViewMode) {
        self.view_mode = view_mode;
    }

    /// Input text.
    pub fn text_input(&mut self, character: char, skip: bool) {
        fn edit_string(
            back_multi: bool,
            text: &mut TextLine,
            index: &mut usize,
            character: char,
        ) -> bool {
            let whitespace = character.is_whitespace();
            let control = character.is_control();
            if !whitespace && !control {
                text.insert(*index, character);
                *index += 1;
            } else if whitespace && character != '\t' {
                if *index != 0 && text.char_at(*index - 1) != ' ' {
                    if *index == text.len() || text.char_at(*index) != ' ' {
                        text.insert(*index, ' ');
                    }
                    *index += 1;
                }
            } else if character == '\x08' && *index > 0 {
                if back_multi {
                    while *index > 0 {
                        *index -= 1;
                        let c = text.remove(*index);
                        if !c.is_alphanumeric() {
                            break;
                        }
                    }
                } else {
                    *index -= 1;
                    text.remove(*index);
                }
                if *index == 0 && text.len() > 0 && text.char_at(0).is_whitespace() {
                    text.remove(0);
                }
            } else if character == '\x7F' && *index < text.len() {
                if back_multi {
                    while *index < text.len() {
                        let c = text.remove(*index);
                        if !c.is_alphanumeric() {
                            break;
                        }
                    }
                } else {
                    text.remove(*index);
                }
            } else {
                return false;
            }
            true
        }

        if character == '\n' || character == '\r' {
            return;
        }
        if character == '\x08' || character == '\x7F' || !skip {
            if let Some(string_searcher) = &mut self.string_searcher {
                if !edit_string(
                    skip,
                    &mut string_searcher.name,
                    &mut string_searcher.name_cursor,
                    character,
                ) {
                    return;
                }
                let subtext = &string_searcher.name;
                if subtext.is_empty() {
                    string_searcher.list.pids.clear();
                    string_searcher.list.tids.clear();
                    string_searcher.index = None;
                } else {
                    string_searcher.list.pids = self
                        .place_nodes
                        .iter()
                        .enumerate()
                        .filter_map(|(pid, node)| {
                            let node_name = node.name.as_str();
                            let subtext = subtext.as_str();
                            if node_name.contains(&subtext) {
                                return Some(pid as u32);
                            }
                            None
                        })
                        .collect();
                    string_searcher.list.tids = self
                        .transition_nodes
                        .iter()
                        .enumerate()
                        .filter_map(|(tid, node)| {
                            let node_name = node.name.as_str();
                            let subtext = subtext.as_str();
                            if node_name.contains(&subtext) {
                                return Some(tid as u32);
                            }

                            None
                        })
                        .collect();
                    match (
                        string_searcher.list.pids.is_empty(),
                        string_searcher.list.tids.is_empty(),
                    ) {
                        (true, true) => string_searcher.index = None,
                        (true, false) => {
                            string_searcher.index = Some(NodeId {
                                kind: NodeKind::Transition,
                                index: 0,
                                subindex: 0,
                            })
                        }
                        (false, true) => {
                            string_searcher.index = Some(NodeId {
                                kind: NodeKind::Place,
                                index: 0,
                                subindex: 0,
                            })
                        }
                        (false, false) => {
                            if let Some(index) = &mut string_searcher.index {
                                index.index = 0;
                            } else {
                                string_searcher.index = Some(NodeId {
                                    kind: NodeKind::Transition,
                                    index: 0,
                                    subindex: 0,
                                });
                            }
                        }
                    }
                }
            } else if self.text_content_shown && self.selected.tids.len() == 1 {
                self.changes = true;
                let node =
                    &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
                if node.content_cursor == 0 && character == '\x08' && node.content_line > 0 {
                    let line = node.content.remove(node.content_line);
                    node.content_line -= 1;
                    if node.content_offset > 0 {
                        node.content_offset -= 1;
                    }
                    node.content_cursor = node.content[node.content_line].len();
                    node.content[node.content_line].join(line);
                } else {
                    edit_string(
                        skip,
                        &mut node.content[node.content_line],
                        &mut node.content_cursor,
                        character,
                    );
                }
            } else {
                self.changes = true;
                for tid in self.selected.tids.keys() {
                    let node = &mut self.transition_nodes[*tid as usize];
                    edit_string(skip, &mut node.name, &mut node.name_cursor, character);
                }
                for pid in self.selected.pids.keys() {
                    let node = &mut self.place_nodes[*pid as usize];
                    edit_string(skip, &mut node.name, &mut node.name_cursor, character);
                }
            }
        }
    }

    /// Finish the current action.
    /// Basically the user presses the enter key.
    pub fn finish(&mut self, clear_old: bool, skip: bool) {
        if self.text_content_shown && self.selected.tids.len() == 1 {
            self.changes = true;
            let node =
                &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
            let line = node.content[node.content_line].split(node.content_cursor);
            node.content_line += 1;
            let line_count = (self.window_size.y / self.node_settings.height * 2.0) as usize - 5;
            if node.content_line >= node.content_offset + line_count {
                node.content_offset = node.content_line - line_count;
            }
            node.content_cursor = 0;
            node.content.insert(node.content_line, line);
        } else if let Some(string_searcher) = &mut self.string_searcher {
            if clear_old {
                self.selected.clear();
            }
            if skip {
                for pid in &string_searcher.list.pids {
                    if !self.selected.pids.contains_key(pid) {
                        self.selected.pids.insert(*pid, 0);
                    }
                }
                for tid in &string_searcher.list.tids {
                    if !self.selected.tids.contains_key(tid) {
                        self.selected.tids.insert(*tid, 0);
                    }
                }
            } else {
                if let Some(index) = string_searcher.index {
                    use NodeKind::*;
                    match index.kind {
                        Place => {
                            let pid = *string_searcher
                                .list
                                .pids
                                .iter()
                                .nth(index.index as usize)
                                .unwrap();
                            if !self.selected.pids.contains_key(&pid) {
                                self.selected.pids.insert(pid, 0);
                            }
                        }
                        Transition => {
                            let tid = *string_searcher
                                .list
                                .tids
                                .iter()
                                .nth(index.index as usize)
                                .unwrap();
                            if !self.selected.tids.contains_key(&tid) {
                                self.selected.tids.insert(tid, 0);
                            }
                        }
                    }
                }
            }
            self.string_searcher = None;
        }
    }

    /// Zoom in or out by a factor.
    pub fn zoom(&mut self, factor: f32, zoom_scale: bool, zoom_distance: bool) {
        self.view_offset = self.relative_mouse_pos();
        if zoom_scale {
            self.zoom *= factor;
        }
        if zoom_distance || !zoom_scale {
            for node in &mut self.transition_nodes {
                for position in &mut node.positions {
                    *position -= self.view_offset;
                    *position *= factor;
                    *position += self.view_offset;
                }
            }
            for node in &mut self.place_nodes {
                for position in &mut node.positions {
                    *position -= self.view_offset;
                    *position *= factor;
                    *position += self.view_offset;
                }
            }
        }
        self.view_offset = self.view_offset * 2.0 - self.relative_mouse_pos();
    }

    /// Release something.
    pub fn release(&mut self, primary: bool, allow_more: bool, clear_old: bool, select_new: bool) {
        if let Some(Grab {
            kind: GrabKind::Move(_),
            moved: false,
            deselect,
            ..
        }) = self.grab
        {
            let click_pos = self.relative_mouse_pos();
            if let Some(node_id) = self.click_info(click_pos) {
                if clear_old {
                    self.selected.clear();
                    self.selected.add(node_id);
                } else {
                    if deselect {
                        self.selected.remove_node(node_id);
                    }
                }
            } else {
                if clear_old {
                    self.selected.clear()
                }
            }
        }

        if !primary {
            if let Some(Grab { kind, .. }) = self.grab {
                match kind {
                    GrabKind::Select => {
                        let node_ids = self.box_info();
                        if clear_old && select_new {
                            self.selected.clear();
                        }
                        for node_id in node_ids {
                            if !select_new {
                                self.selected.remove_node(node_id);
                            } else if !self.selected.contains(&node_id) {
                                self.selected.add(node_id);
                            }
                        }
                    }
                    GrabKind::Connect => self.release_connect(allow_more, clear_old, select_new),
                    GrabKind::Move(_) => (),
                }
            }
        }
        if let Some(snap_size) = self.snapping {
            if let Some(Grab {
                kind: GrabKind::Move(move_type),
                ..
            }) = self.grab
            {
                match move_type {
                    MoveType::Nodes => {
                        for (tid, subindices) in self.selected.tids.iter() {
                            let node = &mut self.transition_nodes[*tid as usize];
                            for &index in subindices {
                                snap(&mut node.positions[index], snap_size);
                            }
                        }
                        for (pid, subindices) in self.selected.pids.iter() {
                            let node = &mut self.place_nodes[*pid as usize];
                            for &index in subindices {
                                snap(&mut node.positions[index], snap_size);
                            }
                        }
                    }
                    MoveType::View => snap(&mut self.view_offset, snap_size),
                }
            }
        }
        self.grab = None;
    }

    /// Fire node.
    pub fn fire_node(&mut self, forward: bool) {
        if let Some(node_id) = self.click_info(self.relative_mouse_pos()) {
            if let Some(net) = self.net.simulated_mut() {
                if let NodeKind::Transition = node_id.kind {
                    let state = &mut net.state_mut(self.simulation);
                    let info = &mut self.state_infos[self.simulation];
                    let fire = if forward {
                        state.fire()
                    } else {
                        state.unfire()
                    };
                    for (i, id) in fire.transitions.iter().enumerate() {
                        if *id == node_id.index {
                            fire.finish(i as u32);
                            info.changes = true;
                            info.update_state(&mut state.state);
                            break;
                        }
                    }
                }
            }
        }
    }

    /// Grab something.
    ///
    /// * `primary`: specifies if the primary button is pressed.
    /// * `clear_old`: specifies if old selections should be deselected.
    /// * `create`: creates new nodes.
    pub fn grab(&mut self, primary: bool, clear_old: bool, create: bool) {
        if let Some(node_id) = self.click_info(self.relative_mouse_pos()) {
            self.grab_node(node_id, !primary, clear_old, create);
        } else {
            self.grab_empty(!primary, clear_old, create)
        }
    }

    /// Set cursor position.
    pub fn set_cursor(&mut self, x: f32, y: f32) {
        if let Some(Grab {
            moved, duplicate, ..
        }) = &mut self.grab
        {
            if !*moved {
                *moved = true;

                if let Some(duplication_type) = duplicate {
                    let linked = *duplication_type == DuplicationType::Linked;

                    struct NodeInfo {
                        kind: NodeKind,
                        index: u32,
                        name: String,
                        positions: Vec<Vector2<f32>>,
                    }
                    let mut node_infos: Vec<NodeInfo> = Vec::new();
                    match self.view_mode {
                        ViewMode::Default => {
                            let pids = &self.selected.pids;
                            let tids = &self.selected.tids;

                            let mut place_mappings = HashMap::new();
                            let mut connecting_tids = Vec::new();
                            for (pid, subindices) in pids.iter() {
                                let index = if !linked {
                                    let place = &self.net.safe_mut().places()[*pid as usize];
                                    let mut duplicate = true;
                                    for tid in place.prev().iter().chain(place.next()) {
                                        if tids.contains_key(tid) {
                                            duplicate = false;
                                        } else {
                                            connecting_tids.push(*tid);
                                        }
                                    }
                                    if duplicate {
                                        self.net.safe_mut().duplicate_place(*pid)
                                    } else {
                                        let net = self.net.safe_mut();
                                        let index = net.add_place();
                                        net.start(index, net.initial_token_counts()[*pid as usize]);
                                        place_mappings.insert(*pid, index);
                                        index
                                    }
                                } else {
                                    *pid
                                };
                                let node = &self.place_nodes[*pid as usize];
                                node_infos.push(NodeInfo {
                                    kind: NodeKind::Place,
                                    index,
                                    name: node.name.to_string(),
                                    positions: subindices
                                        .iter()
                                        .map(|&subindex| node.positions[subindex])
                                        .collect(),
                                });
                            }
                            for (tid, subindices) in tids.iter() {
                                let index = if !linked {
                                    let transition =
                                        &self.net.safe_mut().transitions()[*tid as usize];
                                    let mut prev_ids = Vec::new();
                                    let mut next_ids = Vec::new();
                                    for pid in transition.prev() {
                                        let mapped_pid = if let Some(pid) = place_mappings.get(pid)
                                        {
                                            *pid
                                        } else {
                                            *pid
                                        };
                                        prev_ids.push(mapped_pid);
                                    }
                                    for pid in transition.next() {
                                        let mapped_pid = if let Some(pid) = place_mappings.get(pid)
                                        {
                                            *pid
                                        } else {
                                            *pid
                                        };
                                        next_ids.push(mapped_pid);
                                    }
                                    let index =
                                        self.net.safe_mut().add_connected_transition(&prev_ids[..]);

                                    for pid in next_ids {
                                        self.net.safe_mut().connect_out(index, pid);
                                    }
                                    index
                                } else {
                                    *tid
                                };
                                let node = &self.transition_nodes[*tid as usize];
                                node_infos.push(NodeInfo {
                                    kind: NodeKind::Transition,
                                    index,
                                    name: node.name.to_string(),
                                    positions: subindices
                                        .iter()
                                        .map(|&subindex| node.positions[subindex])
                                        .collect(),
                                });
                            }
                            if !linked {
                                for tid in connecting_tids {
                                    let transition =
                                        &self.net.safe_mut().transitions()[tid as usize];
                                    let mut prev_ids = Vec::new();
                                    let mut next_ids = Vec::new();
                                    for pid in transition.prev() {
                                        let mapped_pid = if let Some(pid) = place_mappings.get(pid)
                                        {
                                            *pid
                                        } else {
                                            *pid
                                        };
                                        prev_ids.push(mapped_pid);
                                    }
                                    for pid in transition.next() {
                                        let mapped_pid = if let Some(pid) = place_mappings.get(pid)
                                        {
                                            *pid
                                        } else {
                                            *pid
                                        };
                                        next_ids.push(mapped_pid);
                                    }
                                    if let Some(net) = self.net.default_mut() {
                                        for pid in prev_ids {
                                            net.connect_in(tid, pid);
                                        }
                                    }
                                    for pid in next_ids {
                                        self.net.safe_mut().connect_out(tid, pid);
                                    }
                                }
                            }
                        }
                        ViewMode::State => {
                            for (pid, subindices) in self.selected.pids.iter() {
                                let index = if linked {
                                    *pid
                                } else {
                                    self.net.safe_mut().duplicate_place(*pid)
                                };
                                let node = &self.place_nodes[*pid as usize];
                                node_infos.push(NodeInfo {
                                    kind: NodeKind::Place,
                                    index,
                                    name: node.name.to_string(),
                                    positions: subindices
                                        .iter()
                                        .map(|&subindex| node.positions[subindex])
                                        .collect(),
                                });
                            }
                        }
                        ViewMode::Actions => {
                            for (tid, subindices) in self.selected.tids.iter() {
                                let index = if linked {
                                    *tid
                                } else {
                                    self.net.safe_mut().duplicate_transition(*tid)
                                };
                                let node = &self.transition_nodes[*tid as usize];
                                node_infos.push(NodeInfo {
                                    kind: NodeKind::Transition,
                                    index,
                                    name: node.name.to_string(),
                                    positions: subindices
                                        .iter()
                                        .map(|&subindex| node.positions[subindex])
                                        .collect(),
                                });
                            }
                        }
                    }
                    let mut new_nodes = Vec::new();
                    for NodeInfo {
                        kind,
                        index,
                        name,
                        positions,
                    } in node_infos
                    {
                        use NodeKind::*;
                        match kind {
                            Transition => {
                                if self.view_mode == ViewMode::State {
                                    continue;
                                }
                                let transition = if linked {
                                    self.add_transition_linked(index, &positions[..])
                                } else {
                                    self.add_transition(index, &positions[..], Some(name))
                                };
                                new_nodes.extend(transition);
                            }
                            Place => {
                                if self.view_mode == ViewMode::Actions {
                                    continue;
                                }
                                let place = if linked {
                                    self.add_place_linked(index, &positions[..])
                                } else {
                                    self.add_place(index, &positions[..], Some(name))
                                };
                                new_nodes.extend(place);
                            }
                        }
                    }
                    let mut new_edit = SelectionList::new();
                    for node in new_nodes {
                        new_edit.add(node);
                    }
                    self.selected = new_edit;
                }
            }
        }

        self.mouse_pos = Vector2::new(x, y);
    }

    fn file_dialog(&self) -> FileDialog {
        let dialog = FileDialog::new();

        if let Some(path) = &self.net_path {
            dialog.set_directory(path)
        } else {
            dialog
        }
    }

    /// Saves the current progress of a simulation.
    pub fn save_progress(&mut self, force_path: bool) {
        if self.net.simulated().is_some() {
            let info = &mut self.state_infos[self.simulation];
            if info.state_path.is_none() || force_path {
                if let Some(path) = info.file_dialog().save_file() {
                    info.state_path = Some(path.as_path().display().to_string());
                } else {
                    return;
                }
            }
            if let Some(net) = self.net.simulated() {
                net.state(self.simulation)
                    .save(info.state_path.as_ref().unwrap());
            }
            info.changes = false;
        }
    }

    /// Saves the current net.
    pub fn save(&mut self, force_path: bool) {
        use std::collections::HashSet as Set;
        let mut names: Set<&str> = Set::new();
        for node in &self.transition_nodes {
            if node.positions.is_empty() {
                continue;
            }
            if names.contains(node.name.as_str()) {
                MessageDialog::new()
                .set_level(MessageLevel::Warning)
                .set_title("Warning!")
                .set_description(                    &format!(
                        "Cannot Save!\nTransition name '{}' found multiple times.\nRename transitions to ensure only one has this name.",
                        node.name.as_str()
                    )[..],
)
                .set_buttons(MessageButtons::Ok)
                .show();
                return;
            } else {
                names.insert(node.name.as_str());
            }
        }

        if self.net_path.is_none() || force_path {
            if let Some(path) = self.file_dialog().pick_folder() {
                self.net_path = Some(path.as_path().display().to_string());
            } else {
                return;
            }
        }

        let folder_path = self.net_path.as_ref().unwrap();
        let folder_name = folder_name(folder_path.as_str());

        let pns = format!("{}/{}.pns", folder_path, folder_name);
        let pnl = format!("{}/{}.pnl", folder_path, folder_name);
        let pnk = format!("{}/{}.pnk", folder_path, folder_name);
        let pnkp = format!("{}/{}.pnkp", folder_path, folder_name);
        let story = format!("{}/{}.story", folder_path, folder_name);

        let net = self.net.safe();
        if let Some(snap_size) = self.snapping {
            snap(&mut self.view_offset, snap_size);
        }

        net.save(&pns);

        if let Ok(mut layout) = File::create(pnl) {
            for node in self.transition_nodes.iter() {
                if !node.positions.is_empty() {
                    if node.positions.len() > 1 {
                        unsafe {
                            let _failed = layout.write_all(&std::mem::transmute::<_, [u8; 8]>((
                                std::f32::NAN,
                                node.positions.len() as u32,
                            )));
                        }
                    }
                    for pos in &node.positions {
                        unsafe {
                            let _failed = layout.write_all(&std::mem::transmute::<_, [u8; 8]>(
                                pos - self.view_offset,
                            ));
                        }
                    }
                } else {
                    unsafe {
                        let _failed =
                            layout.write_all(&std::mem::transmute::<_, [u8; 8]>(0x7FFFFFFFu64));
                    }
                }
            }
            for node in self.place_nodes.iter() {
                if !node.positions.is_empty() {
                    if node.positions.len() > 1 {
                        unsafe {
                            let _failed = layout.write_all(&std::mem::transmute::<_, [u8; 8]>((
                                std::f32::NAN,
                                node.positions.len() as u32,
                            )));
                        }
                    }
                    for pos in &node.positions {
                        unsafe {
                            let _failed = layout.write_all(&std::mem::transmute::<_, [u8; 8]>(
                                pos - self.view_offset,
                            ));
                        }
                    }
                } else {
                    unsafe {
                        let _failed =
                            layout.write_all(&std::mem::transmute::<_, [u8; 8]>(0x7FFFFFFFu64));
                    }
                }
            }
        }

        if let Ok(mut keys) = File::create(pnk) {
            for node in self.transition_nodes.iter() {
                if !node.positions.is_empty() {
                    let line = node.name.as_str();
                    let _failed = writeln!(keys, "{}", line);
                } else {
                    let _failed = writeln!(keys, "");
                }
            }
        }

        if let Ok(mut place_keys) = File::create(pnkp) {
            for node in self.place_nodes.iter() {
                if !node.positions.is_empty() {
                    let line = node.name.as_str();
                    let _failed = writeln!(place_keys, "{}", line);
                } else {
                    let _failed = writeln!(place_keys, "");
                }
            }
        }

        if let Ok(mut story) = File::create(story) {
            for node in self.transition_nodes.iter() {
                if !node.positions.is_empty() {
                    let name = node.name.as_str();
                    let lines: Vec<_> = node.content.iter().collect();
                    if !lines.is_empty() {
                        let _failed = writeln!(story, "# {}", name);
                        let _failed = writeln!(story);
                        for line in lines {
                            let line = line.as_str();
                            let _failed = writeln!(story, "{}", &line);
                        }
                        let _failed = writeln!(story);
                    }
                }
            }
        }

        self.changes = false;
    }

    /// Ensure the net be saved.
    pub fn ensure_saved(&mut self, state_only: bool) -> bool {
        let mut ask = if state_only { false } else { self.changes };
        if !ask {
            for info in &self.state_infos {
                if info.changes {
                    ask = true;
                    break;
                }
            }
        }
        if ask {
            MessageDialog::new()
                .set_level(MessageLevel::Warning)
                .set_title("Warning!")
                .set_description("There are unsaved changes, that will be lost\nAre you sure?")
                .set_buttons(MessageButtons::YesNo)
                .show()
        } else {
            true
        }
    }

    /// Loads the net from a specified path.
    pub fn load_net(&mut self, folder_path: String) {
        let folder_name = folder_name(folder_path.as_str());

        let pns = format!("{}/{}.pns", folder_path, folder_name);
        let pnl = format!("{}/{}.pnl", folder_path, folder_name);
        let pnk = format!("{}/{}.pnk", folder_path, folder_name);
        let pnkp = format!("{}/{}.pnkp", folder_path, folder_name);
        let story = format!("{}/{}.story", folder_path, folder_name);

        let net = if let Some(net) = Net::load(&pns) {
            net
        } else {
            Net::new()
        };
        self.net_path = Some(folder_path);
        let mut names = Vec::new();
        let mut name_set: HashSet<String> = HashSet::new();
        if let Ok(file) = File::open(pnk) {
            for line in BufReader::new(file).lines() {
                if let Ok(line) = line {
                    name_set.insert(line.clone());
                    names.push(line);
                } else {
                    return;
                }
            }
        }
        let mut place_names = Vec::new();
        if let Ok(file) = File::open(pnkp) {
            for line in BufReader::new(file).lines() {
                if let Ok(line) = line {
                    place_names.push(line)
                } else {
                    return;
                }
            }
        }
        let mut contents: HashMap<_, Vec<TextLine>> = HashMap::new();
        if let Ok(file) = File::open(story) {
            let mut first = false;
            let mut empty_count = 0;
            let mut current_key: Option<String> = None;
            let mut current_content = Vec::new();
            for line in BufReader::new(file).lines() {
                if let Ok(line) = line {
                    if line.chars().next() == Some('#') {
                        if let Some(key) = current_key {
                            if !name_set.contains(&key) {
                                names.push(key.clone());
                                name_set.insert(key.clone());
                            }
                            contents.insert(key, current_content);
                            current_content = Vec::new();
                            first = true;
                            empty_count = 0;
                        }

                        current_key = Some(line[1..].trim().to_string());
                    } else if line.is_empty() {
                        if first {
                            first = false;
                        } else {
                            empty_count += 1
                        }
                    } else if current_key.is_some() {
                        current_content.extend((0..empty_count).map(|_| TextLine::new()));
                        empty_count = 0;
                        current_content.push(TextLine::from_string(line));
                    }
                } else {
                    return;
                }
            }
            if let Some(key) = current_key {
                if !name_set.contains(&key) {
                    names.push(key.clone());
                    name_set.insert(key.clone());
                }
                contents.insert(key, current_content);
            }
        }
        self.reset(net, names, place_names);
        for node in &mut self.transition_nodes {
            if let Some(content) = contents.remove(&node.name.to_string()) {
                if !content.is_empty() {
                    node.content = content;
                }
            }
        }
        if let Ok(mut layout) = File::open(pnl) {
            for node in self.transition_nodes.iter_mut() {
                use std::io::Read;
                node.positions.clear();
                let mut buffer: [u8; 8] = unsafe { std::mem::zeroed() };
                let _failed = layout.read(&mut buffer);
                if 0x7FFFFFFFu64 != unsafe { std::mem::transmute(buffer) } {
                    let (first, count): (f32, u32) = unsafe { std::mem::transmute(buffer) };
                    if first.is_nan() {
                        for _i in 0..count {
                            let _failed = layout.read(&mut buffer);
                            node.positions.push(unsafe { std::mem::transmute(buffer) });
                        }
                    } else {
                        node.positions.push(unsafe { std::mem::transmute(buffer) });
                    }
                } else {
                    node.positions.clear();
                }
            }
            for node in self.place_nodes.iter_mut() {
                use std::io::Read;
                node.positions.clear();
                let mut buffer: [u8; 8] = unsafe { std::mem::zeroed() };
                let _failed = layout.read(&mut buffer);
                if 0x7FFFFFFFu64 != unsafe { std::mem::transmute(buffer) } {
                    let (first, count): (f32, u32) = unsafe { std::mem::transmute(buffer) };
                    if first.is_nan() {
                        for _i in 0..count {
                            let _failed = layout.read(&mut buffer);
                            node.positions.push(unsafe { std::mem::transmute(buffer) });
                        }
                    } else {
                        node.positions.push(unsafe { std::mem::transmute(buffer) });
                    }
                } else {
                    node.positions.clear();
                }
            }
        }
    }

    /// Loads the some progress for the simulation.
    pub fn load_progress(&mut self) {
        if self.net.simulated_mut().is_some() {
            let info = &mut self.state_infos[self.simulation];
            if let Some(path) = info.file_dialog().pick_file() {
                let path = path.as_path().display().to_string();
                if let Some(net) = self.net.simulated_mut() {
                    if let Some(_) = net.load_state(&path) {
                        info.state_path = Some(path);
                        self.simulation = net.states.len() - 1;
                    }
                }
            }
        }
    }

    /// Loads a new net.
    pub fn load(&mut self, force_path: bool) {
        if !self.ensure_saved(false) {
            return;
        }
        let path = if let (Some(path), false) = (&self.net_path, force_path) {
            path.clone()
        } else if let Some(path) = self.file_dialog().pick_folder() {
            path.as_path().display().to_string()
        } else {
            return;
        };

        self.load_net(path)
    }

    fn copy_data(&self) -> Option<String> {
        if self.selected.is_empty() {
            None
        } else {
            let mut result = "".to_string();
            let mut place_mappings = HashMap::new();
            let transition_count = self.selected.tids.len();
            for pid in self.selected.pids.keys() {
                place_mappings.insert(pid, place_mappings.len());
            }
            result = format!("{}{}\n", result, place_mappings.len());
            for (pid, subindices) in self.selected.pids.iter() {
                let node = &self.place_nodes[*pid as usize];
                let name = node.name.to_string();
                let count = self.net.safe().initial_token_counts()[*pid as usize];
                result = format!("{}{}\n{}\n{}\n", result, name, count, subindices.len());
                for &subindex in subindices {
                    let pos = &node.positions[subindex];
                    let pos = pos - self.relative_mouse_pos();
                    result = format!("{}{} {}\n", result, pos.x, pos.y);
                }
            }
            result = format!("{}{}\n", result, transition_count);
            for (tid, subindices) in self.selected.tids.iter() {
                let node = &self.transition_nodes[*tid as usize];
                let name = node.name.to_string();
                result = format!("{}{}\n{}\n", result, name, subindices.len());
                for &subindex in subindices {
                    let pos = &node.positions[subindex];
                    let pos = pos - self.relative_mouse_pos();
                    result = format!("{}{} {}\n", result, pos.x, pos.y);
                }
                let transition = &self.net.safe().transitions()[*tid as usize];
                for pid in transition.prev() {
                    if let Some(mapped_pid) = place_mappings.get(pid) {
                        result = format!("{}{} ", result, mapped_pid);
                    }
                }
                result = format!("{}\n", result);
                for pid in transition.next() {
                    if let Some(mapped_pid) = place_mappings.get(pid) {
                        result = format!("{}{} ", result, mapped_pid);
                    }
                }
                result = format!("{}\n", result);
            }
            Some(format!("Petri Net:\n{}", result))
        }
    }

    fn paste_data(&mut self, data: &str, clear_old: bool) -> Option<()> {
        self.changes = true;
        let mut lines = data.lines();
        let kind = lines.next()?;
        if kind.trim_end() != "Petri Net:" {
            return None;
        }
        let mut new_nodes = Vec::new();
        if let Ok(place_count) = lines.next()?.parse::<u32>() {
            let mut new_places = HashMap::new();
            for i in 0..place_count {
                let name = lines.next()?;
                let count = lines.next()?.parse::<u32>().ok()?;
                let mut positions = Vec::new();
                for _ in 0..lines.next()?.parse::<u32>().ok()? {
                    let mut pos_line = lines.next()?.split_whitespace();
                    let offset_x = pos_line.next()?.parse::<f32>().ok()?;
                    let offset_y = pos_line.next()?.parse::<f32>().ok()?;
                    positions.push(self.relative_mouse_pos() + Vector2::new(offset_x, offset_y));
                }
                let index = self.net.safe_mut().add_place();
                self.net.safe_mut().start(index, count);
                let nodes = self.add_place(index, &positions[..], None);
                new_places.insert(i, index);
                self.place_nodes[index as usize].name = TextLine::from_str(name);
                new_nodes.extend(nodes);
            }
            if let Ok(transition_count) = lines.next()?.parse::<u32>() {
                for _i in 0..transition_count {
                    let name = lines.next()?;
                    let mut positions = Vec::new();
                    for _ in 0..lines.next()?.parse::<u32>().ok()? {
                        let mut pos_line = lines.next()?.split_whitespace();
                        let offset_x = pos_line.next()?.parse::<f32>().ok()?;
                        let offset_y = pos_line.next()?.parse::<f32>().ok()?;
                        positions
                            .push(self.relative_mouse_pos() + Vector2::new(offset_x, offset_y));
                    }
                    let ins: Vec<u32> = lines
                        .next()?
                        .split_whitespace()
                        .map(|index| new_places[&index.parse::<u32>().ok().unwrap()])
                        .collect();
                    let outs = lines
                        .next()?
                        .split_whitespace()
                        .map(|index| new_places[&index.parse::<u32>().ok().unwrap()]);
                    let net = &mut self.net.safe_mut();
                    let index = net.add_connected_transition(&ins[..]);
                    for out in outs {
                        net.connect_out(index, out);
                    }
                    let nodes = self.add_transition(index, &positions[..], None);
                    self.transition_nodes[index as usize].name = TextLine::from_str(name);
                    new_nodes.extend(nodes);
                }
            }
        }
        if clear_old {
            self.selected.clear();
        }
        for node in new_nodes {
            self.selected.add(node);
        }
        Some(())
    }

    fn grab_node(&mut self, node_id: NodeId, special: bool, clear_old: bool, duplicate: bool) {
        self.changes = true;
        let grab_pos = self.relative_mouse_pos();
        let contains_node = self.selected.contains(&node_id);
        if clear_old && !contains_node {
            self.selected.clear();
        }
        self.selected.add(node_id);
        use GrabKind::*;
        let grab_kind = if special {
            Connect
        } else {
            Move(MoveType::Nodes)
        };
        let duplicate = if duplicate && !special {
            Some(if clear_old {
                DuplicationType::New
            } else {
                DuplicationType::Linked
            })
        } else {
            None
        };
        self.grab = Some(Grab {
            pos: grab_pos,
            kind: grab_kind,
            moved: false,
            deselect: (self.selected.is_single() || !clear_old) && contains_node,
            duplicate,
        });
    }

    fn grab_empty(&mut self, special: bool, clear_old: bool, create: bool) {
        if create {
            self.changes = true;
            let net = self.net.safe_mut();
            let new_nodes = if special {
                if self.view_mode == ViewMode::Actions {
                    return;
                }
                let pid = net.add_place();
                self.add_place(pid, &[self.relative_mouse_pos()], None)
            } else {
                if self.view_mode == ViewMode::State {
                    return;
                }
                let tid = net.add_transition();
                self.add_transition(tid, &[self.relative_mouse_pos()], None)
            };
            if clear_old {
                self.selected.clear();
            }
            for node in new_nodes {
                self.selected.add(node);
            }
            self.grab = Some(Grab {
                pos: self.relative_mouse_pos(),
                kind: GrabKind::Move(MoveType::Nodes),
                moved: false,
                deselect: false,
                duplicate: None,
            });
        } else {
            self.grab = Some(Grab {
                pos: self.relative_mouse_pos(),
                kind: if special {
                    GrabKind::Select
                } else {
                    GrabKind::Move(MoveType::View)
                },
                moved: false,
                deselect: false,
                duplicate: None,
            });
        }
    }

    /// Move the text cursor in line.
    pub fn move_text_cursor(&mut self, forward: bool, multiple: bool) {
        if self.text_content_shown && self.selected.tids.len() == 1 {
            let node =
                &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
            if multiple && !forward && node.content_cursor > 0 {
                node.content_cursor -= 1;
            }
            loop {
                if forward {
                    if node.content_cursor < node.content[node.content_line].len() {
                        node.content_cursor += 1;
                        if node.content_cursor == node.content[node.content_line].len() {
                            break;
                        }
                    } else {
                        if node.content_line < node.content.len() - 1 {
                            node.content_line += 1;
                            node.content_cursor = 0;
                        }
                        break;
                    }
                } else {
                    if node.content_cursor > 0 {
                        node.content_cursor -= 1;
                    } else {
                        if node.content_line > 0 {
                            node.content_line -= 1;
                            node.content_cursor = node.content[node.content_line].len();
                        }
                        break;
                    }
                }
                if !multiple
                    || !node.content[node.content_line]
                        .char_at(node.content_cursor)
                        .is_alphanumeric()
                {
                    if multiple && !forward {
                        node.content_cursor += 1;
                    }
                    break;
                }
            }
        } else if let Some(string_searcher) = &mut self.string_searcher {
            if multiple && !forward && string_searcher.name_cursor > 0 {
                string_searcher.name_cursor -= 1;
            }
            loop {
                if forward {
                    if string_searcher.name_cursor < string_searcher.name.len() {
                        string_searcher.name_cursor += 1;
                        if string_searcher.name_cursor == string_searcher.name.len() {
                            break;
                        }
                    } else {
                        break;
                    }
                } else {
                    if string_searcher.name_cursor > 0 {
                        string_searcher.name_cursor -= 1
                    } else {
                        break;
                    }
                }
                if !multiple
                    || !string_searcher
                        .name
                        .char_at(string_searcher.name_cursor)
                        .is_alphanumeric()
                {
                    if multiple && !forward {
                        string_searcher.name_cursor += 1;
                    }
                    break;
                }
            }
        } else {
            if forward {
                for pid in self.selected.pids.keys() {
                    let node = &mut self.place_nodes[*pid as usize];
                    loop {
                        if node.name_cursor < node.name.len() {
                            node.name_cursor += 1;
                            if node.name_cursor == node.name.len() {
                                break;
                            }
                        } else {
                            break;
                        }
                        if !multiple || !node.name.char_at(node.name_cursor).is_alphanumeric() {
                            break;
                        }
                    }
                }
                for tid in self.selected.tids.keys() {
                    let node = &mut self.transition_nodes[*tid as usize];
                    loop {
                        if node.name_cursor < node.name.len() {
                            node.name_cursor += 1;
                            if node.name_cursor == node.name.len() {
                                break;
                            }
                        } else {
                            break;
                        }
                        if !multiple || !node.name.char_at(node.name_cursor).is_alphanumeric() {
                            break;
                        }
                    }
                }
            } else {
                for pid in self.selected.pids.keys() {
                    let node = &mut self.place_nodes[*pid as usize];
                    if multiple && node.name_cursor > 0 {
                        node.name_cursor -= 1;
                    }
                    loop {
                        if node.name_cursor > 0 {
                            node.name_cursor -= 1
                        } else {
                            break;
                        }
                        if !multiple || !node.name.char_at(node.name_cursor).is_alphanumeric() {
                            if multiple {
                                node.name_cursor += 1;
                            }
                            break;
                        }
                    }
                }
                for tid in self.selected.tids.keys() {
                    let node = &mut self.transition_nodes[*tid as usize];
                    if multiple && node.name_cursor > 0 {
                        node.name_cursor -= 1;
                    }
                    loop {
                        if node.name_cursor > 0 {
                            node.name_cursor -= 1
                        } else {
                            break;
                        }
                        if !multiple || !node.name.char_at(node.name_cursor).is_alphanumeric() {
                            if multiple {
                                node.name_cursor += 1;
                            }
                            break;
                        }
                    }
                }
            }
        }
    }

    /// Move the text cursor by a line.
    pub fn move_line(&mut self, down: bool) {
        if self.text_content_shown && self.selected.tids.len() == 1 {
            let node =
                &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
            if down {
                if node.content_line < node.content.len() - 1 {
                    node.content_line += 1;
                    if node.content_cursor > node.content[node.content_line].len() {
                        node.content_cursor = node.content[node.content_line].len();
                    }
                    let line_count =
                        (self.window_size.y / self.node_settings.height * 2.0) as usize - 5;
                    if node.content_line >= node.content_offset + line_count {
                        node.content_offset = node.content_line - line_count;
                    }
                } else {
                    node.content_cursor = node.content[node.content_line].len();
                }
            } else {
                if node.content_line > 0 {
                    node.content_line -= 1;
                    if node.content_cursor > node.content[node.content_line].len() {
                        node.content_cursor = node.content[node.content_line].len();
                    }
                    if node.content_line < node.content_offset {
                        node.content_offset = node.content_line;
                    }
                } else {
                    node.content_cursor = 0;
                }
            }
        }
    }

    /// Move the text by a page.
    pub fn move_page(&mut self, down: bool) {
        if self.text_content_shown && self.selected.tids.len() == 1 {
            let node =
                &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];
            let line_count = (self.window_size.y / self.node_settings.height * 2.0) as usize - 5;
            if down {
                if node.content_offset + line_count * 2 < node.content.len() - 1 {
                    node.content_offset += line_count;
                } else if node.content_offset + line_count < node.content.len() - 1 {
                    node.content_offset = node.content.len() - 1 - line_count;
                }

                if node.content_line + line_count < node.content.len() - 1 {
                    node.content_line += line_count;
                } else {
                    node.content_line = node.content.len() - 1;
                    node.content_cursor = node.content[node.content_line].len();
                }
            } else {
                if node.content_offset > line_count {
                    node.content_offset -= line_count;
                } else {
                    node.content_offset = 0;
                }
                if node.content_line > line_count {
                    node.content_line -= line_count;
                } else {
                    node.content_line = 0;
                    node.content_cursor = 0;
                }
            }
            if node.content[node.content_line].len() < node.content_cursor {
                node.content_cursor = node.content[node.content_line].len();
            }
        }
    }

    /// Move the text to the start or end of line or file.
    pub fn move_text_cursor_border(&mut self, end: bool, all_lines: bool) {
        if self.text_content_shown && self.selected.tids.len() == 1 {
            let node =
                &mut self.transition_nodes[*self.selected.tids.keys().next().unwrap() as usize];

            if end {
                if all_lines {
                    let line_count =
                        (self.window_size.y / self.node_settings.height * 2.0) as usize - 5;
                    if node.content.len() > line_count {
                        node.content_offset = node.content.len() - 1 - line_count;
                    }
                    node.content_line = node.content.len() - 1;
                }
                node.content_cursor = node.content[node.content_line].len();
            } else {
                if all_lines {
                    node.content_offset = 0;
                    node.content_line = 0;
                }
                node.content_cursor = 0;
            }
        } else if let Some(string_searcher) = &mut self.string_searcher {
            if end {
                string_searcher.name_cursor = string_searcher.name.len();
            } else {
                string_searcher.name_cursor = 0;
            }
        } else {
            if end {
                for pid in self.selected.pids.keys() {
                    let node = &mut self.place_nodes[*pid as usize];
                    node.name_cursor = node.name.len();
                }
                for tid in self.selected.tids.keys() {
                    let node = &mut self.transition_nodes[*tid as usize];
                    node.name_cursor = node.name.len();
                }
            } else {
                for pid in self.selected.pids.keys() {
                    self.place_nodes[*pid as usize].name_cursor = 0;
                }
                for tid in self.selected.tids.keys() {
                    self.transition_nodes[*tid as usize].name_cursor = 0;
                }
            }
        }
    }

    fn connect_place(
        &mut self,
        connect_node: NodeId,
        new_nodes: &mut Vec<NodeId>,
        allow_more: bool,
    ) {
        let mut new_transitions = Vec::new();
        for (&pid, subindices) in self.selected.pids.iter() {
            if connect_node.kind != NodeKind::Place {
                let connected = if let Some(net) = self.net.default_mut() {
                    net.connect_in(connect_node.index, pid)
                } else {
                    false
                };
                if !connected && allow_more {
                    self.net.safe_mut().disconnect_in(connect_node.index, pid);
                }
            } else if pid != connect_node.index || allow_more {
                let node = &self.place_nodes[pid as usize];
                let node_pos: Vector2<f32> =
                    subindices.iter().map(|&tid| node.positions[tid]).sum();
                let node_pos = node_pos / subindices.len() as f32;
                let other = &self.place_nodes[connect_node.index as usize];
                let other_pos = other.positions[connect_node.subindex];
                let new_pos = (node_pos + other_pos) / 2.0;
                if let Some(net) = self.net.default_mut() {
                    let new_id = net.add_transition();
                    net.connect_in(new_id, pid);
                    net.connect_out(new_id, connect_node.index);
                    new_transitions.push((new_id, new_pos));
                }
            } else {
                new_nodes.push(connect_node);
            }
        }
        for (new_id, new_pos) in new_transitions {
            let transition = self.add_transition(new_id, &[new_pos], None);
            new_nodes.extend(transition);
        }
    }

    fn connect_transition(
        &mut self,
        connect_node: NodeId,
        new_nodes: &mut Vec<NodeId>,
        allow_more: bool,
    ) {
        let mut new_places = Vec::new();
        for (&tid, subindices) in self.selected.tids.iter() {
            if connect_node.kind != NodeKind::Transition {
                let connected = self.net.safe_mut().connect_out(tid, connect_node.index);
                if !connected && allow_more {
                    if let Some(net) = self.net.default_mut() {
                        net.disconnect_out(tid, connect_node.index);
                    }
                }
            } else if tid != connect_node.index || allow_more {
                let node = &self.transition_nodes[tid as usize];
                let node_pos: Vector2<f32> =
                    subindices.iter().map(|&tid| node.positions[tid]).sum();
                let node_pos = node_pos / subindices.len() as f32;
                let other = &self.transition_nodes[connect_node.index as usize];
                let other_pos = other.positions[connect_node.subindex];
                let new_pos = (node_pos + other_pos) / 2.0;
                if let Some(net) = self.net.default_mut() {
                    let new_id = net.add_place();
                    net.connect_in(connect_node.index, new_id);
                    net.connect_out(tid, new_id);
                    new_places.push((new_id, new_pos));
                }
            } else {
                new_nodes.push(connect_node);
            }
        }
        for (new_id, new_pos) in new_places {
            let place = self.add_place(new_id, &[new_pos], None);
            new_nodes.extend(place);
        }
    }

    fn release_connect(&mut self, allow_more: bool, clear_old: bool, select_new: bool) {
        let mut new_nodes = Vec::new();
        if let Some(connect_node) = self.click_info(self.relative_mouse_pos()) {
            self.connect_place(connect_node, &mut new_nodes, allow_more);
            self.connect_transition(connect_node, &mut new_nodes, allow_more);
        } else if allow_more {
            match self.view_mode {
                ViewMode::Default => {
                    if self.selected.pids.len() > 0 {
                        let pids: Vec<_> = self.selected.pids.keys().cloned().collect();
                        let connect_id = self.net.safe_mut().add_connected_transition(&pids[..]);
                        let transition = self.add_transition(
                            connect_id,
                            &[self.snapped_relative_mouse_pos()],
                            None,
                        );
                        new_nodes.extend(transition);
                    }
                    if self.selected.tids.len() > 0 {
                        let connect_id = self.net.safe_mut().add_place();
                        let place =
                            self.add_place(connect_id, &[self.snapped_relative_mouse_pos()], None);
                        new_nodes.extend(place);
                        for tid in self.selected.tids.keys() {
                            self.net.safe_mut().connect_out(*tid, connect_id);
                        }
                    }
                }
                ViewMode::State => {
                    let connect_id = self.net.safe_mut().add_place();
                    let new_pos = self.snapped_relative_mouse_pos();
                    let place = self.add_place(connect_id, &[new_pos], None);
                    new_nodes.extend(place);
                    let selected_pids: Vec<_> = self.selected.pids.keys().cloned().collect();
                    for pid in selected_pids {
                        let node = &self.place_nodes[pid as usize];
                        if let Some((pos, _)) = node.positions.iter().fold(None, |result, pos| {
                            let new_distance = (pos - self.relative_mouse_pos()).norm_squared();
                            if let Some((position, distance)) = result {
                                if new_distance >= distance {
                                    return Some((position, distance));
                                }
                            }
                            Some((*pos, new_distance))
                        }) {
                            let new_id = self.net.safe_mut().add_connected_transition(&[pid]);
                            let transition =
                                self.add_transition(new_id, &[(pos + new_pos) / 2.0], None);
                            self.net.safe_mut().connect_out(new_id, connect_id);
                            new_nodes.extend(transition);
                        }
                    }
                }
                ViewMode::Actions => {
                    let mut new_ids = Vec::new();
                    let selected_tids: Vec<_> = self.selected.tids.keys().cloned().collect();
                    let new_pos = self.snapped_relative_mouse_pos();
                    for tid in selected_tids {
                        let node = &self.transition_nodes[tid as usize];
                        if let Some((pos, _)) = node.positions.iter().fold(None, |result, pos| {
                            let new_distance = (pos - self.relative_mouse_pos()).norm_squared();
                            if let Some((position, distance)) = result {
                                if new_distance >= distance {
                                    return Some((position, distance));
                                }
                            }
                            Some((*pos, new_distance))
                        }) {
                            let new_id = self.net.safe_mut().add_place();
                            self.net.safe_mut().connect_out(tid, new_id);
                            let place = self.add_place(new_id, &[(pos + new_pos) / 2.0], None);
                            new_nodes.extend(place);
                            new_ids.push(new_id);
                        }
                    }
                    let connect_id = self.net.safe_mut().add_connected_transition(&new_ids[..]);
                    let transition = self.add_transition(connect_id, &[new_pos], None);
                    new_nodes.extend(transition);
                }
            }
        }
        if clear_old {
            self.selected.clear()
        }
        if select_new {
            for node in new_nodes {
                self.selected.add(node);
            }
        }
    }
}
