use copypasta::ClipboardContext;
use nalgebra::Vector2;
use pns::{DynamicNet, Net, SafeNet};

use crate::edit_list::SelectionList;
use crate::feature_types::{Grab, StringSearcher, ViewMode};
use crate::node::Node;
use crate::node_settings::NodeSettings;
use crate::state_info::StateInfo;

const B: u32 = 0x40;

/// This represents the editor you want to use to edit and display petri nets.
pub struct Editor {
    net: DynamicNet,
    place_nodes: Vec<Node>,
    transition_nodes: Vec<Node>,
    text_content_shown: bool,
    width: u32,
    height: u32,
    window_size: Vector2<f32>,
    zoom: f32,
    selected: SelectionList,
    grab: Option<Grab>,
    view_mode: ViewMode,
    net_path: Option<String>,
    state_infos: Vec<StateInfo>,
    simulation: usize,
    mouse_pos: Vector2<f32>,
    view_offset: Vector2<f32>,
    clipboard: ClipboardContext,
    string_searcher: Option<StringSearcher>,
    changes: bool,
    snapping: Option<f32>,
    node_settings: NodeSettings,
}

impl Editor {
    /// Creates a new editor, which contains a renderer and knows about the window size.
    pub fn new(width: u32, height: u32, node_settings: NodeSettings) -> Editor {
        let net = Net::new();

        Editor {
            net: net.dynamic(),
            place_nodes: Vec::new(),
            transition_nodes: Vec::new(),
            text_content_shown: false,
            width,
            height,
            window_size: Vector2::new(width as f32, height as f32),
            zoom: 1.0,
            selected: SelectionList::new(),
            grab: None,
            view_mode: ViewMode::Default,
            net_path: None,
            state_infos: Vec::new(),
            simulation: 0,
            mouse_pos: Vector2::new(0.0, 0.0),
            view_offset: Vector2::new(0.0, 0.0),
            string_searcher: None,
            clipboard: ClipboardContext::new().unwrap(),
            changes: false,
            snapping: Some(16.0),
            node_settings,
        }
    }

    fn reset(&mut self, mut net: Net, names: Vec<String>, place_names: Vec<String>) {
        let mut names = names.into_iter();
        let mut place_names = place_names.into_iter();

        for _ in net.places().len()..place_names.len() {
            net.add_place();
        }
        let mut place_nodes = Vec::with_capacity(net.places().len());

        for _ in net.transitions().len()..names.len() {
            net.add_transition();
        }
        let mut transition_nodes = Vec::with_capacity(net.transitions().len());

        let r = self.width - B;

        for i in 0..(net.places().len() as u32) {
            let name = place_names.next().unwrap_or(String::new());
            let node = Node::new(
                &[Vector2::new(
                    (B + i * 0x100 % r) as f32,
                    (B + i * 0x100 / r * 0x40) as f32,
                )],
                name,
            );
            place_nodes.push(node);
        }

        for i in 0..(net.transitions().len() as u32) {
            let name = names.next().unwrap_or(if i == 0 {
                "Intro".to_string()
            } else {
                format!("Act {}", i)
            });
            let node = Node::new(
                &[Vector2::new(
                    (B + (0x80 + i * 0x100) % r) as f32,
                    (B + (0x80 + i * 0x100) / r * 0x40) as f32,
                )],
                name,
            );
            transition_nodes.push(node);
        }

        self.net = net.dynamic();

        self.state_infos = Vec::new();

        self.place_nodes = place_nodes;
        self.transition_nodes = transition_nodes;
        self.selected = SelectionList::new();
        self.view_offset = Vector2::new(0.0, 0.0);

        self.simulation = 0;
        self.changes = false;
    }

    // helpers

    fn relative_mouse_pos(&self) -> Vector2<f32> {
        (self.mouse_pos - self.window_size / 2.0) * self.zoom + self.view_offset
    }

    /// Recieve the horizontal size of the editor.
    pub fn width(&self) -> u32 {
        self.width
    }

    /// Recieve the vertical size of the editor.
    pub fn height(&self) -> u32 {
        self.height
    }
}

mod input;
mod render;
mod update;
