use std::collections::{BTreeSet as Set, HashMap as Map};

#[derive(PartialEq, Copy, Clone)]
pub enum NodeKind {
    Transition,
    Place,
}

#[derive(Copy, Clone, PartialEq)]
pub struct NodeId {
    pub kind: NodeKind,
    pub index: u32,
    pub subindex: usize,
}

pub struct EditList {
    pub tids: Set<u32>,
    pub pids: Set<u32>,
}

impl EditList {
    pub fn new() -> EditList {
        EditList {
            pids: Set::new(),
            tids: Set::new(),
        }
    }
}

pub struct SelectionMap(Map<u32, Set<usize>>);

impl SelectionMap {
    fn new() -> SelectionMap {
        SelectionMap(Map::new())
    }

    pub fn insert(&mut self, index: u32, subindex: usize) {
        if let Some(set) = self.0.get_mut(&index) {
            set.insert(subindex);
        } else {
            let mut set = Set::new();
            set.insert(subindex);
            self.0.insert(index, set);
        }
    }

    pub fn contains(&self, index: &u32, subindex: &usize) -> bool {
        if let Some(set) = self.0.get(index) {
            set.contains(subindex)
        } else {
            false
        }
    }

    pub fn contains_key(&self, index: &u32) -> bool {
        self.0.contains_key(index)
    }

    pub fn iter(&self) -> impl Iterator<Item = (&u32, &Set<usize>)> {
        self.0.iter()
    }

    pub fn keys(&self) -> impl Iterator<Item = &u32> {
        self.0.keys()
    }

    fn remove(&mut self, index: &u32, subindex: &usize) {
        if let Some(set) = self.0.get_mut(index) {
            set.remove(subindex);
            if set.is_empty() {
                self.0.remove(index);
            }
        }
    }

    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn is_single(&self) -> bool {
        if self.len() != 1 {
            return false;
        }
        self.iter().next().unwrap().1.len() == 1
    }

    fn clear(&mut self) {
        self.0.clear()
    }
}

pub struct SelectionList {
    pub tids: SelectionMap,
    pub pids: SelectionMap,
}

impl SelectionList {
    pub fn new() -> SelectionList {
        SelectionList {
            pids: SelectionMap::new(),
            tids: SelectionMap::new(),
        }
    }

    pub fn is_empty(&self) -> bool {
        self.pids.is_empty() && self.tids.is_empty()
    }

    pub fn is_single(&self) -> bool {
        (self.pids.is_empty() && self.tids.is_single())
            || (self.pids.is_single() && self.tids.is_empty())
    }

    pub fn clear(&mut self) {
        self.pids.clear();
        self.tids.clear();
    }

    pub fn add(&mut self, node: NodeId) {
        use NodeKind::*;
        match node.kind {
            Place => self.pids.insert(node.index, node.subindex),
            Transition => self.tids.insert(node.index, node.subindex),
        }
    }

    pub fn contains(&mut self, node: &NodeId) -> bool {
        use NodeKind::*;
        match node.kind {
            Place => self.pids.contains(&node.index, &node.subindex),
            Transition => self.tids.contains(&node.index, &node.subindex),
        }
    }

    pub fn remove_node(&mut self, node: NodeId) {
        use NodeKind::*;
        match node.kind {
            Place => self.pids.remove(&node.index, &node.subindex),
            Transition => self.tids.remove(&node.index, &node.subindex),
        }
    }
}
