// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright (c) DUSK NETWORK. All rights reserved.

// For the Bls and BN254 curves, the large prime p is different.
// This leads to diffences in the subsequent difference in constants
// we have defined below.
// These are the required constants for the
// Currently making the s_i usize.

use dusk_bls12_381::BlsScalar;

/// Constant V
pub const V: usize = 643;
/// Constant N
pub const N: u64 = 27;
/// Note this is currently backwards, e.g. S\[0\] should = 673. But doesn't
/// matter for now
pub const S: [u64; 27] = [
    651, 658, 656, 666, 663, 654, 668, 677, 681, 683, 669, 681, 680, 677, 675,
    668, 675, 683, 681, 683, 683, 655, 680, 683, 667, 678, 673,
];
/// Constant T_s
pub const T_S: u32 = 4;

/// Montgomery form of the s_i, needed to input them as variables
pub const S_I_DECOMPOSITION_MONTGOMERY: [BlsScalar; 27] = [
    BlsScalar([
        6571299961350,
        9554885865219436026,
        15585515709981716521,
        2712848100111761021,
    ]),
    BlsScalar([
        6597069765120,
        10243417561410045440,
        11884946391852638217,
        7931979164453719756,
    ]),
    BlsScalar([
        6579889895940,
        15933311121852823036,
        8203077912702173214,
        4452558454892413933,
    ]),
    BlsScalar([
        6335076760125,
        9392260008042033603,
        15688370324364089678,
        987673527917684992,
    ]),
    BlsScalar([
        6438155975205,
        12146386792804471259,
        886093051847776462,
        3417453711575968316,
    ]),
    BlsScalar([
        6588479830530,
        3864992304776658430,
        820640115422629908,
        6192268809673066845,
    ]),
    BlsScalar([
        6549825124875,
        2832194760490744309,
        15594866129471023172,
        7586944250014904550,
    ]),
    BlsScalar([
        6571299961350,
        9554885865219436026,
        15585515709981716521,
        2712848100111761021,
    ]),
    BlsScalar([
        6635724470775,
        11276215105695959561,
        15557464451513796569,
        6537303724111882050,
    ]),
    BlsScalar([
        6524055321105,
        2143663064300134895,
        848691373890549860,
        2367813185672945816,
    ]),
    BlsScalar([
        6635724470775,
        11276215105695959561,
        15557464451513796569,
        6537303724111882050,
    ]),
    BlsScalar([
        6579889895940,
        15933311121852823036,
        8203077912702173214,
        4452558454892413933,
    ]),
    BlsScalar([
        6644314405365,
        17654640362329346571,
        8175026654234253262,
        8277014078892534962,
    ]),
    BlsScalar([
        6579889895940,
        15933311121852823036,
        8203077912702173214,
        4452558454892413933,
    ]),
    BlsScalar([
        6627134536185,
        4897789849062572551,
        4493158175083788260,
        4797593369331229139,
    ]),
    BlsScalar([
        6644314405365,
        17654640362329346571,
        8175026654234253262,
        8277014078892534962,
    ]),
    BlsScalar([
        6644314405365,
        17654640362329346571,
        8175026654234253262,
        8277014078892534962,
    ]),
    BlsScalar([
        6644314405365,
        17654640362329346571,
        8175026654234253262,
        8277014078892534962,
    ]),
    BlsScalar([
        6588479830530,
        3864992304776658430,
        820640115422629908,
        6192268809673066845,
    ]),
    BlsScalar([
        6618544601595,
        16966108666138737157,
        11875595972363331566,
        3057883014550576227,
    ]),
    BlsScalar([
        6609954667005,
        10587683409505350147,
        811289695933323257,
        1318172659769923316,
    ]),
    BlsScalar([
        6665789241840,
        5930587393348486672,
        8165676234744946612,
        3402917928989391433,
    ]),
    BlsScalar([
        6657199307250,
        17998906210424651278,
        15548114032024489918,
        1663207574208738521,
    ]),
    BlsScalar([
        6549825124875,
        2832194760490744309,
        15594866129471023172,
        7586944250014904550,
    ]),
    BlsScalar([
        6524055321105,
        2143663064300134895,
        848691373890549860,
        2367813185672945816,
    ]),
    BlsScalar([
        6665789241840,
        5930587393348486672,
        8165676234744946612,
        3402917928989391433,
    ]),
    BlsScalar([
        6438155975205,
        12146386792804471259,
        886093051847776462,
        3417453711575968316,
    ]),
];

/// Divisors used in modular reduciton via precomputation
pub const BLS_DIVISORS: [u64; 26] = [
    12483978167071014912,
    12538021362599460864,
    12501992565580496896,
    12033618204333965312,
    12231776587938267136,
    12520006964089978880,
    12447949370052050944,
    12483978167071014912,
    12610078956637388800,
    12393906174523604992,
    12610078956637388800,
    12501992565580496896,
    12628093355146870784,
    12501992565580496896,
    12592064558127906816,
    12628093355146870784,
    12628093355146870784,
    12628093355146870784,
    12520006964089978880,
    12574050159618424832,
    12556035761108942848,
    12664122152165834752,
    12646107753656352768,
    12447949370052050944,
    12393906174523604992,
    12664122152165834752,
];

/// Reciprocals used in modular reduciton via precomputation
pub const BLS_RECIP: [u64; 26] = [
    8810782522940637207,
    8693293184161972600,
    8771506548017510134,
    9830899536288323915,
    9372793380603527698,
    8732343597482651052,
    8889675508748597233,
    8810782522940637207,
    8538207256974135319,
    9008875012741874045,
    8538207256974135319,
    8771506548017510134,
    8499712319269878989,
    8771506548017510134,
    8576812337561665629,
    8499712319269878989,
    8499712319269878989,
    8499712319269878989,
    8732343597482651052,
    8615528034425951041,
    8654354823677221489,
    8423050992405072644,
    8461327053752814274,
    8889675508748597233,
    9008875012741874045,
    8423050992405072644,
];

/// Predefined S-box
pub const SBOX: [u16; 659] = [
    15, 187, 186, 168, 493, 102, 296, 11, 164, 155, 527, 103, 192, 589, 543,
    450, 647, 72, 343, 386, 279, 616, 225, 140, 313, 586, 276, 57, 162, 68,
    179, 445, 418, 364, 46, 591, 541, 218, 0, 437, 618, 157, 657, 49, 120, 469,
    142, 325, 183, 123, 23, 468, 619, 217, 472, 226, 212, 406, 4, 499, 182, 51,
    141, 86, 596, 70, 149, 355, 351, 245, 52, 193, 311, 244, 375, 300, 399,
    590, 143, 24, 190, 517, 208, 539, 544, 236, 393, 34, 203, 60, 151, 243,
    542, 299, 368, 289, 272, 567, 280, 599, 625, 341, 462, 509, 153, 374, 213,
    477, 310, 347, 572, 71, 579, 158, 412, 587, 63, 172, 640, 173, 101, 439, 5,
    92, 501, 500, 330, 633, 630, 328, 488, 356, 89, 224, 383, 96, 643, 585,
    422, 41, 295, 642, 571, 247, 239, 600, 561, 319, 480, 570, 652, 134, 620,
    484, 525, 333, 177, 209, 8, 211, 402, 478, 574, 148, 365, 83, 635, 44, 646,
    204, 414, 413, 398, 449, 363, 588, 65, 617, 658, 126, 178, 536, 624, 201,
    513, 506, 384, 336, 382, 348, 223, 316, 629, 88, 18, 278, 287, 524, 257,
    421, 639, 424, 452, 511, 564, 538, 214, 514, 307, 31, 93, 471, 104, 528,
    234, 352, 255, 534, 580, 113, 360, 526, 614, 532, 603, 537, 1, 370, 121,
    430, 32, 417, 426, 391, 644, 358, 206, 3, 504, 13, 557, 444, 284, 584, 39,
    251, 176, 508, 94, 156, 33, 273, 496, 246, 321, 58, 21, 165, 638, 436, 10,
    145, 194, 498, 267, 292, 90, 497, 505, 510, 80, 435, 303, 42, 533, 529,
    453, 329, 428, 35, 337, 269, 229, 297, 85, 562, 440, 357, 95, 50, 559, 446,
    656, 606, 457, 459, 390, 59, 611, 306, 623, 188, 650, 582, 170, 249, 16,
    380, 230, 130, 169, 138, 612, 207, 227, 598, 47, 483, 73, 67, 106, 175,
    655, 22, 77, 133, 283, 377, 112, 232, 429, 117, 111, 332, 6, 324, 7, 409,
    302, 260, 216, 320, 166, 475, 465, 45, 366, 519, 335, 200, 215, 205, 262,
    419, 147, 237, 282, 359, 174, 379, 441, 551, 473, 605, 427, 474, 387, 84,
    171, 222, 37, 565, 48, 549, 161, 521, 566, 518, 568, 403, 597, 397, 154,
    649, 53, 522, 416, 240, 372, 645, 261, 314, 309, 395, 373, 20, 119, 27,
    608, 340, 609, 361, 503, 241, 602, 30, 275, 569, 423, 454, 150, 621, 415,
    344, 535, 411, 540, 199, 442, 371, 404, 210, 322, 432, 492, 560, 250, 132,
    627, 233, 202, 304, 641, 338, 74, 575, 408, 425, 291, 135, 318, 601, 159,
    489, 556, 385, 548, 554, 81, 362, 108, 270, 405, 136, 576, 55, 389, 354,
    604, 388, 97, 198, 317, 334, 458, 491, 259, 583, 369, 129, 546, 87, 327,
    266, 401, 550, 69, 274, 615, 400, 181, 353, 196, 456, 595, 420, 122, 392,
    185, 516, 466, 476, 75, 235, 530, 448, 594, 378, 455, 447, 577, 285, 99,
    558, 653, 410, 461, 160, 331, 290, 563, 613, 219, 394, 29, 552, 9, 189,
    298, 137, 56, 636, 12, 581, 2, 109, 339, 127, 36, 443, 573, 523, 451, 479,
    286, 28, 116, 312, 628, 545, 54, 82, 651, 482, 131, 26, 396, 271, 593, 124,
    107, 515, 114, 407, 654, 268, 342, 277, 254, 14, 79, 191, 43, 252, 512,
    256, 220, 381, 66, 481, 19, 228, 367, 487, 434, 349, 144, 460, 91, 495, 78,
    195, 490, 67, 486, 64, 105, 467, 231, 507, 376, 248, 631, 520, 464, 221,
    433, 622, 531, 197, 61, 163, 98, 648, 146, 238, 494, 125, 76, 242, 463,
    326, 38, 152, 438, 345, 637, 40, 17, 281, 323, 110, 118, 578, 139, 315,
    115, 62, 470, 293, 265, 258, 553, 301, 610, 555, 305, 634, 308, 626, 180,
    253, 350, 502, 184, 431, 294, 264, 288, 632, 25, 607, 485, 592, 263, 128,
    547, 100, 346,
];

/// Decomposition of -1 = [v_n, v_{n-1} ..., v_1], i.e. the representation of
/// q-1
pub const BLS_SCALAR_REAL: [u16; 27] = [
    660, 660, 673, 663, 674, 682, 687, 683, 669, 684, 672, 666, 680, 662, 686,
    668, 661, 678, 692, 686, 689, 660, 690, 687, 683, 674, 678,
];

/// decomposition = [s_n, s_{n-1} ..., s_1]
pub const DECOMPOSITION_S_I: [BlsScalar; 27] = [
    BlsScalar([693, 0, 0, 0]),
    BlsScalar([696, 0, 0, 0]),
    BlsScalar([694, 0, 0, 0]),
    BlsScalar([668, 0, 0, 0]),
    BlsScalar([679, 0, 0, 0]),
    BlsScalar([695, 0, 0, 0]),
    BlsScalar([691, 0, 0, 0]),
    BlsScalar([693, 0, 0, 0]),
    BlsScalar([700, 0, 0, 0]),
    BlsScalar([688, 0, 0, 0]),
    BlsScalar([700, 0, 0, 0]),
    BlsScalar([694, 0, 0, 0]),
    BlsScalar([701, 0, 0, 0]),
    BlsScalar([694, 0, 0, 0]),
    BlsScalar([699, 0, 0, 0]),
    BlsScalar([701, 0, 0, 0]),
    BlsScalar([701, 0, 0, 0]),
    BlsScalar([701, 0, 0, 0]),
    BlsScalar([695, 0, 0, 0]),
    BlsScalar([698, 0, 0, 0]),
    BlsScalar([697, 0, 0, 0]),
    BlsScalar([703, 0, 0, 0]),
    BlsScalar([702, 0, 0, 0]),
    BlsScalar([691, 0, 0, 0]),
    BlsScalar([688, 0, 0, 0]),
    BlsScalar([703, 0, 0, 0]),
    BlsScalar([679, 0, 0, 0]),
];

/// Remainers in Montgomery form
pub const REMAINDER_MONT: [BlsScalar; 703] = [
    BlsScalar([0, 0, 0, 0]),
    BlsScalar([
        8589934590,
        6378425256633387010,
        11064306276430008309,
        1739710354780652911,
    ]),
    BlsScalar([
        17179869180,
        12756850513266774020,
        3681868479150465002,
        3479420709561305823,
    ]),
    BlsScalar([
        25769803770,
        688531696190609414,
        14746174755580473312,
        5219131064341958734,
    ]),
    BlsScalar([
        34359738360,
        7066956952823996424,
        7363736958300930005,
        6958841419122611646,
    ]),
    BlsScalar([
        47244640245,
        7411222800919301131,
        14736824336091166661,
        345034914438815205,
    ]),
    BlsScalar([
        55834574835,
        13789648057552688141,
        7354386538811623354,
        2084745269219468117,
    ]),
    BlsScalar([
        64424509425,
        1721329240476523535,
        18418692815241631664,
        3824455624000121028,
    ]),
    BlsScalar([
        73014444015,
        8099754497109910545,
        11036255017962088357,
        5564165978780773940,
    ]),
    BlsScalar([
        81604378605,
        14478179753743297555,
        3653817220682545050,
        7303876333561426852,
    ]),
    BlsScalar([
        94489280490,
        14822445601838602262,
        11026904598472781706,
        690069828877630411,
    ]),
    BlsScalar([
        103079215080,
        2754126784762437656,
        3644466801193238400,
        2429780183658283323,
    ]),
    BlsScalar([
        111669149670,
        9132552041395824666,
        14708773077623246709,
        4169490538438936234,
    ]),
    BlsScalar([
        120259084260,
        15510977298029211676,
        7326335280343703402,
        5909200893219589146,
    ]),
    BlsScalar([
        128849018850,
        3442658480953047070,
        18390641556773711712,
        7648911248000242057,
    ]),
    BlsScalar([
        141733920735,
        3786924329048351777,
        7316984860854396752,
        1035104743316445617,
    ]),
    BlsScalar([
        150323855325,
        10165349585681738787,
        18381291137284405061,
        2774815098097098528,
    ]),
    BlsScalar([
        158913789915,
        16543774842315125797,
        10998853340004861754,
        4514525452877751440,
    ]),
    BlsScalar([
        167503724505,
        4475456025238961191,
        3616415542725318448,
        6254235807658404352,
    ]),
    BlsScalar([
        176093659095,
        10853881281872348201,
        14680721819155326757,
        7993946162439057263,
    ]),
    BlsScalar([
        188978560980,
        11198147129967652908,
        3607065123236011797,
        1380139657755260823,
    ]),
    BlsScalar([
        197568495570,
        17576572386601039918,
        14671371399666020106,
        3119850012535913734,
    ]),
    BlsScalar([
        206158430160,
        5508253569524875312,
        7288933602386476800,
        4859560367316566646,
    ]),
    BlsScalar([
        214748364750,
        11886678826158262322,
        18353239878816485109,
        6599270722097219557,
    ]),
    BlsScalar([
        223338299340,
        18265104082791649332,
        10970802081536941802,
        8338981076877872469,
    ]),
    BlsScalar([
        236223201225,
        162625857177402423,
        18343889459327178459,
        1725174572194076028,
    ]),
    BlsScalar([
        244813135815,
        6541051113810789433,
        10961451662047635152,
        3464884926974728940,
    ]),
    BlsScalar([
        253403070405,
        12919476370444176443,
        3579013864768091845,
        5204595281755381852,
    ]),
    BlsScalar([
        261993004995,
        851157553368011837,
        14643320141198100155,
        6944305636536034763,
    ]),
    BlsScalar([
        274877906880,
        1195423401463316544,
        3569663445278785195,
        330499131852238323,
    ]),
    BlsScalar([
        283467841470,
        7573848658096703554,
        14633969721708793504,
        2070209486632891234,
    ]),
    BlsScalar([
        292057776060,
        13952273914730090564,
        7251531924429250197,
        3809919841413544146,
    ]),
    BlsScalar([
        300647710650,
        1883955097653925958,
        18315838200859258507,
        5549630196194197057,
    ]),
    BlsScalar([
        309237645240,
        8262380354287312968,
        10933400403579715200,
        7289340550974849969,
    ]),
    BlsScalar([
        322122547125,
        8606646202382617675,
        18306487781369951856,
        675534046291053528,
    ]),
    BlsScalar([
        330712481715,
        14985071459016004685,
        10924049984090408549,
        2415244401071706440,
    ]),
    BlsScalar([
        339302416305,
        2916752641939840079,
        3541612186810865243,
        4154954755852359352,
    ]),
    BlsScalar([
        347892350895,
        9295177898573227089,
        14605918463240873552,
        5894665110633012263,
    ]),
    BlsScalar([
        356482285485,
        15673603155206614099,
        7223480665961330245,
        7634375465413665175,
    ]),
    BlsScalar([
        369367187370,
        16017869003301918806,
        14596568043751566901,
        1020568960729868734,
    ]),
    BlsScalar([
        377957121960,
        3949550186225754200,
        7214130246472023595,
        2760279315510521646,
    ]),
    BlsScalar([
        386547056550,
        10327975442859141210,
        18278436522902031904,
        4499989670291174557,
    ]),
    BlsScalar([
        395136991140,
        16706400699492528220,
        10895998725622488597,
        6239700025071827469,
    ]),
    BlsScalar([
        403726925730,
        4638081882416363614,
        3513560928342945291,
        7979410379852480381,
    ]),
    BlsScalar([
        416611827615,
        4982347730511668321,
        10886648306133181947,
        1365603875168683940,
    ]),
    BlsScalar([
        425201762205,
        11360772987145055331,
        3504210508853638640,
        3105314229949336852,
    ]),
    BlsScalar([
        433791696795,
        17739198243778442341,
        14568516785283646949,
        4845024584729989763,
    ]),
    BlsScalar([
        442381631385,
        5670879426702277735,
        7186078988004103643,
        6584734939510642675,
    ]),
    BlsScalar([
        450971565975,
        12049304683335664745,
        18250385264434111952,
        8324445294291295586,
    ]),
    BlsScalar([
        463856467860,
        12393570531430969452,
        7176728568514796992,
        1710638789607499146,
    ]),
    BlsScalar([
        472446402450,
        325251714354804846,
        18241034844944805302,
        3450349144388152057,
    ]),
    BlsScalar([
        481036337040,
        6703676970988191856,
        10858597047665261995,
        5190059499168804969,
    ]),
    BlsScalar([
        489626271630,
        13082102227621578866,
        3476159250385718688,
        6929769853949457881,
    ]),
    BlsScalar([
        502511173515,
        13426368075716883573,
        10849246628175955344,
        315963349265661440,
    ]),
    BlsScalar([
        511101108105,
        1358049258640718967,
        3466808830896412038,
        2055673704046314352,
    ]),
    BlsScalar([
        519691042695,
        7736474515274105977,
        14531115107326420347,
        3795384058826967263,
    ]),
    BlsScalar([
        528280977285,
        14114899771907492987,
        7148677310046877040,
        5535094413607620175,
    ]),
    BlsScalar([
        536870911875,
        2046580954831328381,
        18212983586476885350,
        7274804768388273086,
    ]),
    BlsScalar([
        549755813760,
        2390846802926633088,
        7139326890557570390,
        660998263704476646,
    ]),
    BlsScalar([
        558345748350,
        8769272059560020098,
        18203633166987578699,
        2400708618485129557,
    ]),
    BlsScalar([
        566935682940,
        15147697316193407108,
        10821195369708035392,
        4140418973265782469,
    ]),
    BlsScalar([
        575525617530,
        3079378499117242502,
        3438757572428492086,
        5880129328046435381,
    ]),
    BlsScalar([
        584115552120,
        9457803755750629512,
        14503063848858500395,
        7619839682827088292,
    ]),
    BlsScalar([
        597000454005,
        9802069603845934219,
        3429407152939185435,
        1006033178143291852,
    ]),
    BlsScalar([
        605590388595,
        16180494860479321229,
        14493713429369193744,
        2745743532923944763,
    ]),
    BlsScalar([
        614180323185,
        4112176043403156623,
        7111275632089650438,
        4485453887704597675,
    ]),
    BlsScalar([
        622770257775,
        10490601300036543633,
        18175581908519658747,
        6225164242485250586,
    ]),
    BlsScalar([
        631360192365,
        16869026556669930643,
        10793144111240115440,
        7964874597265903498,
    ]),
    BlsScalar([
        644245094250,
        17213292404765235350,
        18166231489030352096,
        1351068092582107057,
    ]),
    BlsScalar([
        652835028840,
        5144973587689070744,
        10783793691750808790,
        3090778447362759969,
    ]),
    BlsScalar([
        661424963430,
        11523398844322457754,
        3401355894471265483,
        4830488802143412881,
    ]),
    BlsScalar([
        670014898020,
        17901824100955844764,
        14465662170901273792,
        6570199156924065792,
    ]),
    BlsScalar([
        678604832610,
        5833505283879680158,
        7083224373621730486,
        8309909511704718704,
    ]),
    BlsScalar([
        691489734495,
        6177771131974984865,
        14456311751411967142,
        1696103007020922263,
    ]),
    BlsScalar([
        700079669085,
        12556196388608371875,
        7073873954132423835,
        3435813361801575175,
    ]),
    BlsScalar([
        708669603675,
        487877571532207269,
        18138180230562432145,
        5175523716582228086,
    ]),
    BlsScalar([
        717259538265,
        6866302828165594279,
        10755742433282888838,
        6915234071362880998,
    ]),
    BlsScalar([
        730144440150,
        7210568676260898986,
        18128829811073125494,
        301427566679084557,
    ]),
    BlsScalar([
        738734374740,
        13588993932894285996,
        10746392013793582187,
        2041137921459737469,
    ]),
    BlsScalar([
        747324309330,
        1520675115818121390,
        3363954216514038881,
        3780848276240390381,
    ]),
    BlsScalar([
        755914243920,
        7899100372451508400,
        14428260492944047190,
        5520558631021043292,
    ]),
    BlsScalar([
        764504178510,
        14277525629084895410,
        7045822695664503883,
        7260268985801696204,
    ]),
    BlsScalar([
        777389080395,
        14621791477180200117,
        14418910073454740539,
        646462481117899763,
    ]),
    BlsScalar([
        785979014985,
        2553472660104035511,
        7036472276175197233,
        2386172835898552675,
    ]),
    BlsScalar([
        794568949575,
        8931897916737422521,
        18100778552605205542,
        4125883190679205586,
    ]),
    BlsScalar([
        803158884165,
        15310323173370809531,
        10718340755325662235,
        5865593545459858498,
    ]),
    BlsScalar([
        811748818755,
        3242004356294644925,
        3335902958046118929,
        7605303900240511410,
    ]),
    BlsScalar([
        824633720640,
        3586270204389949632,
        10708990335836355585,
        991497395556714969,
    ]),
    BlsScalar([
        833223655230,
        9964695461023336642,
        3326552538556812278,
        2731207750337367881,
    ]),
    BlsScalar([
        841813589820,
        16343120717656723652,
        14390858814986820587,
        4470918105118020792,
    ]),
    BlsScalar([
        850403524410,
        4274801900580559046,
        7008421017707277281,
        6210628459898673704,
    ]),
    BlsScalar([
        858993459000,
        10653227157213946056,
        18072727294137285590,
        7950338814679326615,
    ]),
    BlsScalar([
        871878360885,
        10997493005309250763,
        6999070598217970630,
        1336532309995530175,
    ]),
    BlsScalar([
        880468295475,
        17375918261942637773,
        18063376874647978939,
        3076242664776183086,
    ]),
    BlsScalar([
        889058230065,
        5307599444866473167,
        10680939077368435633,
        4815953019556835998,
    ]),
    BlsScalar([
        897648164655,
        11686024701499860177,
        3298501280088892326,
        6555663374337488910,
    ]),
    BlsScalar([
        906238099245,
        18064449958133247187,
        14362807556518900635,
        8295373729118141821,
    ]),
    BlsScalar([
        919123001130,
        18408715806228551894,
        3289150860599585675,
        1681567224434345381,
    ]),
    BlsScalar([
        927712935720,
        6340396989152387288,
        14353457137029593985,
        3421277579214998292,
    ]),
    BlsScalar([
        936302870310,
        12718822245785774298,
        6971019339750050678,
        5160987933995651204,
    ]),
    BlsScalar([
        944892804900,
        650503428709609692,
        18035325616180058988,
        6900698288776304115,
    ]),
    BlsScalar([
        957777706785,
        994769276804914399,
        6961668920260744028,
        286891784092507675,
    ]),
    BlsScalar([
        966367641375,
        7373194533438301409,
        18025975196690752337,
        2026602138873160586,
    ]),
    BlsScalar([
        974957575965,
        13751619790071688419,
        10643537399411209030,
        3766312493653813498,
    ]),
    BlsScalar([
        983547510555,
        1683300972995523813,
        3261099602131665724,
        5506022848434466410,
    ]),
    BlsScalar([
        992137445145,
        8061726229628910823,
        14325405878561674033,
        7245733203215119321,
    ]),
    BlsScalar([
        1005022347030,
        8405992077724215530,
        3251749182642359073,
        631926698531322881,
    ]),
    BlsScalar([
        1013612281620,
        14784417334357602540,
        14316055459072367382,
        2371637053311975792,
    ]),
    BlsScalar([
        1022202216210,
        2716098517281437934,
        6933617661792824076,
        4111347408092628704,
    ]),
    BlsScalar([
        1030792150800,
        9094523773914824944,
        17997923938222832385,
        5851057762873281615,
    ]),
    BlsScalar([
        1039382085390,
        15472949030548211954,
        10615486140943289078,
        7590768117653934527,
    ]),
    BlsScalar([
        1052266987275,
        15817214878643516661,
        17988573518733525734,
        976961612970138086,
    ]),
    BlsScalar([
        1060856921865,
        3748896061567352055,
        10606135721453982428,
        2716671967750790998,
    ]),
    BlsScalar([
        1069446856455,
        10127321318200739065,
        3223697924174439121,
        4456382322531443910,
    ]),
    BlsScalar([
        1078036791045,
        16505746574834126075,
        14288004200604447430,
        6196092677312096821,
    ]),
    BlsScalar([
        1086626725635,
        4437427757757961469,
        6905566403324904124,
        7935803032092749733,
    ]),
    BlsScalar([
        1099511627520,
        4781693605853266176,
        14278653781115140780,
        1321996527408953292,
    ]),
    BlsScalar([
        1108101562110,
        11160118862486653186,
        6896215983835597473,
        3061706882189606204,
    ]),
    BlsScalar([
        1116691496700,
        17538544119120040196,
        17960522260265605782,
        4801417236970259115,
    ]),
    BlsScalar([
        1125281431290,
        5470225302043875590,
        10578084462986062476,
        6541127591750912027,
    ]),
    BlsScalar([
        1133871365880,
        11848650558677262600,
        3195646665706519169,
        8280837946531564939,
    ]),
    BlsScalar([
        1146756267765,
        12192916406772567307,
        10568734043496755825,
        1667031441847768498,
    ]),
    BlsScalar([
        1155346202355,
        124597589696402701,
        3186296246217212519,
        3406741796628421410,
    ]),
    BlsScalar([
        1163936136945,
        6503022846329789711,
        14250602522647220828,
        5146452151409074321,
    ]),
    BlsScalar([
        1172526071535,
        12881448102963176721,
        6868164725367677521,
        6886162506189727233,
    ]),
    BlsScalar([
        1185410973420,
        13225713951058481428,
        14241252103157914177,
        272356001505930792,
    ]),
    BlsScalar([
        1194000908010,
        1157395133982316822,
        6858814305878370871,
        2012066356286583704,
    ]),
    BlsScalar([
        1202590842600,
        7535820390615703832,
        17923120582308379180,
        3751776711067236615,
    ]),
    BlsScalar([
        1211180777190,
        13914245647249090842,
        10540682785028835873,
        5491487065847889527,
    ]),
    BlsScalar([
        1219770711780,
        1845926830172926236,
        3158244987749292567,
        7231197420628542439,
    ]),
    BlsScalar([
        1232655613665,
        2190192678268230943,
        10531332365539529223,
        617390915944745998,
    ]),
    BlsScalar([
        1241245548255,
        8568617934901617953,
        3148894568259985916,
        2357101270725398910,
    ]),
    BlsScalar([
        1249835482845,
        14947043191535004963,
        14213200844689994225,
        4096811625506051821,
    ]),
    BlsScalar([
        1258425417435,
        2878724374458840357,
        6830763047410450919,
        5836521980286704733,
    ]),
    BlsScalar([
        1267015352025,
        9257149631092227367,
        17895069323840459228,
        7576232335067357644,
    ]),
    BlsScalar([
        1279900253910,
        9601415479187532074,
        6821412627921144268,
        962425830383561204,
    ]),
    BlsScalar([
        1288490188500,
        15979840735820919084,
        17885718904351152577,
        2702136185164214115,
    ]),
    BlsScalar([
        1297080123090,
        3911521918744754478,
        10503281107071609271,
        4441846539944867027,
    ]),
    BlsScalar([
        1305670057680,
        10289947175378141488,
        3120843309792065964,
        6181556894725519939,
    ]),
    BlsScalar([
        1314259992270,
        16668372432011528498,
        14185149586222074273,
        7921267249506172850,
    ]),
    BlsScalar([
        1327144894155,
        17012638280106833205,
        3111492890302759313,
        1307460744822376410,
    ]),
    BlsScalar([
        1335734828745,
        4944319463030668599,
        14175799166732767623,
        3047171099603029321,
    ]),
    BlsScalar([
        1344324763335,
        11322744719664055609,
        6793361369453224316,
        4786881454383682233,
    ]),
    BlsScalar([
        1352914697925,
        17701169976297442619,
        17857667645883232625,
        6526591809164335144,
    ]),
    BlsScalar([
        1361504632515,
        5632851159221278013,
        10475229848603689319,
        8266302163944988056,
    ]),
    BlsScalar([
        1374389534400,
        5977117007316582720,
        17848317226393925975,
        1652495659261191615,
    ]),
    BlsScalar([
        1382979468990,
        12355542263949969730,
        10465879429114382668,
        3392206014041844527,
    ]),
    BlsScalar([
        1391569403580,
        287223446873805124,
        3083441631834839362,
        5131916368822497439,
    ]),
    BlsScalar([
        1400159338170,
        6665648703507192134,
        14147747908264847671,
        6871626723603150350,
    ]),
    BlsScalar([
        1413044240055,
        7009914551602496841,
        3074091212345532711,
        257820218919353910,
    ]),
    BlsScalar([
        1421634174645,
        13388339808235883851,
        14138397488775541020,
        1997530573700006821,
    ]),
    BlsScalar([
        1430224109235,
        1320020991159719245,
        6755959691495997714,
        3737240928480659733,
    ]),
    BlsScalar([
        1438814043825,
        7698446247793106255,
        17820265967926006023,
        5476951283261312644,
    ]),
    BlsScalar([
        1447403978415,
        14076871504426493265,
        10437828170646462716,
        7216661638041965556,
    ]),
    BlsScalar([
        1460288880300,
        14421137352521797972,
        17810915548436699372,
        602855133358169115,
    ]),
    BlsScalar([
        1468878814890,
        2352818535445633366,
        10428477751157156066,
        2342565488138822027,
    ]),
    BlsScalar([
        1477468749480,
        8731243792079020376,
        3046039953877612759,
        4082275842919474939,
    ]),
    BlsScalar([
        1486058684070,
        15109669048712407386,
        14110346230307621068,
        5821986197700127850,
    ]),
    BlsScalar([
        1494648618660,
        3041350231636242780,
        6727908433028077762,
        7561696552480780762,
    ]),
    BlsScalar([
        1507533520545,
        3385616079731547487,
        14100995810818314418,
        947890047796984321,
    ]),
    BlsScalar([
        1516123455135,
        9764041336364934497,
        6718558013538771111,
        2687600402577637233,
    ]),
    BlsScalar([
        1524713389725,
        16142466592998321507,
        17782864289968779420,
        4427310757358290144,
    ]),
    BlsScalar([
        1533303324315,
        4074147775922156901,
        10400426492689236114,
        6167021112138943056,
    ]),
    BlsScalar([
        1541893258905,
        10452573032555543911,
        3017988695409692807,
        7906731466919595968,
    ]),
    BlsScalar([
        1554778160790,
        10796838880650848618,
        10391076073199929463,
        1292924962235799527,
    ]),
    BlsScalar([
        1563368095380,
        17175264137284235628,
        3008638275920386156,
        3032635317016452439,
    ]),
    BlsScalar([
        1571958029970,
        5106945320208071022,
        14072944552350394466,
        4772345671797105350,
    ]),
    BlsScalar([
        1580547964560,
        11485370576841458032,
        6690506755070851159,
        6512056026577758262,
    ]),
    BlsScalar([
        1589137899150,
        17863795833474845042,
        17754813031500859468,
        8251766381358411173,
    ]),
    BlsScalar([
        1602022801035,
        18208061681570149749,
        6681156335581544508,
        1637959876674614733,
    ]),
    BlsScalar([
        1610612735625,
        6139742864493985143,
        17745462612011552818,
        3377670231455267644,
    ]),
    BlsScalar([
        1619202670215,
        12518168121127372153,
        10363024814732009511,
        5117380586235920556,
    ]),
    BlsScalar([
        1627792604805,
        449849304051207547,
        2980587017452466205,
        6857090941016573468,
    ]),
    BlsScalar([
        1640677506690,
        794115152146512254,
        10353674395242702861,
        243284436332777027,
    ]),
    BlsScalar([
        1649267441280,
        7172540408779899264,
        2971236597963159554,
        1982994791113429939,
    ]),
    BlsScalar([
        1657857375870,
        13550965665413286274,
        14035542874393167863,
        3722705145894082850,
    ]),
    BlsScalar([
        1666447310460,
        1482646848337121668,
        6653105077113624557,
        5462415500674735762,
    ]),
    BlsScalar([
        1675037245050,
        7861072104970508678,
        17717411353543632866,
        7202125855455388673,
    ]),
    BlsScalar([
        1687922146935,
        8205337953065813385,
        6643754657624317906,
        588319350771592233,
    ]),
    BlsScalar([
        1696512081525,
        14583763209699200395,
        17708060934054326215,
        2328029705552245144,
    ]),
    BlsScalar([
        1705102016115,
        2515444392623035789,
        10325623136774782909,
        4067740060332898056,
    ]),
    BlsScalar([
        1713691950705,
        8893869649256422799,
        2943185339495239602,
        5807450415113550968,
    ]),
    BlsScalar([
        1722281885295,
        15272294905889809809,
        14007491615925247911,
        7547160769894203879,
    ]),
    BlsScalar([
        1735166787180,
        15616560753985114516,
        2933834920005932951,
        933354265210407439,
    ]),
    BlsScalar([
        1743756721770,
        3548241936908949910,
        13998141196435941261,
        2673064619991060350,
    ]),
    BlsScalar([
        1752346656360,
        9926667193542336920,
        6615703399156397954,
        4412774974771713262,
    ]),
    BlsScalar([
        1760936590950,
        16305092450175723930,
        17680009675586406263,
        6152485329552366173,
    ]),
    BlsScalar([
        1769526525540,
        4236773633099559324,
        10297571878306862957,
        7892195684333019085,
    ]),
    BlsScalar([
        1782411427425,
        4581039481194864031,
        17670659256097099613,
        1278389179649222644,
    ]),
    BlsScalar([
        1791001362015,
        10959464737828251041,
        10288221458817556306,
        3018099534429875556,
    ]),
    BlsScalar([
        1799591296605,
        17337889994461638051,
        2905783661538012999,
        4757809889210528468,
    ]),
    BlsScalar([
        1808181231195,
        5269571177385473445,
        13970089937968021309,
        6497520243991181379,
    ]),
    BlsScalar([
        1816771165785,
        11647996434018860455,
        6587652140688478002,
        8237230598771834291,
    ]),
    BlsScalar([
        1829656067670,
        11992262282114165162,
        13960739518478714658,
        1623424094088037850,
    ]),
    BlsScalar([
        1838246002260,
        18370687538747552172,
        6578301721199171351,
        3363134448868690762,
    ]),
    BlsScalar([
        1846835936850,
        6302368721671387566,
        17642607997629179661,
        5102844803649343673,
    ]),
    BlsScalar([
        1855425871440,
        12680793978304774576,
        10260170200349636354,
        6842555158429996585,
    ]),
    BlsScalar([
        1868310773325,
        13025059826400079283,
        17633257578139873010,
        228748653746200144,
    ]),
    BlsScalar([
        1876900707915,
        956741009323914677,
        10250819780860329704,
        1968459008526853056,
    ]),
    BlsScalar([
        1885490642505,
        7335166265957301687,
        2868381983580786397,
        3708169363307505968,
    ]),
    BlsScalar([
        1894080577095,
        13713591522590688697,
        13932688260010794706,
        5447879718088158879,
    ]),
    BlsScalar([
        1902670511685,
        1645272705514524091,
        6550250462731251400,
        7187590072868811791,
    ]),
    BlsScalar([
        1915555413570,
        1989538553609828798,
        13923337840521488056,
        573783568185015350,
    ]),
    BlsScalar([
        1924145348160,
        8367963810243215808,
        6540900043241944749,
        2313493922965668262,
    ]),
    BlsScalar([
        1932735282750,
        14746389066876602818,
        17605206319671953058,
        4053204277746321173,
    ]),
    BlsScalar([
        1941325217340,
        2678070249800438212,
        10222768522392409752,
        5792914632526974085,
    ]),
    BlsScalar([
        1949915151930,
        9056495506433825222,
        2840330725112866445,
        7532624987307626997,
    ]),
    BlsScalar([
        1962800053815,
        9400761354529129929,
        10213418102903103101,
        918818482623830556,
    ]),
    BlsScalar([
        1971389988405,
        15779186611162516939,
        2830980305623559794,
        2658528837404483468,
    ]),
    BlsScalar([
        1979979922995,
        3710867794086352333,
        13895286582053568104,
        4398239192185136379,
    ]),
    BlsScalar([
        1988569857585,
        10089293050719739343,
        6512848784774024797,
        6137949546965789291,
    ]),
    BlsScalar([
        1997159792175,
        16467718307353126353,
        17577155061204033106,
        7877659901746442202,
    ]),
    BlsScalar([
        2010044694060,
        16811984155448431060,
        6503498365284718146,
        1263853397062645762,
    ]),
    BlsScalar([
        2018634628650,
        4743665338372266454,
        17567804641714726456,
        3003563751843298673,
    ]),
    BlsScalar([
        2027224563240,
        11122090595005653464,
        10185366844435183149,
        4743274106623951585,
    ]),
    BlsScalar([
        2035814497830,
        17500515851639040474,
        2802929047155639842,
        6482984461404604497,
    ]),
    BlsScalar([
        2044404432420,
        5432197034562875868,
        13867235323585648152,
        8222694816185257408,
    ]),
    BlsScalar([
        2057289334305,
        5776462882658180575,
        2793578627666333192,
        1608888311501460968,
    ]),
    BlsScalar([
        2065879268895,
        12154888139291567585,
        13857884904096341501,
        3348598666282113879,
    ]),
    BlsScalar([
        2074469203485,
        86569322215402979,
        6475447106816798195,
        5088309021062766791,
    ]),
    BlsScalar([
        2083059138075,
        6464994578848789989,
        17539753383246806504,
        6828019375843419702,
    ]),
    BlsScalar([
        2095944039960,
        6809260426944094696,
        6466096687327491544,
        214212871159623262,
    ]),
    BlsScalar([
        2104533974550,
        13187685683577481706,
        17530402963757499853,
        1953923225940276173,
    ]),
    BlsScalar([
        2113123909140,
        1119366866501317100,
        10147965166477956547,
        3693633580720929085,
    ]),
    BlsScalar([
        2121713843730,
        7497792123134704110,
        2765527369198413240,
        5433343935501581997,
    ]),
    BlsScalar([
        2130303778320,
        13876217379768091120,
        13829833645628421549,
        7173054290282234908,
    ]),
    BlsScalar([
        2143188680205,
        14220483227863395827,
        2756176949709106589,
        559247785598438468,
    ]),
    BlsScalar([
        2151778614795,
        2152164410787231221,
        13820483226139114899,
        2298958140379091379,
    ]),
    BlsScalar([
        2160368549385,
        8530589667420618231,
        6438045428859571592,
        4038668495159744291,
    ]),
    BlsScalar([
        2168958483975,
        14909014924054005241,
        17502351705289579901,
        5778378849940397202,
    ]),
    BlsScalar([
        2177548418565,
        2840696106977840635,
        10119913908010036595,
        7518089204721050114,
    ]),
    BlsScalar([
        2190433320450,
        3184961955073145342,
        17493001285800273251,
        904282700037253673,
    ]),
    BlsScalar([
        2199023255040,
        9563387211706532352,
        10110563488520729944,
        2643993054817906585,
    ]),
    BlsScalar([
        2207613189630,
        15941812468339919362,
        2728125691241186637,
        4383703409598559497,
    ]),
    BlsScalar([
        2216203124220,
        3873493651263754756,
        13792431967671194947,
        6123413764379212408,
    ]),
    BlsScalar([
        2224793058810,
        10251918907897141766,
        6409994170391651640,
        7863124119159865320,
    ]),
    BlsScalar([
        2237677960695,
        10596184755992446473,
        13783081548181888296,
        1249317614476068879,
    ]),
    BlsScalar([
        2246267895285,
        16974610012625833483,
        6400643750902344989,
        2989027969256721791,
    ]),
    BlsScalar([
        2254857829875,
        4906291195549668877,
        17464950027332353299,
        4728738324037374702,
    ]),
    BlsScalar([
        2263447764465,
        11284716452183055887,
        10082512230052809992,
        6468448678818027614,
    ]),
    BlsScalar([
        2272037699055,
        17663141708816442897,
        2700074432773266685,
        8208159033598680526,
    ]),
    BlsScalar([
        2284922600940,
        18007407556911747604,
        10073161810563503341,
        1594352528914884085,
    ]),
    BlsScalar([
        2293512535530,
        5939088739835582998,
        2690724013283960035,
        3334062883695536997,
    ]),
    BlsScalar([
        2302102470120,
        12317513996468970008,
        13755030289713968344,
        5073773238476189908,
    ]),
    BlsScalar([
        2310692404710,
        249195179392805402,
        6372592492434425038,
        6813483593256842820,
    ]),
    BlsScalar([
        2323577306595,
        593461027488110109,
        13745679870224661694,
        199677088573046379,
    ]),
    BlsScalar([
        2332167241185,
        6971886284121497119,
        6363242072945118387,
        1939387443353699291,
    ]),
    BlsScalar([
        2340757175775,
        13350311540754884129,
        17427548349375126696,
        3679097798134352202,
    ]),
    BlsScalar([
        2349347110365,
        1281992723678719523,
        10045110552095583390,
        5418808152915005114,
    ]),
    BlsScalar([
        2357937044955,
        7660417980312106533,
        2662672754816040083,
        7158518507695658026,
    ]),
    BlsScalar([
        2370821946840,
        8004683828407411240,
        10035760132606276739,
        544712003011861585,
    ]),
    BlsScalar([
        2379411881430,
        14383109085040798250,
        2653322335326733432,
        2284422357792514497,
    ]),
    BlsScalar([
        2388001816020,
        2314790267964633644,
        13717628611756741742,
        4024132712573167408,
    ]),
    BlsScalar([
        2396591750610,
        8693215524598020654,
        6335190814477198435,
        5763843067353820320,
    ]),
    BlsScalar([
        2405181685200,
        15071640781231407664,
        17399497090907206744,
        7503553422134473231,
    ]),
    BlsScalar([
        2418066587085,
        15415906629326712371,
        6325840394987891784,
        889746917450676791,
    ]),
    BlsScalar([
        2426656521675,
        3347587812250547765,
        17390146671417900094,
        2629457272231329702,
    ]),
    BlsScalar([
        2435246456265,
        9726013068883934775,
        10007708874138356787,
        4369167627011982614,
    ]),
    BlsScalar([
        2443836390855,
        16104438325517321785,
        2625271076858813480,
        6108877981792635526,
    ]),
    BlsScalar([
        2452426325445,
        4036119508441157179,
        13689577353288821790,
        7848588336573288437,
    ]),
    BlsScalar([
        2465311227330,
        4380385356536461886,
        2615920657369506830,
        1234781831889491997,
    ]),
    BlsScalar([
        2473901161920,
        10758810613169848896,
        13680226933799515139,
        2974492186670144908,
    ]),
    BlsScalar([
        2482491096510,
        17137235869803235906,
        6297789136519971832,
        4714202541450797820,
    ]),
    BlsScalar([
        2491081031100,
        5068917052727071300,
        17362095412949980142,
        6453912896231450731,
    ]),
    BlsScalar([
        2499670965690,
        11447342309360458310,
        9979657615670436835,
        8193623251012103643,
    ]),
    BlsScalar([
        2512555867575,
        11791608157455763017,
        17352744993460673491,
        1579816746328307202,
    ]),
    BlsScalar([
        2521145802165,
        18170033414089150027,
        9970307196181130184,
        3319527101108960114,
    ]),
    BlsScalar([
        2529735736755,
        6101714597012985421,
        2587869398901586878,
        5059237455889613026,
    ]),
    BlsScalar([
        2538325671345,
        12480139853646372431,
        13652175675331595187,
        6798947810670265937,
    ]),
    BlsScalar([
        2551210573230,
        12824405701741677138,
        2578518979412280227,
        185141305986469497,
    ]),
    BlsScalar([
        2559800507820,
        756086884665512532,
        13642825255842288537,
        1924851660767122408,
    ]),
    BlsScalar([
        2568390442410,
        7134512141298899542,
        6260387458562745230,
        3664562015547775320,
    ]),
    BlsScalar([
        2576980377000,
        13512937397932286552,
        17324693734992753539,
        5404272370328428231,
    ]),
    BlsScalar([
        2585570311590,
        1444618580856121946,
        9942255937713210233,
        7143982725109081143,
    ]),
    BlsScalar([
        2598455213475,
        1788884428951426653,
        17315343315503446889,
        530176220425284702,
    ]),
    BlsScalar([
        2607045148065,
        8167309685584813663,
        9932905518223903582,
        2269886575205937614,
    ]),
    BlsScalar([
        2615635082655,
        14545734942218200673,
        2550467720944360275,
        4009596929986590526,
    ]),
    BlsScalar([
        2624225017245,
        2477416125142036067,
        13614773997374368585,
        5749307284767243437,
    ]),
    BlsScalar([
        2632814951835,
        8855841381775423077,
        6232336200094825278,
        7489017639547896349,
    ]),
    BlsScalar([
        2645699853720,
        9200107229870727784,
        13605423577885061934,
        875211134864099908,
    ]),
    BlsScalar([
        2654289788310,
        15578532486504114794,
        6222985780605518627,
        2614921489644752820,
    ]),
    BlsScalar([
        2662879722900,
        3510213669427950188,
        17287292057035526937,
        4354631844425405731,
    ]),
    BlsScalar([
        2671469657490,
        9888638926061337198,
        9904854259755983630,
        6094342199206058643,
    ]),
    BlsScalar([
        2680059592080,
        16267064182694724208,
        2522416462476440323,
        7834052553986711555,
    ]),
    BlsScalar([
        2692944493965,
        16611330030790028915,
        9895503840266676979,
        1220246049302915114,
    ]),
    BlsScalar([
        2701534428555,
        4543011213713864309,
        2513066042987133673,
        2959956404083568026,
    ]),
    BlsScalar([
        2710124363145,
        10921436470347251319,
        13577372319417141982,
        4699666758864220937,
    ]),
    BlsScalar([
        2718714297735,
        17299861726980638329,
        6194934522137598675,
        6439377113644873849,
    ]),
    BlsScalar([
        2727304232325,
        5231542909904473723,
        17259240798567606985,
        8179087468425526760,
    ]),
    BlsScalar([
        2740189134210,
        5575808757999778430,
        6185584102648292025,
        1565280963741730320,
    ]),
    BlsScalar([
        2748779068800,
        11954234014633165440,
        17249890379078300334,
        3304991318522383231,
    ]),
    BlsScalar([
        2757369003390,
        18332659271266552450,
        9867452581798757027,
        5044701673303036143,
    ]),
    BlsScalar([
        2765958937980,
        6264340454190387844,
        2485014784519213721,
        6784412028083689055,
    ]),
    BlsScalar([
        2778843839865,
        6608606302285692551,
        9858102162309450377,
        170605523399892614,
    ]),
    BlsScalar([
        2787433774455,
        12987031558919079561,
        2475664365029907070,
        1910315878180545526,
    ]),
    BlsScalar([
        2796023709045,
        918712741842914955,
        13539970641459915380,
        3650026232961198437,
    ]),
    BlsScalar([
        2804613643635,
        7297137998476301965,
        6157532844180372073,
        5389736587741851349,
    ]),
    BlsScalar([
        2813203578225,
        13675563255109688975,
        17221839120610380382,
        7129446942522504260,
    ]),
    BlsScalar([
        2826088480110,
        14019829103204993682,
        6148182424691065422,
        515640437838707820,
    ]),
    BlsScalar([
        2834678414700,
        1951510286128829076,
        17212488701121073732,
        2255350792619360731,
    ]),
    BlsScalar([
        2843268349290,
        8329935542762216086,
        9830050903841530425,
        3995061147400013643,
    ]),
    BlsScalar([
        2851858283880,
        14708360799395603096,
        2447613106561987118,
        5734771502180666555,
    ]),
    BlsScalar([
        2860448218470,
        2640041982319438490,
        13511919382991995428,
        7474481856961319466,
    ]),
    BlsScalar([
        2873333120355,
        2984307830414743197,
        2438262687072680468,
        860675352277523026,
    ]),
    BlsScalar([
        2881923054945,
        9362733087048130207,
        13502568963502688777,
        2600385707058175937,
    ]),
    BlsScalar([
        2890512989535,
        15741158343681517217,
        6120131166223145470,
        4340096061838828849,
    ]),
    BlsScalar([
        2899102924125,
        3672839526605352611,
        17184437442653153780,
        6079806416619481760,
    ]),
    BlsScalar([
        2907692858715,
        10051264783238739621,
        9801999645373610473,
        7819516771400134672,
    ]),
    BlsScalar([
        2920577760600,
        10395530631334044328,
        17175087023163847129,
        1205710266716338231,
    ]),
    BlsScalar([
        2929167695190,
        16773955887967431338,
        9792649225884303822,
        2945420621496991143,
    ]),
    BlsScalar([
        2937757629780,
        4705637070891266732,
        2410211428604760516,
        4685130976277644055,
    ]),
    BlsScalar([
        2946347564370,
        11084062327524653742,
        13474517705034768825,
        6424841331058296966,
    ]),
    BlsScalar([
        2954937498960,
        17462487584158040752,
        6092079907755225518,
        8164551685838949878,
    ]),
    BlsScalar([
        2967822400845,
        17806753432253345459,
        13465167285545462174,
        1550745181155153437,
    ]),
    BlsScalar([
        2976412335435,
        5738434615177180853,
        6082729488265918868,
        3290455535935806349,
    ]),
    BlsScalar([
        2985002270025,
        12116859871810567863,
        17147035764695927177,
        5030165890716459260,
    ]),
    BlsScalar([
        2993592204615,
        48541054734403257,
        9764597967416383871,
        6769876245497112172,
    ]),
    BlsScalar([
        3006477106500,
        392806902829707964,
        17137685345206620527,
        156069740813315731,
    ]),
    BlsScalar([
        3015067041090,
        6771232159463094974,
        9755247547927077220,
        1895780095593968643,
    ]),
    BlsScalar([
        3023656975680,
        13149657416096481984,
        2372809750647533913,
        3635490450374621555,
    ]),
    BlsScalar([
        3032246910270,
        1081338599020317378,
        13437116027077542223,
        5375200805155274466,
    ]),
    BlsScalar([
        3040836844860,
        7459763855653704388,
        6054678229797998916,
        7114911159935927378,
    ]),
    BlsScalar([
        3053721746745,
        7804029703749009095,
        13427765607588235572,
        501104655252130937,
    ]),
    BlsScalar([
        3062311681335,
        14182454960382396105,
        6045327810308692265,
        2240815010032783849,
    ]),
    BlsScalar([
        3070901615925,
        2114136143306231499,
        17109634086738700575,
        3980525364813436760,
    ]),
    BlsScalar([
        3079491550515,
        8492561399939618509,
        9727196289459157268,
        5720235719594089672,
    ]),
    BlsScalar([
        3088081485105,
        14870986656573005519,
        2344758492179613961,
        7459946074374742584,
    ]),
    BlsScalar([
        3100966386990,
        15215252504668310226,
        9717845869969850617,
        846139569690946143,
    ]),
    BlsScalar([
        3109556321580,
        3146933687592145620,
        2335408072690307311,
        2585849924471599055,
    ]),
    BlsScalar([
        3118146256170,
        9525358944225532630,
        13399714349120315620,
        4325560279252251966,
    ]),
    BlsScalar([
        3126736190760,
        15903784200858919640,
        6017276551840772313,
        6065270634032904878,
    ]),
    BlsScalar([
        3135326125350,
        3835465383782755034,
        17081582828270780623,
        7804980988813557789,
    ]),
    BlsScalar([
        3148211027235,
        4179731231878059741,
        6007926132351465663,
        1191174484129761349,
    ]),
    BlsScalar([
        3156800961825,
        10558156488511446751,
        17072232408781473972,
        2930884838910414260,
    ]),
    BlsScalar([
        3165390896415,
        16936581745144833761,
        9689794611501930665,
        4670595193691067172,
    ]),
    BlsScalar([
        3173980831005,
        4868262928068669155,
        2307356814222387359,
        6410305548471720084,
    ]),
    BlsScalar([
        3182570765595,
        11246688184702056165,
        13371663090652395668,
        8150015903252372995,
    ]),
    BlsScalar([
        3195455667480,
        11590954032797360872,
        2298006394733080708,
        1536209398568576555,
    ]),
    BlsScalar([
        3204045602070,
        17969379289430747882,
        13362312671163089017,
        3275919753349229466,
    ]),
    BlsScalar([
        3212635536660,
        5901060472354583276,
        5979874873883545711,
        5015630108129882378,
    ]),
    BlsScalar([
        3221225471250,
        12279485728987970286,
        17044181150313554020,
        6755340462910535289,
    ]),
    BlsScalar([
        3234110373135,
        12623751577083274993,
        5970524454394239060,
        141533958226738849,
    ]),
    BlsScalar([
        3242700307725,
        555432760007110387,
        17034830730824247370,
        1881244313007391760,
    ]),
    BlsScalar([
        3251290242315,
        6933858016640497397,
        9652392933544704063,
        3620954667788044672,
    ]),
    BlsScalar([
        3259880176905,
        13312283273273884407,
        2269955136265160756,
        5360665022568697584,
    ]),
    BlsScalar([
        3268470111495,
        1243964456197719801,
        13334261412695169066,
        7100375377349350495,
    ]),
    BlsScalar([
        3281355013380,
        1588230304293024508,
        2260604716775854106,
        486568872665554055,
    ]),
    BlsScalar([
        3289944947970,
        7966655560926411518,
        13324910993205862415,
        2226279227446206966,
    ]),
    BlsScalar([
        3298534882560,
        14345080817559798528,
        5942473195926319108,
        3965989582226859878,
    ]),
    BlsScalar([
        3307124817150,
        2276762000483633922,
        17006779472356327418,
        5705699937007512789,
    ]),
    BlsScalar([
        3315714751740,
        8655187257117020932,
        9624341675076784111,
        7445410291788165701,
    ]),
    BlsScalar([
        3328599653625,
        8999453105212325639,
        16997429052867020767,
        831603787104369260,
    ]),
    BlsScalar([
        3337189588215,
        15377878361845712649,
        9614991255587477460,
        2571314141885022172,
    ]),
    BlsScalar([
        3345779522805,
        3309559544769548043,
        2232553458307934154,
        4311024496665675084,
    ]),
    BlsScalar([
        3354369457395,
        9687984801402935053,
        13296859734737942463,
        6050734851446327995,
    ]),
    BlsScalar([
        3362959391985,
        16066410058036322063,
        5914421937458399156,
        7790445206226980907,
    ]),
    BlsScalar([
        3375844293870,
        16410675906131626770,
        13287509315248635812,
        1176638701543184466,
    ]),
    BlsScalar([
        3384434228460,
        4342357089055462164,
        5905071517969092506,
        2916349056323837378,
    ]),
    BlsScalar([
        3393024163050,
        10720782345688849174,
        16969377794399100815,
        4656059411104490289,
    ]),
    BlsScalar([
        3401614097640,
        17099207602322236184,
        9586939997119557508,
        6395769765885143201,
    ]),
    BlsScalar([
        3410204032230,
        5030888785246071578,
        2204502199840014202,
        8135480120665796113,
    ]),
    BlsScalar([
        3423088934115,
        5375154633341376285,
        9577589577630250858,
        1521673615981999672,
    ]),
    BlsScalar([
        3431678868705,
        11753579889974763295,
        2195151780350707551,
        3261383970762652584,
    ]),
    BlsScalar([
        3440268803295,
        18132005146608150305,
        13259458056780715860,
        5001094325543305495,
    ]),
    BlsScalar([
        3448858737885,
        6063686329531985699,
        5877020259501172554,
        6740804680323958407,
    ]),
    BlsScalar([
        3461743639770,
        6407952177627290406,
        13250107637291409210,
        126998175640161966,
    ]),
    BlsScalar([
        3470333574360,
        12786377434260677416,
        5867669840011865903,
        1866708530420814878,
    ]),
    BlsScalar([
        3478923508950,
        718058617184512810,
        16931976116441874213,
        3606418885201467789,
    ]),
    BlsScalar([
        3487513443540,
        7096483873817899820,
        9549538319162330906,
        5346129239982120701,
    ]),
    BlsScalar([
        3496103378130,
        13474909130451286830,
        2167100521882787599,
        7085839594762773613,
    ]),
    BlsScalar([
        3508988280015,
        13819174978546591537,
        9540187899673024255,
        472033090078977172,
    ]),
    BlsScalar([
        3517578214605,
        1750856161470426931,
        2157750102393480949,
        2211743444859630084,
    ]),
    BlsScalar([
        3526168149195,
        8129281418103813941,
        13222056378823489258,
        3951453799640282995,
    ]),
    BlsScalar([
        3534758083785,
        14507706674737200951,
        5839618581543945951,
        5691164154420935907,
    ]),
    BlsScalar([
        3543348018375,
        2439387857661036345,
        16903924857973954261,
        7430874509201588818,
    ]),
    BlsScalar([
        3556232920260,
        2783653705756341052,
        5830268162054639301,
        817068004517792378,
    ]),
    BlsScalar([
        3564822854850,
        9162078962389728062,
        16894574438484647610,
        2556778359298445289,
    ]),
    BlsScalar([
        3573412789440,
        15540504219023115072,
        9512136641205104303,
        4296488714079098201,
    ]),
    BlsScalar([
        3582002724030,
        3472185401946950466,
        2129698843925560997,
        6036199068859751113,
    ]),
    BlsScalar([
        3590592658620,
        9850610658580337476,
        13194005120355569306,
        7775909423640404024,
    ]),
    BlsScalar([
        3603477560505,
        10194876506675642183,
        2120348424436254346,
        1162102918956607584,
    ]),
    BlsScalar([
        3612067495095,
        16573301763309029193,
        13184654700866262655,
        2901813273737260495,
    ]),
    BlsScalar([
        3620657429685,
        4504982946232864587,
        5802216903586719349,
        4641523628517913407,
    ]),
    BlsScalar([
        3629247364275,
        10883408202866251597,
        16866523180016727658,
        6381233983298566318,
    ]),
    BlsScalar([
        3637837298865,
        17261833459499638607,
        9484085382737184351,
        8120944338079219230,
    ]),
    BlsScalar([
        3650722200750,
        17606099307594943314,
        16857172760527421007,
        1507137833395422789,
    ]),
    BlsScalar([
        3659312135340,
        5537780490518778708,
        9474734963247877701,
        3246848188176075701,
    ]),
    BlsScalar([
        3667902069930,
        11916205747152165718,
        2092297165968334394,
        4986558542956728613,
    ]),
    BlsScalar([
        3676492004520,
        18294631003785552728,
        13156603442398342703,
        6726268897737381524,
    ]),
    BlsScalar([
        3689376906405,
        192152778171305819,
        2082946746479027744,
        112462393053585084,
    ]),
    BlsScalar([
        3697966840995,
        6570578034804692829,
        13147253022909036053,
        1852172747834237995,
    ]),
    BlsScalar([
        3706556775585,
        12949003291438079839,
        5764815225629492746,
        3591883102614890907,
    ]),
    BlsScalar([
        3715146710175,
        880684474361915233,
        16829121502059501056,
        5331593457395543818,
    ]),
    BlsScalar([
        3723736644765,
        7259109730995302243,
        9446683704779957749,
        7071303812176196730,
    ]),
    BlsScalar([
        3736621546650,
        7603375579090606950,
        16819771082570194405,
        457497307492400289,
    ]),
    BlsScalar([
        3745211481240,
        13981800835723993960,
        9437333285290651098,
        2197207662273053201,
    ]),
    BlsScalar([
        3753801415830,
        1913482018647829354,
        2054895488011107792,
        3936918017053706113,
    ]),
    BlsScalar([
        3762391350420,
        8291907275281216364,
        13119201764441116101,
        5676628371834359024,
    ]),
    BlsScalar([
        3770981285010,
        14670332531914603374,
        5736763967161572794,
        7416338726615011936,
    ]),
    BlsScalar([
        3783866186895,
        15014598380009908081,
        13109851344951809450,
        802532221931215495,
    ]),
    BlsScalar([
        3792456121485,
        2946279562933743475,
        5727413547672266144,
        2542242576711868407,
    ]),
    BlsScalar([
        3801046056075,
        9324704819567130485,
        16791719824102274453,
        4281952931492521318,
    ]),
    BlsScalar([
        3809635990665,
        15703130076200517495,
        9409282026822731146,
        6021663286273174230,
    ]),
    BlsScalar([
        3818225925255,
        3634811259124352889,
        2026844229543187840,
        7761373641053827142,
    ]),
    BlsScalar([
        3831110827140,
        3979077107219657596,
        9399931607333424496,
        1147567136370030701,
    ]),
    BlsScalar([
        3839700761730,
        10357502363853044606,
        2017493810053881189,
        2887277491150683613,
    ]),
    BlsScalar([
        3848290696320,
        16735927620486431616,
        13081800086483889498,
        4626987845931336524,
    ]),
    BlsScalar([
        3856880630910,
        4667608803410267010,
        5699362289204346192,
        6366698200711989436,
    ]),
    BlsScalar([
        3865470565500,
        11046034060043654020,
        16763668565634354501,
        8106408555492642347,
    ]),
    BlsScalar([
        3878355467385,
        11390299908138958727,
        5690011869715039541,
        1492602050808845907,
    ]),
    BlsScalar([
        3886945401975,
        17768725164772345737,
        16754318146145047850,
        3232312405589498818,
    ]),
    BlsScalar([
        3895535336565,
        5700406347696181131,
        9371880348865504544,
        4972022760370151730,
    ]),
    BlsScalar([
        3904125271155,
        12078831604329568141,
        1989442551585961237,
        6711733115150804642,
    ]),
    BlsScalar([
        3917010173040,
        12423097452424872848,
        9362529929376197893,
        97926610467008201,
    ]),
    BlsScalar([
        3925600107630,
        354778635348708242,
        1980092132096654587,
        1837636965247661113,
    ]),
    BlsScalar([
        3934190042220,
        6733203891982095252,
        13044398408526662896,
        3577347320028314024,
    ]),
    BlsScalar([
        3942779976810,
        13111629148615482262,
        5661960611247119589,
        5317057674808966936,
    ]),
    BlsScalar([
        3951369911400,
        1043310331539317656,
        16726266887677127899,
        7056768029589619847,
    ]),
    BlsScalar([
        3964254813285,
        1387576179634622363,
        5652610191757812939,
        442961524905823407,
    ]),
    BlsScalar([
        3972844747875,
        7766001436268009373,
        16716916468187821248,
        2182671879686476318,
    ]),
    BlsScalar([
        3981434682465,
        14144426692901396383,
        9334478670908277941,
        3922382234467129230,
    ]),
    BlsScalar([
        3990024617055,
        2076107875825231777,
        1952040873628734635,
        5662092589247782142,
    ]),
    BlsScalar([
        3998614551645,
        8454533132458618787,
        13016347150058742944,
        7401802944028435053,
    ]),
    BlsScalar([
        4011499453530,
        8798798980553923494,
        1942690454139427984,
        787996439344638613,
    ]),
    BlsScalar([
        4020089388120,
        15177224237187310504,
        13006996730569436293,
        2527706794125291524,
    ]),
    BlsScalar([
        4028679322710,
        3108905420111145898,
        5624558933289892987,
        4267417148905944436,
    ]),
    BlsScalar([
        4037269257300,
        9487330676744532908,
        16688865209719901296,
        6007127503686597347,
    ]),
    BlsScalar([
        4045859191890,
        15865755933377919918,
        9306427412440357989,
        7746837858467250259,
    ]),
    BlsScalar([
        4058744093775,
        16210021781473224625,
        16679514790230594645,
        1133031353783453818,
    ]),
    BlsScalar([
        4067334028365,
        4141702964397060019,
        9297076992951051339,
        2872741708564106730,
    ]),
    BlsScalar([
        4075923962955,
        10520128221030447029,
        1914639195671508032,
        4612452063344759642,
    ]),
    BlsScalar([
        4084513897545,
        16898553477663834039,
        12978945472101516341,
        6352162418125412553,
    ]),
    BlsScalar([
        4093103832135,
        4830234660587669433,
        5596507674821973035,
        8091872772906065465,
    ]),
    BlsScalar([
        4105988734020,
        5174500508682974140,
        12969595052612209691,
        1478066268222269024,
    ]),
    BlsScalar([
        4114578668610,
        11552925765316361150,
        5587157255332666384,
        3217776623002921936,
    ]),
    BlsScalar([
        4123168603200,
        17931351021949748160,
        16651463531762674693,
        4957486977783574847,
    ]),
    BlsScalar([
        4131758537790,
        5863032204873583554,
        9269025734483131387,
        6697197332564227759,
    ]),
    BlsScalar([
        4144643439675,
        6207298052968888261,
        16642113112273368043,
        83390827880431318,
    ]),
    BlsScalar([
        4153233374265,
        12585723309602275271,
        9259675314993824736,
        1823101182661084230,
    ]),
    BlsScalar([
        4161823308855,
        517404492526110665,
        1877237517714281430,
        3562811537441737142,
    ]),
    BlsScalar([
        4170413243445,
        6895829749159497675,
        12941543794144289739,
        5302521892222390053,
    ]),
    BlsScalar([
        4179003178035,
        13274255005792884685,
        5559105996864746432,
        7042232247003042965,
    ]),
    BlsScalar([
        4191888079920,
        13618520853888189392,
        12932193374654983088,
        428425742319246524,
    ]),
    BlsScalar([
        4200478014510,
        1550202036812024786,
        5549755577375439782,
        2168136097099899436,
    ]),
    BlsScalar([
        4209067949100,
        7928627293445411796,
        16614061853805448091,
        3907846451880552347,
    ]),
    BlsScalar([
        4217657883690,
        14307052550078798806,
        9231624056525904784,
        5647556806661205259,
    ]),
    BlsScalar([
        4226247818280,
        2238733733002634200,
        1849186259246361478,
        7387267161441858171,
    ]),
    BlsScalar([
        4239132720165,
        2582999581097938907,
        9222273637036598134,
        773460656758061730,
    ]),
    BlsScalar([
        4247722654755,
        8961424837731325917,
        1839835839757054827,
        2513171011538714642,
    ]),
    BlsScalar([
        4256312589345,
        15339850094364712927,
        12904142116187063136,
        4252881366319367553,
    ]),
    BlsScalar([
        4264902523935,
        3271531277288548321,
        5521704318907519830,
        5992591721100020465,
    ]),
    BlsScalar([
        4273492458525,
        9649956533921935331,
        16586010595337528139,
        7732302075880673376,
    ]),
    BlsScalar([
        4286377360410,
        9994222382017240038,
        5512353899418213179,
        1118495571196876936,
    ]),
    BlsScalar([
        4294967295000,
        16372647638650627048,
        16576660175848221488,
        2858205925977529847,
    ]),
    BlsScalar([
        4303557229590,
        4304328821574462442,
        9194222378568678182,
        4597916280758182759,
    ]),
    BlsScalar([
        4312147164180,
        10682754078207849452,
        1811784581289134875,
        6337626635538835671,
    ]),
    BlsScalar([
        4320737098770,
        17061179334841236462,
        12876090857719143184,
        8077336990319488582,
    ]),
    BlsScalar([
        4333622000655,
        17405445182936541169,
        1802434161799828224,
        1463530485635692142,
    ]),
    BlsScalar([
        4342211935245,
        5337126365860376563,
        12866740438229836534,
        3203240840416345053,
    ]),
    BlsScalar([
        4350801869835,
        11715551622493763573,
        5484302640950293227,
        4942951195196997965,
    ]),
    BlsScalar([
        4359391804425,
        18093976879127150583,
        16548608917380301536,
        6682661549977650876,
    ]),
    BlsScalar([
        4372276706310,
        18438242727222455290,
        5474952221460986576,
        68855045293854436,
    ]),
    BlsScalar([
        4380866640900,
        6369923910146290684,
        16539258497890994886,
        1808565400074507347,
    ]),
    BlsScalar([
        4389456575490,
        12748349166779677694,
        9156820700611451579,
        3548275754855160259,
    ]),
    BlsScalar([
        4398046510080,
        680030349703513088,
        1774382903331908273,
        5287986109635813171,
    ]),
    BlsScalar([
        4406636444670,
        7058455606336900098,
        12838689179761916582,
        7027696464416466082,
    ]),
    BlsScalar([
        4419521346555,
        7402721454432204805,
        1765032483842601622,
        413889959732669642,
    ]),
    BlsScalar([
        4428111281145,
        13781146711065591815,
        12829338760272609931,
        2153600314513322553,
    ]),
    BlsScalar([
        4436701215735,
        1712827893989427209,
        5446900962993066625,
        3893310669293975465,
    ]),
    BlsScalar([
        4445291150325,
        8091253150622814219,
        16511207239423074934,
        5633021024074628376,
    ]),
    BlsScalar([
        4453881084915,
        14469678407256201229,
        9128769442143531627,
        7372731378855281288,
    ]),
    BlsScalar([
        4466765986800,
        14813944255351505936,
        16501856819933768283,
        758924874171484847,
    ]),
    BlsScalar([
        4475355921390,
        2745625438275341330,
        9119419022654224977,
        2498635228952137759,
    ]),
    BlsScalar([
        4483945855980,
        9124050694908728340,
        1736981225374681670,
        4238345583732790671,
    ]),
    BlsScalar([
        4492535790570,
        15502475951542115350,
        12801287501804689979,
        5978055938513443582,
    ]),
    BlsScalar([
        4501125725160,
        3434157134465950744,
        5418849704525146673,
        7717766293294096494,
    ]),
    BlsScalar([
        4514010627045,
        3778422982561255451,
        12791937082315383329,
        1103959788610300053,
    ]),
    BlsScalar([
        4522600561635,
        10156848239194642461,
        5409499285035840022,
        2843670143390952965,
    ]),
    BlsScalar([
        4531190496225,
        16535273495828029471,
        16473805561465848331,
        4583380498171605876,
    ]),
    BlsScalar([
        4539780430815,
        4466954678751864865,
        9091367764186305025,
        6323090852952258788,
    ]),
    BlsScalar([
        4548370365405,
        10845379935385251875,
        1708929966906761718,
        8062801207732911700,
    ]),
    BlsScalar([
        4561255267290,
        11189645783480556582,
        9082017344696998374,
        1448994703049115259,
    ]),
    BlsScalar([
        4569845201880,
        17568071040113943592,
        1699579547417455067,
        3188705057829768171,
    ]),
    BlsScalar([
        4578435136470,
        5499752223037778986,
        12763885823847463377,
        4928415412610421082,
    ]),
    BlsScalar([
        4587025071060,
        11878177479671165996,
        5381448026567920070,
        6668125767391073994,
    ]),
    BlsScalar([
        4599909972945,
        12222443327766470703,
        12754535404358156726,
        54319262707277553,
    ]),
    BlsScalar([
        4608499907535,
        154124510690306097,
        5372097607078613420,
        1794029617487930465,
    ]),
    BlsScalar([
        4617089842125,
        6532549767323693107,
        16436403883508621729,
        3533739972268583376,
    ]),
    BlsScalar([
        4625679776715,
        12910975023957080117,
        9053966086229078422,
        5273450327049236288,
    ]),
    BlsScalar([
        4634269711305,
        842656206880915511,
        1671528288949535116,
        7013160681829889200,
    ]),
    BlsScalar([
        4647154613190,
        1186922054976220218,
        9044615666739771772,
        399354177146092759,
    ]),
    BlsScalar([
        4655744547780,
        7565347311609607228,
        1662177869460228465,
        2139064531926745671,
    ]),
    BlsScalar([
        4664334482370,
        13943772568242994238,
        12726484145890236774,
        3878774886707398582,
    ]),
    BlsScalar([
        4672924416960,
        1875453751166829632,
        5344046348610693468,
        5618485241488051494,
    ]),
    BlsScalar([
        4681514351550,
        8253879007800216642,
        16408352625040701777,
        7358195596268704405,
    ]),
    BlsScalar([
        4694399253435,
        8598144855895521349,
        5334695929121386817,
        744389091584907965,
    ]),
    BlsScalar([
        4702989188025,
        14976570112528908359,
        16399002205551395126,
        2484099446365560876,
    ]),
    BlsScalar([
        4711579122615,
        2908251295452743753,
        9016564408271851820,
        4223809801146213788,
    ]),
    BlsScalar([
        4720169057205,
        9286676552086130763,
        1634126610992308513,
        5963520155926866700,
    ]),
    BlsScalar([
        4728758991795,
        15665101808719517773,
        12698432887422316822,
        7703230510707519611,
    ]),
    BlsScalar([
        4741643893680,
        16009367656814822480,
        1624776191503001862,
        1089424006023723171,
    ]),
    BlsScalar([
        4750233828270,
        3941048839738657874,
        12689082467933010172,
        2829134360804376082,
    ]),
    BlsScalar([
        4758823762860,
        10319474096372044884,
        5306644670653466865,
        4568844715585028994,
    ]),
    BlsScalar([
        4767413697450,
        16697899353005431894,
        16370950947083475174,
        6308555070365681905,
    ]),
    BlsScalar([
        4776003632040,
        4629580535929267288,
        8988513149803931868,
        8048265425146334817,
    ]),
    BlsScalar([
        4788888533925,
        4973846384024571995,
        16361600527594168524,
        1434458920462538376,
    ]),
    BlsScalar([
        4797478468515,
        11352271640657959005,
        8979162730314625217,
        3174169275243191288,
    ]),
    BlsScalar([
        4806068403105,
        17730696897291346015,
        1596724933035081910,
        4913879630023844200,
    ]),
    BlsScalar([
        4814658337695,
        5662378080215181409,
        12661031209465090220,
        6653589984804497111,
    ]),
    BlsScalar([
        4827543239580,
        6006643928310486116,
        1587374513545775260,
        39783480120700671,
    ]),
    BlsScalar([
        4836133174170,
        12385069184943873126,
        12651680789975783569,
        1779493834901353582,
    ]),
    BlsScalar([
        4844723108760,
        316750367867708520,
        5269242992696240263,
        3519204189682006494,
    ]),
    BlsScalar([
        4853313043350,
        6695175624501095530,
        16333549269126248572,
        5258914544462659405,
    ]),
    BlsScalar([
        4861902977940,
        13073600881134482540,
        8951111471846705265,
        6998624899243312317,
    ]),
    BlsScalar([
        4874787879825,
        13417866729229787247,
        16324198849636941921,
        384818394559515876,
    ]),
    BlsScalar([
        4883377814415,
        1349547912153622641,
        8941761052357398615,
        2124528749340168788,
    ]),
    BlsScalar([
        4891967749005,
        7727973168787009651,
        1559323255077855308,
        3864239104120821700,
    ]),
    BlsScalar([
        4900557683595,
        14106398425420396661,
        12623629531507863617,
        5603949458901474611,
    ]),
    BlsScalar([
        4909147618185,
        2038079608344232055,
        5241191734228320311,
        7343659813682127523,
    ]),
    BlsScalar([
        4922032520070,
        2382345456439536762,
        12614279112018556967,
        729853308998331082,
    ]),
    BlsScalar([
        4930622454660,
        8760770713072923772,
        5231841314739013660,
        2469563663778983994,
    ]),
    BlsScalar([
        4939212389250,
        15139195969706310782,
        16296147591169021969,
        4209274018559636905,
    ]),
    BlsScalar([
        4947802323840,
        3070877152630146176,
        8913709793889478663,
        5948984373340289817,
    ]),
    BlsScalar([
        4956392258430,
        9449302409263533186,
        1531271996609935356,
        7688694728120942729,
    ]),
    BlsScalar([
        4969277160315,
        9793568257358837893,
        8904359374400172012,
        1074888223437146288,
    ]),
    BlsScalar([
        4977867094905,
        16171993513992224903,
        1521921577120628705,
        2814598578217799200,
    ]),
    BlsScalar([
        4986457029495,
        4103674696916060297,
        12586227853550637015,
        4554308932998452111,
    ]),
    BlsScalar([
        4995046964085,
        10482099953549447307,
        5203790056271093708,
        6294019287779105023,
    ]),
    BlsScalar([
        5003636898675,
        16860525210182834317,
        16268096332701102017,
        8033729642559757934,
    ]),
    BlsScalar([
        5016521800560,
        17204791058278139024,
        5194439636781787057,
        1419923137875961494,
    ]),
    BlsScalar([
        5025111735150,
        5136472241201974418,
        16258745913211795367,
        3159633492656614405,
    ]),
    BlsScalar([
        5033701669740,
        11514897497835361428,
        8876308115932252060,
        4899343847437267317,
    ]),
    BlsScalar([
        5042291604330,
        17893322754468748438,
        1493870318652708753,
        6639054202217920229,
    ]),
    BlsScalar([
        5055176506215,
        18237588602564053145,
        8866957696442945409,
        25247697534123788,
    ]),
    BlsScalar([
        5063766440805,
        6169269785487888539,
        1484519899163402103,
        1764958052314776700,
    ]),
    BlsScalar([
        5072356375395,
        12547695042121275549,
        12548826175593410412,
        3504668407095429611,
    ]),
    BlsScalar([
        5080946309985,
        479376225045110943,
        5166388378313867106,
        5244378761876082523,
    ]),
    BlsScalar([
        5089536244575,
        6857801481678497953,
        16230694654743875415,
        6984089116656735434,
    ]),
    BlsScalar([
        5102421146460,
        7202067329773802660,
        5157037958824560455,
        370282611972938994,
    ]),
    BlsScalar([
        5111011081050,
        13580492586407189670,
        16221344235254568764,
        2109992966753591905,
    ]),
    BlsScalar([
        5119601015640,
        1512173769331025064,
        8838906437975025458,
        3849703321534244817,
    ]),
    BlsScalar([
        5128190950230,
        7890599025964412074,
        1456468640695482151,
        5589413676314897729,
    ]),
    BlsScalar([
        5136780884820,
        14269024282597799084,
        12520774917125490460,
        7329124031095550640,
    ]),
    BlsScalar([
        5149665786705,
        14613290130693103791,
        1447118221206175500,
        715317526411754200,
    ]),
    BlsScalar([
        5158255721295,
        2544971313616939185,
        12511424497636183810,
        2455027881192407111,
    ]),
    BlsScalar([
        5166845655885,
        8923396570250326195,
        5128986700356640503,
        4194738235973060023,
    ]),
    BlsScalar([
        5175435590475,
        15301821826883713205,
        16193292976786648812,
        5934448590753712934,
    ]),
    BlsScalar([
        5184025525065,
        3233503009807548599,
        8810855179507105506,
        7674158945534365846,
    ]),
    BlsScalar([
        5196910426950,
        3577768857902853306,
        16183942557297342162,
        1060352440850569405,
    ]),
    BlsScalar([
        5205500361540,
        9956194114536240316,
        8801504760017798855,
        2800062795631222317,
    ]),
    BlsScalar([
        5214090296130,
        16334619371169627326,
        1419066962738255548,
        4539773150411875229,
    ]),
    BlsScalar([
        5222680230720,
        4266300554093462720,
        12483373239168263858,
        6279483505192528140,
    ]),
    BlsScalar([
        5231270165310,
        10644725810726849730,
        5100935441888720551,
        8019193859973181052,
    ]),
    BlsScalar([
        5244155067195,
        10988991658822154437,
        12474022819678957207,
        1405387355289384611,
    ]),
    BlsScalar([
        5252745001785,
        17367416915455541447,
        5091585022399413900,
        3145097710070037523,
    ]),
    BlsScalar([
        5261334936375,
        5299098098379376841,
        16155891298829422210,
        4884808064850690434,
    ]),
    BlsScalar([
        5269924870965,
        11677523355012763851,
        8773453501549878903,
        6624518419631343346,
    ]),
    BlsScalar([
        5282809772850,
        12021789203108068558,
        16146540879340115559,
        10711914947546905,
    ]),
    BlsScalar([
        5291399707440,
        18400214459741455568,
        8764103082060572252,
        1750422269728199817,
    ]),
    BlsScalar([
        5299989642030,
        6331895642665290962,
        1381665284781028946,
        3490132624508852729,
    ]),
    BlsScalar([
        5308579576620,
        12710320899298677972,
        12445971561211037255,
        5229842979289505640,
    ]),
    BlsScalar([
        5317169511210,
        642002082222513366,
        5063533763931493949,
        6969553334070158552,
    ]),
    BlsScalar([
        5330054413095,
        986267930317818073,
        12436621141721730605,
        355746829386362111,
    ]),
    BlsScalar([
        5338644347685,
        7364693186951205083,
        5054183344442187298,
        2095457184167015023,
    ]),
    BlsScalar([
        5347234282275,
        13743118443584592093,
        16118489620872195607,
        3835167538947667934,
    ]),
    BlsScalar([
        5355824216865,
        1674799626508427487,
        8736051823592652301,
        5574877893728320846,
    ]),
    BlsScalar([
        5364414151455,
        8053224883141814497,
        1353614026313108994,
        7314588248508973758,
    ]),
    BlsScalar([
        5377299053340,
        8397490731237119204,
        8726701404103345650,
        700781743825177317,
    ]),
    BlsScalar([
        5385888987930,
        14775915987870506214,
        1344263606823802343,
        2440492098605830229,
    ]),
    BlsScalar([
        5394478922520,
        2707597170794341608,
        12408569883253810653,
        4180202453386483140,
    ]),
    BlsScalar([
        5403068857110,
        9086022427427728618,
        5026132085974267346,
        5919912808167136052,
    ]),
    BlsScalar([
        5411658791700,
        15464447684061115628,
        16090438362404275655,
        7659623162947788963,
    ]),
    BlsScalar([
        5424543693585,
        15808713532156420335,
        5016781666484960695,
        1045816658263992523,
    ]),
    BlsScalar([
        5433133628175,
        3740394715080255729,
        16081087942914969005,
        2785527013044645434,
    ]),
    BlsScalar([
        5441723562765,
        10118819971713642739,
        8698650145635425698,
        4525237367825298346,
    ]),
    BlsScalar([
        5450313497355,
        16497245228347029749,
        1316212348355882391,
        6264947722605951258,
    ]),
    BlsScalar([
        5458903431945,
        4428926411270865143,
        12380518624785890701,
        8004658077386604169,
    ]),
    BlsScalar([
        5471788333830,
        4773192259366169850,
        1306861928866575741,
        1390851572702807729,
    ]),
    BlsScalar([
        5480378268420,
        11151617515999556860,
        12371168205296584050,
        3130561927483460640,
    ]),
    BlsScalar([
        5488968203010,
        17530042772632943870,
        4988730408017040743,
        4870272282264113552,
    ]),
    BlsScalar([
        5497558137600,
        5461723955556779264,
        16053036684447049053,
        6609982637044766463,
    ]),
    BlsScalar([
        5506148072190,
        11840149212190166274,
        8670598887167505746,
        8349692991825419375,
    ]),
    BlsScalar([
        5519032974075,
        12184415060285470981,
        16043686264957742402,
        1735886487141622934,
    ]),
    BlsScalar([
        5527622908665,
        116096243209306375,
        8661248467678199096,
        3475596841922275846,
    ]),
    BlsScalar([
        5536212843255,
        6494521499842693385,
        1278810670398655789,
        5215307196702928758,
    ]),
    BlsScalar([
        5544802777845,
        12872946756476080395,
        12343116946828664098,
        6955017551483581669,
    ]),
    BlsScalar([
        5557687679730,
        13217212604571385102,
        1269460250909349138,
        341211046799785229,
    ]),
    BlsScalar([
        5566277614320,
        1148893787495220496,
        12333766527339357448,
        2080921401580438140,
    ]),
    BlsScalar([
        5574867548910,
        7527319044128607506,
        4951328730059814141,
        3820631756361091052,
    ]),
    BlsScalar([
        5583457483500,
        13905744300761994516,
        16015635006489822450,
        5560342111141743963,
    ]),
    BlsScalar([
        5592047418090,
        1837425483685829910,
        8633197209210279144,
        7300052465922396875,
    ]),
    BlsScalar([
        5604932319975,
        2181691331781134617,
        16006284587000515800,
        686245961238600434,
    ]),
    BlsScalar([
        5613522254565,
        8560116588414521627,
        8623846789720972493,
        2425956316019253346,
    ]),
    BlsScalar([
        5622112189155,
        14938541845047908637,
        1241408992441429186,
        4165666670799906258,
    ]),
    BlsScalar([
        5630702123745,
        2870223027971744031,
        12305715268871437496,
        5905377025580559169,
    ]),
    BlsScalar([
        5639292058335,
        9248648284605131041,
        4923277471591894189,
        7645087380361212081,
    ]),
    BlsScalar([
        5652176960220,
        9592914132700435748,
        12296364849382130845,
        1031280875677415640,
    ]),
    BlsScalar([
        5660766894810,
        15971339389333822758,
        4913927052102587538,
        2770991230458068552,
    ]),
    BlsScalar([
        5669356829400,
        3903020572257658152,
        15978233328532595848,
        4510701585238721463,
    ]),
    BlsScalar([
        5677946763990,
        10281445828891045162,
        8595795531253052541,
        6250411940019374375,
    ]),
    BlsScalar([
        5686536698580,
        16659871085524432172,
        1213357733973509234,
        7990122294800027287,
    ]),
    BlsScalar([
        5699421600465,
        17004136933619736879,
        8586445111763745890,
        1376315790116230846,
    ]),
    BlsScalar([
        5708011535055,
        4935818116543572273,
        1204007314484202584,
        3116026144896883758,
    ]),
    BlsScalar([
        5716601469645,
        11314243373176959283,
        12268313590914210893,
        4855736499677536669,
    ]),
    BlsScalar([
        5725191404235,
        17692668629810346293,
        4885875793634667586,
        6595446854458189581,
    ]),
    BlsScalar([
        5733781338825,
        5624349812734181687,
        15950182070064675896,
        8335157209238842492,
    ]),
    BlsScalar([
        5746666240710,
        5968615660829486394,
        4876525374145360936,
        1721350704555046052,
    ]),
    BlsScalar([
        5755256175300,
        12347040917462873404,
        15940831650575369245,
        3461061059335698963,
    ]),
    BlsScalar([
        5763846109890,
        278722100386708798,
        8558393853295825939,
        5200771414116351875,
    ]),
    BlsScalar([
        5772436044480,
        6657147357020095808,
        1175956056016282632,
        6940481768897004787,
    ]),
    BlsScalar([
        5785320946365,
        7001413205115400515,
        8549043433806519288,
        326675264213208346,
    ]),
    BlsScalar([
        5793910880955,
        13379838461748787525,
        1166605636526975981,
        2066385618993861258,
    ]),
    BlsScalar([
        5802500815545,
        1311519644672622919,
        12230911912956984291,
        3806095973774514169,
    ]),
    BlsScalar([
        5811090750135,
        7689944901306009929,
        4848474115677440984,
        5545806328555167081,
    ]),
    BlsScalar([
        5819680684725,
        14068370157939396939,
        15912780392107449293,
        7285516683335819992,
    ]),
    BlsScalar([
        5832565586610,
        14412636006034701646,
        4839123696188134333,
        671710178652023552,
    ]),
    BlsScalar([
        5841155521200,
        2344317188958537040,
        15903429972618142643,
        2411420533432676463,
    ]),
    BlsScalar([
        5849745455790,
        8722742445591924050,
        8520992175338599336,
        4151130888213329375,
    ]),
    BlsScalar([
        5858335390380,
        15101167702225311060,
        1138554378059056029,
        5890841242993982287,
    ]),
    BlsScalar([
        5866925324970,
        3032848885149146454,
        12202860654489064339,
        7630551597774635198,
    ]),
    BlsScalar([
        5879810226855,
        3377114733244451161,
        1129203958569749379,
        1016745093090838758,
    ]),
    BlsScalar([
        5888400161445,
        9755539989877838171,
        12193510234999757688,
        2756455447871491669,
    ]),
    BlsScalar([
        5896990096035,
        16133965246511225181,
        4811072437720214381,
        4496165802652144581,
    ]),
    BlsScalar([
        5905580030625,
        4065646429435060575,
        15875378714150222691,
        6235876157432797492,
    ]),
    BlsScalar([
        5914169965215,
        10444071686068447585,
        8492940916870679384,
        7975586512213450404,
    ]),
    BlsScalar([
        5927054867100,
        10788337534163752292,
        15866028294660916040,
        1361780007529653963,
    ]),
    BlsScalar([
        5935644801690,
        17166762790797139302,
        8483590497381372733,
        3101490362310306875,
    ]),
    BlsScalar([
        5944234736280,
        5098443973720974696,
        1101152700101829427,
        4841200717090959787,
    ]),
    BlsScalar([
        5952824670870,
        11476869230354361706,
        12165458976531837736,
        6580911071871612698,
    ]),
    BlsScalar([
        5961414605460,
        17855294486987748716,
        4783021179252294429,
        8320621426652265610,
    ]),
    BlsScalar([
        5974299507345,
        18199560335083053423,
        12156108557042531085,
        1706814921968469169,
    ]),
    BlsScalar([
        5982889441935,
        6131241518006888817,
        4773670759762987779,
        3446525276749122081,
    ]),
    BlsScalar([
        5991479376525,
        12509666774640275827,
        15837977036192996088,
        5186235631529774992,
    ]),
    BlsScalar([
        6000069311115,
        441347957564111221,
        8455539238913452782,
        6925945986310427904,
    ]),
    BlsScalar([
        6012954213000,
        785613805659415928,
        15828626616703689438,
        312139481626631463,
    ]),
    BlsScalar([
        6021544147590,
        7164039062292802938,
        8446188819424146131,
        2051849836407284375,
    ]),
    BlsScalar([
        6030134082180,
        13542464318926189948,
        1063751022144602824,
        3791560191187937287,
    ]),
    BlsScalar([
        6038724016770,
        1474145501850025342,
        12128057298574611134,
        5531270545968590198,
    ]),
    BlsScalar([
        6047313951360,
        7852570758483412352,
        4745619501295067827,
        7270980900749243110,
    ]),
    BlsScalar([
        6060198853245,
        8196836606578717059,
        12118706879085304483,
        657174396065446669,
    ]),
    BlsScalar([
        6068788787835,
        14575261863212104069,
        4736269081805761176,
        2396884750846099581,
    ]),
    BlsScalar([
        6077378722425,
        2506943046135939463,
        15800575358235769486,
        4136595105626752492,
    ]),
    BlsScalar([
        6085968657015,
        8885368302769326473,
        8418137560956226179,
        5876305460407405404,
    ]),
    BlsScalar([
        6094558591605,
        15263793559402713483,
        1035699763676682872,
        7616015815188058316,
    ]),
    BlsScalar([
        6107443493490,
        15608059407498018190,
        8408787141466919528,
        1002209310504261875,
    ]),
    BlsScalar([
        6116033428080,
        3539740590421853584,
        1026349344187376222,
        2741919665284914787,
    ]),
    BlsScalar([
        6124623362670,
        9918165847055240594,
        12090655620617384531,
        4481630020065567698,
    ]),
    BlsScalar([
        6133213297260,
        16296591103688627604,
        4708217823337841224,
        6221340374846220610,
    ]),
    BlsScalar([
        6141803231850,
        4228272286612462998,
        15772524099767849534,
        7961050729626873521,
    ]),
    BlsScalar([
        6154688133735,
        4572538134707767705,
        4698867403848534574,
        1347244224943077081,
    ]),
    BlsScalar([
        6163278068325,
        10950963391341154715,
        15763173680278542883,
        3086954579723729992,
    ]),
    BlsScalar([
        6171868002915,
        17329388647974541725,
        8380735882998999576,
        4826664934504382904,
    ]),
    BlsScalar([
        6180457937505,
        5261069830898377119,
        998298085719456270,
        6566375289285035816,
    ]),
    BlsScalar([
        6189047872095,
        11639495087531764129,
        12062604362149464579,
        8306085644065688727,
    ]),
    BlsScalar([
        6201932773980,
        11983760935627068836,
        988947666230149619,
        1692279139381892287,
    ]),
    BlsScalar([
        6210522708570,
        18362186192260455846,
        12053253942660157928,
        3431989494162545198,
    ]),
    BlsScalar([
        6219112643160,
        6293867375184291240,
        4670816145380614622,
        5171699848943198110,
    ]),
    BlsScalar([
        6227702577750,
        12672292631817678250,
        15735122421810622931,
        6911410203723851021,
    ]),
    BlsScalar([
        6240587479635,
        13016558479912982957,
        4661465725891307971,
        297603699040054581,
    ]),
    BlsScalar([
        6249177414225,
        948239662836818351,
        15725772002321316281,
        2037314053820707492,
    ]),
    BlsScalar([
        6257767348815,
        7326664919470205361,
        8343334205041772974,
        3777024408601360404,
    ]),
    BlsScalar([
        6266357283405,
        13705090176103592371,
        960896407762229667,
        5516734763382013316,
    ]),
    BlsScalar([
        6274947217995,
        1636771359027427765,
        12025202684192237977,
        7256445118162666227,
    ]),
    BlsScalar([
        6287832119880,
        1981037207122732472,
        951545988272923017,
        642638613478869787,
    ]),
    BlsScalar([
        6296422054470,
        8359462463756119482,
        12015852264702931326,
        2382348968259522698,
    ]),
    BlsScalar([
        6305011989060,
        14737887720389506492,
        4633414467423388019,
        4122059323040175610,
    ]),
    BlsScalar([
        6313601923650,
        2669568903313341886,
        15697720743853396329,
        5861769677820828521,
    ]),
    BlsScalar([
        6322191858240,
        9047994159946728896,
        8315282946573853022,
        7601480032601481433,
    ]),
    BlsScalar([
        6335076760125,
        9392260008042033603,
        15688370324364089678,
        987673527917684992,
    ]),
    BlsScalar([
        6343666694715,
        15770685264675420613,
        8305932527084546371,
        2727383882698337904,
    ]),
    BlsScalar([
        6352256629305,
        3702366447599256007,
        923494729805003065,
        4467094237478990816,
    ]),
    BlsScalar([
        6360846563895,
        10080791704232643017,
        11987801006235011374,
        6206804592259643727,
    ]),
    BlsScalar([
        6369436498485,
        16459216960866030027,
        4605363208955468067,
        7946514947040296639,
    ]),
    BlsScalar([
        6382321400370,
        16803482808961334734,
        11978450586745704723,
        1332708442356500198,
    ]),
    BlsScalar([
        6390911334960,
        4735163991885170128,
        4596012789466161417,
        3072418797137153110,
    ]),
    BlsScalar([
        6399501269550,
        11113589248518557138,
        15660319065896169726,
        4812129151917806021,
    ]),
    BlsScalar([
        6408091204140,
        17492014505151944148,
        8277881268616626419,
        6551839506698458933,
    ]),
    BlsScalar([
        6416681138730,
        5423695688075779542,
        895443471337083113,
        8291549861479111845,
    ]),
    BlsScalar([
        6429566040615,
        5767961536171084249,
        8268530849127319769,
        1677743356795315404,
    ]),
    BlsScalar([
        6438155975205,
        12146386792804471259,
        886093051847776462,
        3417453711575968316,
    ]),
    BlsScalar([
        6446745909795,
        78067975728306653,
        11950399328277784772,
        5157164066356621227,
    ]),
    BlsScalar([
        6455335844385,
        6456493232361693663,
        4567961530998241465,
        6896874421137274139,
    ]),
    BlsScalar([
        6468220746270,
        6800759080456998370,
        11941048908788478121,
        283067916453477698,
    ]),
    BlsScalar([
        6476810680860,
        13179184337090385380,
        4558611111508934814,
        2022778271234130610,
    ]),
    BlsScalar([
        6485400615450,
        1110865520014220774,
        15622917387938943124,
        3762488626014783521,
    ]),
    BlsScalar([
        6493990550040,
        7489290776647607784,
        8240479590659399817,
        5502198980795436433,
    ]),
    BlsScalar([
        6502580484630,
        13867716033280994794,
        858041793379856510,
        7241909335576089345,
    ]),
    BlsScalar([
        6515465386515,
        14211981881376299501,
        8231129171170093166,
        628102830892292904,
    ]),
    BlsScalar([
        6524055321105,
        2143663064300134895,
        848691373890549860,
        2367813185672945816,
    ]),
    BlsScalar([
        6532645255695,
        8522088320933521905,
        11912997650320558169,
        4107523540453598727,
    ]),
    BlsScalar([
        6541235190285,
        14900513577566908915,
        4530559853041014862,
        5847233895234251639,
    ]),
    BlsScalar([
        6549825124875,
        2832194760490744309,
        15594866129471023172,
        7586944250014904550,
    ]),
    BlsScalar([
        6562710026760,
        3176460608586049016,
        4521209433551708212,
        973137745331108110,
    ]),
    BlsScalar([
        6571299961350,
        9554885865219436026,
        15585515709981716521,
        2712848100111761021,
    ]),
    BlsScalar([
        6579889895940,
        15933311121852823036,
        8203077912702173214,
        4452558454892413933,
    ]),
    BlsScalar([
        6588479830530,
        3864992304776658430,
        820640115422629908,
        6192268809673066845,
    ]),
    BlsScalar([
        6597069765120,
        10243417561410045440,
        11884946391852638217,
        7931979164453719756,
    ]),
    BlsScalar([
        6609954667005,
        10587683409505350147,
        811289695933323257,
        1318172659769923316,
    ]),
    BlsScalar([
        6618544601595,
        16966108666138737157,
        11875595972363331566,
        3057883014550576227,
    ]),
    BlsScalar([
        6627134536185,
        4897789849062572551,
        4493158175083788260,
        4797593369331229139,
    ]),
    BlsScalar([
        6635724470775,
        11276215105695959561,
        15557464451513796569,
        6537303724111882050,
    ]),
    BlsScalar([
        6644314405365,
        17654640362329346571,
        8175026654234253262,
        8277014078892534962,
    ]),
    BlsScalar([
        6657199307250,
        17998906210424651278,
        15548114032024489918,
        1663207574208738521,
    ]),
];
