pub mod types;
pub mod utils;
use serde_json::json;
use serde_json::to_string;
// use std::collections::HashMap;
pub use types::*;

pub async fn new(playit_opts: Option<PlayItOpts>) -> Result<PlayIt, String> {
  return PlayIt::new(playit_opts).await;
}

impl PlayIt {
  pub async fn new(playit_opts: Option<PlayItOpts>) -> Result<Self, String> {
    return utils::get_defaults(playit_opts).await;
  }

  #[cfg(feature = "create-tunnels")]
  pub async fn create_tunnel(&self, mut options: CreateTunnelOptions) -> Tunnel {
    options.ip = Option::from(options.ip.unwrap_or(String::from("127.0.0.1")));

    let tunnel_id = self
      .req_client
      .post(self.api_path.join("/account/tunnels").unwrap())
      .body(
        to_string(&json!({
          "id": null,
          "game": format!("custom-{}", options.proto.to_lowercase()),
          "local_port": options.port,
          "local_ip": options.ip,
          "local_proto": options.proto,
          "agent_id": &self
                        .req_client
                        .get(self.api_path.join("/account/agents").unwrap())
                        .send()
                        .await
                        .unwrap()
                        .json::<Agents>()
                        .await
                        .unwrap()
                        .agents
                        .into_iter()
                        .find(|agent| agent.key == self.agent_key)
                        .unwrap()
                        .id,
          "domain_id": null
        }))
        .unwrap(),
      )
      .send()
      .await
      .unwrap()
      .json::<Id>()
      .await
      .unwrap()
      .id;

    return loop {
      let data = self
        .req_client
        .get(self.api_path.join("/account/tunnels").unwrap())
        .send()
        .await
        .unwrap()
        .json::<Tunnels>()
        .await
        .unwrap()
        .tunnels
        .into_iter()
        .find(|tunnel| tunnel.id == tunnel_id)
        .unwrap();

      if !data.domain_id.is_none() && !data.connect_address.is_none() {
        break data;
      }
    };
  }

  #[cfg(feature = "enable-tunnels")]
  pub async fn enable_tunnel(&self, id: i32) {
    self
      .req_client
      .get(
        self
          .api_path
          .join(&format!("/account/tunnels/{}/enable", id))
          .unwrap(),
      )
      .send()
      .await
      .expect(&format!("Failed To Enable The Tunnel"));
  }

  #[cfg(feature = "disable-tunnels")]
  pub async fn disable_tunnel(&self, id: i32) {
    self
      .req_client
      .get(
        self
          .api_path
          .join(&format!("/account/tunnels/{}/disable", id))
          .unwrap(),
      )
      .send()
      .await
      .expect(&format!("Failed To Failed The Tunnel"));
  }

  pub fn as_ref(&self) -> &Self {
    return &self;
  }
}

#[cfg(test)]
mod tests {
  use crate::types::*;
  use std::io::BufRead;
  use std::io::BufReader;
  #[tokio::test]
  async fn start() {
    let mut playit = PlayIt::new(None).await.unwrap();

    let tunnel = playit
      .create_tunnel(CreateTunnelOptions {
        ip: None,
        proto: String::from("tcp"),
        port: 8080,
      })
      .await;

    print!("\n\n");
    println!("Architexture: {}", playit.arch);
    println!("Operating System: {}", playit.os);
    println!("PlayIt Tunnel: {}", tunnel.connect_address.unwrap());
    println!(
      "PlayIt Config File: {}",
      playit.config_file.to_str().unwrap()
    );
    println!("PlayIt Binary: {}", playit.binary.to_str().unwrap());
    println!("PlayIt Version: {}", playit.version);
    println!("PlayIt Agent Key: {}", playit.agent_key);
    println!("PlayIt Server: {}", playit.server);
    println!("PlayIt First Lines Of PlayIt Output: \n{}", {
      let mut command_stderr: Vec<u8> = Vec::new();
      let mut command_stdout: Vec<u8> = Vec::new();

      BufReader::new(playit.playit.stderr.as_mut().unwrap())
        .read_until(b'\n', &mut command_stderr)
        .unwrap();
      BufReader::new(playit.playit.stdout.as_mut().unwrap())
        .read_until(b'\n', &mut command_stdout)
        .unwrap();
      String::from_utf8([command_stdout, command_stderr].concat()).unwrap()
    });
    print!("\n\n");
  }
}
