/* Copyright (c) Fortanix, Inc.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#[cfg(not(test))]
extern crate rand;
extern crate yasna;
extern crate bit_vec;
extern crate num_bigint;
extern crate pkix;

#[cfg(not(test))]
use self::rand::distributions::{IndependentSample, Range};
use self::bit_vec::BitVec;
use self::num_bigint::BigUint;
use self::yasna::models::{ObjectIdentifier, TaggedDerValue};
use self::yasna::tags::{TAG_UTF8STRING, TAG_PRINTABLESTRING};
use self::pkix::types::{DateTime, DerSequence, Attribute, Extension, RsaPkcs15, Sha256};
use self::pkix::{DerWrite, oid};
use self::pkix::pkcs10::{CertificationRequest, CertificationRequestInfo};
use self::pkix::x509::{Certificate, TbsCertificate};
use self::pkix::cms::*;
use self::pkix::algorithms::*;

#[cfg(not(test))]
pub fn random_printable_string(len: usize) -> Vec<u8> {
    let map = b"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789'()+,-.=/:? ";

    let between = Range::new(0usize, 74);
    let mut rng = rand::thread_rng();
    (0..len).map(|_| map[between.ind_sample(&mut rng)]).collect()
}

#[cfg(test)]
fn deterministic_printable_string(len: usize) -> Vec<u8> {
    let mut ret = "test".repeat((len + 3) / 4).into_bytes();
    ret.truncate(len);
    ret
}

pub fn csr(get_random_printable_string: fn(usize) -> Vec<u8>)
    -> CertificationRequest<CertificationRequestInfo<'static, DerSequence<'static>>, RsaPkcs15<Sha256>, BitVec>
{
    let oid_rsa_encryption = ObjectIdentifier::from(vec![1, 2, 840, 113549, 1, 1, 1]);

    let rsapubkey = yasna::construct_der(|writer| {
        writer.write_sequence(|writer| {
            // modulus
            writer.next().write_biguint(&BigUint::parse_bytes(b"221819298104283420426235175597403841201", 10).unwrap());
            // public exponent
            writer.next().write_u8(3);
        })
    });

    let exts = yasna::construct_der(|writer| writer.write_sequence(|writer|{
        Extension {
                oid: oid::basicConstraints.clone(),
                critical: true,
                value: vec![0x30, 0],
            }
            .write(writer.next());
    }));

    CertificationRequest {
            reqinfo: CertificationRequestInfo {
                subject:
                    vec![(oid::description.clone(),
                          TaggedDerValue::from_tag_and_bytes(TAG_UTF8STRING, b"Known keys only".to_vec())),
                         (oid::dnQualifier.clone(),
                          TaggedDerValue::from_tag_and_bytes(TAG_PRINTABLESTRING, get_random_printable_string(42)))].into(),
                spki: yasna::construct_der(|w|
                    w.write_sequence(|writer| {
                        writer.next().write_sequence(|writer| {
                            oid_rsa_encryption.write(writer.next());
                            writer.next().write_null();
                        });
                        BitVec::from_bytes(&rsapubkey).write(writer.next());
                    })
                ).into(),
                attributes: vec![Attribute {
                                  oid: oid::extensionRequest.clone(),
                                  value: vec![exts.into()],
                              }],
            },
            sigalg: RsaPkcs15(Sha256),
            sig: BitVec::from_elem(8, false),
        }
}

pub fn csr_der(get_random_printable_string: fn(usize) -> Vec<u8>) -> Vec<u8> {
    yasna::construct_der(|w| csr(get_random_printable_string).write(w))
}


pub fn cert(get_random_printable_string: fn(usize) -> Vec<u8>)
    -> Certificate<TbsCertificate<i64, RsaPkcs15<Sha256>, DerSequence<'static>>, RsaPkcs15<Sha256>, BitVec>
{
    let oid_rsa_encryption = ObjectIdentifier::from(vec![1, 2, 840, 113549, 1, 1, 1]);

    let rsapubkey = yasna::construct_der(|writer| {
        writer.write_sequence(|writer| {
            // modulus
            writer.next().write_biguint(&BigUint::parse_bytes(b"221819298104283420426235175597403841201", 10).unwrap());
            // public exponent
            writer.next().write_u8(3);
        })
    });

    Certificate {
            tbscert: TbsCertificate {
                serial: 0,
                sigalg: RsaPkcs15(Sha256),
                issuer:
                    vec![(oid::dnQualifier.clone(),
                          TaggedDerValue::from_tag_and_bytes(TAG_PRINTABLESTRING, get_random_printable_string(42)))].into(),
                validity_notbefore: DateTime::new(1970, 1, 1, 0, 0, 0),
                validity_notafter: DateTime::new(1970, 1, 1, 0, 0, 0),
                subject:
                    vec![(oid::description.clone(),
                          TaggedDerValue::from_tag_and_bytes(TAG_UTF8STRING, b"Known keys only".to_vec())),
                         (oid::dnQualifier.clone(),
                          TaggedDerValue::from_tag_and_bytes(TAG_PRINTABLESTRING, get_random_printable_string(42)))].into(),
                spki: yasna::construct_der(|w|
                    w.write_sequence(|writer| {
                        writer.next().write_sequence(|writer| {
                            oid_rsa_encryption.write(writer.next());
                            writer.next().write_null();
                        });
                        BitVec::from_bytes(&rsapubkey).write(writer.next());
                    })
                ).into(),
                extensions: vec![Extension {
                                  oid: oid::basicConstraints.clone(),
                                  critical: true,
                                  value: vec![0x30, 0],
                              }],
            },
            sigalg: RsaPkcs15(Sha256),
            sig: BitVec::from_elem(8, false),
        }
}

pub fn cert_der(get_random_printable_string: fn(usize) -> Vec<u8>) -> Vec<u8> {
    yasna::construct_der(|w| cert(get_random_printable_string).write(w))
}

pub fn enveloped_data_v2() -> EnvelopedDataV2 {
    let key_enc_algo = RSAES_OAEP {
        hashAlgorithm: HashAlgorithm::from(sha256 {}),
        maskGenAlgorithm: MaskGenAlgorithm::from(mgf1 {}),
    };
    let recipient_info = KeyTransRecipientInfoV2::new(vec![0; 20], key_enc_algo, &vec![0xff; 128]);
    let mgr_zone_sealed_proto : ObjectIdentifier = vec![1, 3, 6, 1, 4, 1, 49690, 3, 1].into();
    let encrypted_content_info = EncryptedContentInfo {
        contentType: mgr_zone_sealed_proto,
        contentEncryptionAlgorithm: aes128_cbc::new(&[0; 16]).unwrap().into(),
    };
    EnvelopedDataV2::new(
        vec![recipient_info],
        encrypted_content_info,
        &vec![0xff; 128],
    )
}

pub fn fake_callback(_: &[u8]) -> Result<Vec<u8>, ()> {
    Ok(vec![0xff; 32])
}

pub fn signed_data_v3(get_random_printable_string: fn(usize) -> Vec<u8>) -> SignedDataV3 {
    let enveloped_data: EnvelopedData = enveloped_data_v2().into();
    let encapsulated_content: EncapsulatedContentInfo = enveloped_data.into();
    let cert = cert(get_random_printable_string);
    let cert = yasna::construct_der(|w| cert.write(w));
    let signer_key_id = vec![0; 20];

    let signature_algo = RSASSA_PSS {
        hashAlgorithm: HashAlgorithm::from(sha256 {}),
        maskAlgorithm: MaskGenAlgorithm::from(mgf1 {}),
        saltLength: 32.into(),
    };
    SignedDataV3Builder::new(encapsulated_content)
        .add_digest_algorithm(sha256 {}, Box::new(fake_callback))
        .unwrap()
        .add_signer_info_v3(
            vec![cert],
            signer_key_id,
            sha256 {},
            vec![],
            signature_algo,
            Box::new(fake_callback),
        )
        .unwrap()
        .sign()
        .unwrap()
}

#[test]
fn fake_csr() {
    use yasna::BERDecodable;

    let der = vec![
                 0x30, 0x81, 0xb7, 0x30, 0x81, 0xa1, 0x02, 0x01,
                 0x00, 0x30, 0x4f, 0x31, 0x18, 0x30, 0x16, 0x06,
                 0x03, 0x55, 0x04, 0x0d, 0x0c, 0x0f, 0x4b, 0x6e,
                 0x6f, 0x77, 0x6e, 0x20, 0x6b, 0x65, 0x79, 0x73,
                 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x31, 0x33, 0x30,
                 0x31, 0x06, 0x03, 0x55, 0x04, 0x2e, 0x13, 0x2a,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x30, 0x2a, 0x30, 0x0d, 0x06, 0x09,
                 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                 0x01, 0x05, 0x00, 0x03, 0x19, 0x00, 0x30, 0x16,
                 0x02, 0x11, 0x00, 0xa6, 0xe0, 0xd7, 0xc4, 0xc4,
                 0xbf, 0xcf, 0x69, 0x63, 0x60, 0xca, 0x77, 0x3d,
                 0x00, 0xa2, 0xb1, 0x02, 0x01, 0x03, 0xa0, 0x1f,
                 0x30, 0x1d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
                 0xf7, 0x0d, 0x01, 0x09, 0x0e, 0x31, 0x10, 0x30,
                 0x0e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
                 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30,
                 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
                 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x02,
                 0x00, 0x00];

    let csr = csr(deterministic_printable_string);
    assert_eq!(yasna::construct_der(|w| csr.write(w)), der);
    assert_eq!(yasna::parse_der(&der, |r| CertificationRequest::decode_ber(r)).unwrap(), csr);
}

#[test]
fn fake_cert() {
    use yasna::BERDecodable;

    let der = vec![
                 0x30, 0x82, 0x01, 0x13, 0x30, 0x81, 0xfd, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x0d,
                 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x35, 0x31, 0x33,
                 0x30, 0x31, 0x06, 0x03, 0x55, 0x04, 0x2e, 0x13, 0x2a, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
                 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74,
                 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65,
                 0x30, 0x1e, 0x17, 0x0d, 0x37, 0x30, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
                 0x17, 0x0d, 0x37, 0x30, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x4f,
                 0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0d, 0x0c, 0x0f, 0x4b, 0x6e, 0x6f, 0x77, 0x6e, 0x20,
                 0x6b, 0x65, 0x79, 0x73, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x31, 0x33, 0x30, 0x31, 0x06, 0x03, 0x55, 0x04,
                 0x2e, 0x13, 0x2a, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65,
                 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73,
                 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x30, 0x2a, 0x30, 0x0d, 0x06, 0x09,
                 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x19, 0x00, 0x30, 0x16, 0x02,
                 0x11, 0x00, 0xa6, 0xe0, 0xd7, 0xc4, 0xc4, 0xbf, 0xcf, 0x69, 0x63, 0x60, 0xca, 0x77, 0x3d, 0x00, 0xa2,
                 0xb1, 0x02, 0x01, 0x03, 0xa3, 0x10, 0x30, 0x0e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01,
                 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                 0x0b, 0x05, 0x00, 0x03, 0x02, 0x00, 0x00];

    let cert = cert(deterministic_printable_string);
    assert_eq!(yasna::construct_der(|w| cert.write(w)), der);
    assert_eq!(yasna::parse_der(&der, |r| Certificate::decode_ber(r)).unwrap(), cert);
}

#[test]
fn no_extensions() {
    use yasna::BERDecodable;

    let mut cert = cert(deterministic_printable_string);
    cert.tbscert.extensions = vec![];

    let der = yasna::construct_der(|w| cert.write(w));
    assert_eq!(
        yasna::parse_der(&der, |r| Certificate::decode_ber(r)).unwrap(),
        cert
    );
}

#[test]
fn fake_cms_enveloped_data() {
    use self::pkix::pem::{der_to_pem, PEM_CMS};
    let env_data = enveloped_data_v2();
    let env_data: EnvelopedData = env_data.into();
    let content_info: ContentInfo = env_data.into();
    let der = yasna::construct_der(|w| content_info.write(w));
    let expected_der = vec![
        0x30, 0x82, 0x1, 0xa6, 0x6, 0xb, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x10, 0x1,
        0x6, 0x30, 0x82, 0x1, 0x95, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x7, 0x3,
        0x30, 0x82, 0x1, 0x86, 0x2, 0x1, 0x2, 0x31, 0x81, 0xd0, 0x30, 0x81, 0xcd, 0x2, 0x1, 0x2,
        0x4, 0x14, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x7,
        0x30, 0x22, 0xa0, 0xf, 0x30, 0xd, 0x6, 0x9, 0x60, 0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x2,
        0x1, 0x5, 0x0, 0xa1, 0xf, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1,
        0x8, 0x5, 0x0, 0x4, 0x81, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x30, 0x2b,
        0x6, 0xa, 0x2b, 0x6, 0x1, 0x4, 0x1, 0x83, 0x84, 0x1a, 0x3, 0x1, 0x30, 0x1d, 0x6, 0x9, 0x60,
        0x86, 0x48, 0x1, 0x65, 0x3, 0x4, 0x1, 0x2, 0x4, 0x10, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x81, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff,
    ];
    let pem = der_to_pem(&der, &PEM_CMS);
    println!("PEM is {}", pem);
    assert_eq!(der, expected_der);
}

#[test]
fn fake_cms_signed_data() {
    use self::pkix::pem::{der_to_pem, PEM_CMS};
    use yasna::BERDecodable;
    let signed_data: SignedData = signed_data_v3(deterministic_printable_string).into();
    let content_info: ContentInfo = signed_data.into();
    let der = yasna::construct_der(|w| content_info.write(w));
    for (cnt, byte) in der.iter().enumerate() {
        if (cnt % 16) == 0 {
            print!("\n");
        }
        print!(" 0x{:02x},", byte.clone());
    }
    print!("\n");
    let expected_der = vec![
        0x30, 0x82, 0x03, 0xc9, 0x06, 0x0b, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x10,
        0x01, 0x06, 0x30, 0x82, 0x03, 0xb8, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
        0x07, 0x02, 0x30, 0x82, 0x03, 0xa9, 0x02, 0x01, 0x03, 0x31, 0x0f, 0x30, 0x0d, 0x06, 0x09,
        0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0x30, 0x82, 0x01, 0x9d,
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x03, 0xa0, 0x82, 0x01, 0x8e,
        0x04, 0x82, 0x01, 0x8a, 0x30, 0x82, 0x01, 0x86, 0x02, 0x01, 0x02, 0x31, 0x81, 0xd0, 0x30,
        0x81, 0xcd, 0x02, 0x01, 0x02, 0x04, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x2f, 0x06,
        0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x07, 0x30, 0x22, 0xa0, 0x0f, 0x30,
        0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xa1,
        0x0f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x08, 0x05,
        0x00, 0x04, 0x81, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x30, 0x2b, 0x06,
        0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x83, 0x84, 0x1a, 0x03, 0x01, 0x30, 0x1d, 0x06, 0x09,
        0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x01, 0x02, 0x04, 0x10, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x81, 0x80,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xa0, 0x82, 0x01, 0x1b, 0x04, 0x82, 0x01,
        0x17, 0x30, 0x82, 0x01, 0x13, 0x30, 0x81, 0xfd, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01,
        0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05,
        0x00, 0x30, 0x35, 0x31, 0x33, 0x30, 0x31, 0x06, 0x03, 0x55, 0x04, 0x2e, 0x13, 0x2a, 0x74,
        0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
        0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73,
        0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x30, 0x1e, 0x17, 0x0d,
        0x37, 0x30, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0x0d,
        0x37, 0x30, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x4f,
        0x31, 0x18, 0x30, 0x16, 0x06, 0x03, 0x55, 0x04, 0x0d, 0x0c, 0x0f, 0x4b, 0x6e, 0x6f, 0x77,
        0x6e, 0x20, 0x6b, 0x65, 0x79, 0x73, 0x20, 0x6f, 0x6e, 0x6c, 0x79, 0x31, 0x33, 0x30, 0x31,
        0x06, 0x03, 0x55, 0x04, 0x2e, 0x13, 0x2a, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74,
        0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73,
        0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65, 0x73, 0x74, 0x74, 0x65,
        0x73, 0x74, 0x74, 0x65, 0x30, 0x2a, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
        0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x19, 0x00, 0x30, 0x16, 0x02, 0x11, 0x00, 0xa6,
        0xe0, 0xd7, 0xc4, 0xc4, 0xbf, 0xcf, 0x69, 0x63, 0x60, 0xca, 0x77, 0x3d, 0x00, 0xa2, 0xb1,
        0x02, 0x01, 0x03, 0xa3, 0x10, 0x30, 0x0e, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
        0x01, 0xff, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
        0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x02, 0x00, 0x00, 0x31, 0x81, 0xd2, 0x30, 0x81,
        0xcf, 0x02, 0x01, 0x03, 0x80, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x0d, 0x06, 0x09,
        0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05, 0x00, 0xa0, 0x4d, 0x30, 0x1a,
        0x06, 0x0b, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x10, 0x01, 0x06, 0x31, 0x0b,
        0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x03, 0x30, 0x2f, 0x06, 0x09,
        0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04, 0x31, 0x22, 0x04, 0x20, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x30, 0x34, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x27,
        0xa0, 0x0f, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01,
        0x05, 0x00, 0xa1, 0x0f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
        0x01, 0x08, 0x05, 0x00, 0xa2, 0x03, 0x02, 0x01, 0x20, 0x04, 0x20, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    ];
    let pem = der_to_pem(&der, &PEM_CMS);
    println!("PEM is\n{}\n", pem);
    assert_eq!(der, expected_der);
    let decoded: ContentInfo = yasna::parse_der(&der, |r| ContentInfo::decode_ber(r)).unwrap();
    assert_eq!(content_info, decoded);
}
