use pkgmanager::PkgManager;
use std::process::Child;

pub mod pkgmanager;
mod pacman;
mod apt;

use apt::Apt;
use pacman::Pacman;


#[derive(Debug)]
pub enum PkgWrapperError {
    NoSupportedPackageManagerFound,
    UserIsNotRoot
}

pub struct PkgWrapper {
    pkg_manager: Box<dyn PkgManager>
}

impl PkgWrapper {
    pub fn new(no_confirm:bool) -> Result<Self, PkgWrapperError> {
        use nix::unistd::getuid;
        if getuid().as_raw() != 0 {
            return Err(PkgWrapperError::UserIsNotRoot);
        }
        let vec: Vec<Box<dyn PkgManager>> = vec![
            Box::new(Pacman::new()),
            Box::new(Apt::new())
        ];

        for mut pkg in vec {
            if std::path::Path::new(&format!("/bin/{}", pkg.get_name())).exists() {
                pkg.no_confirm(no_confirm);
                return Ok(Self {
                    pkg_manager:pkg
                })
            }
        }
        Err(PkgWrapperError::NoSupportedPackageManagerFound)
    }
    pub fn with_custom_pkg_managers(no_confirm: bool, vec: Vec<Box<dyn PkgManager>>) -> Result<Self, PkgWrapperError> {
        use nix::unistd::getuid;
        if getuid().as_raw() != 0 {
            return Err(PkgWrapperError::UserIsNotRoot);
        }
        for mut pkg in vec {
            if std::path::Path::new(&format!("/bin/{}", pkg.get_name())).exists() {
                pkg.no_confirm(no_confirm);
                return Ok(Self {
                    pkg_manager:pkg
                })
            }
        }
        Err(PkgWrapperError::NoSupportedPackageManagerFound)
    }
    pub fn install_pkg<T:ToString>(&mut self, pkg:T) -> Result<Child, std::io::Error>{
        self.pkg_manager.install(pkg.to_string())
    }

    pub fn uninstall_pkg<T:ToString>(&mut self, pkg:T) -> Result<Child, std::io::Error> {
        self.pkg_manager.uninstall(pkg.to_string())
    }
}


#[cfg(test)]
mod tests {
    use crate::PkgWrapper;

    #[test]
    fn detect() {
        PkgWrapper::new(true).unwrap();
    }
    #[test]
    fn install() {
        PkgWrapper::new(true).unwrap().install_pkg("htop").ok();
    }
}
