//! Pure Rust implementation of Public-Key Cryptography Standards (PKCS) #8:
//! Private-Key Information Syntax Specification ([RFC 5208]), with additional
//! support for PKCS#8v2 asymmetric key packages ([RFC 5958])
//!
//! # About
//! This library provides generalized PKCS#8 support designed to work with a
//! number of different algorithms. It supports `no_std` platforms including
//! ones without a heap (albeit with reduced functionality).
//!
//! It supports decoding/encoding the following types:
//!
//! - [`EncryptedPrivateKeyInfo`]: (with `pkcs5` feature) encrypted key.
//! - [`PrivateKeyInfo`]: algorithm identifier and data representing a private key.
//!   Optionally also includes public key data for asymmetric keys.
//! - [`SubjectPublicKeyInfo`]: algorithm identifier and data representing a public key
//!   (re-exported from the [`spki`] crate)
//!
//! When the `alloc` feature is enabled, the following additional types are
//! available which provide more convenient decoding/encoding support:
//!
//! - [`EncryptedPrivateKeyDocument`]: (with `pkcs5` feature) heap-backed encrypted key.
//! - [`PrivateKeyDocument`]: heap-backed storage for serialized [`PrivateKeyInfo`].
//! - [`PublicKeyDocument`]: heap-backed storage for serialized [`SubjectPublicKeyInfo`].
//!
//! When the `pem` feature is enabled, it also supports decoding/encoding
//! documents from "PEM encoding" format as defined in RFC 7468.
//!
//! # Supported Algorithms
//! This crate has been written generically so it can be used to implement
//! PKCS#8 support for any algorithm.
//!
//! However, it's only tested against keys generated by OpenSSL for the
//! following algorithms:
//!
//! - ECC (`id-ecPublicKey`)
//! - Ed25519 (`Ed25519`)
//! - RSA (`rsaEncryption`)
//!
//! Please open an issue if you encounter trouble using it with other
//! algorithms.
//!
//! # Encrypted Private Key Support
//! [`EncryptedPrivateKeyInfo`] supports decoding/encoding encrypted PKCS#8
//! private keys and is gated under the `pkcs5` feature. The corresponding
//! [`EncryptedPrivateKeyDocument`] type provides heap-backed storage
//! (`alloc` feature required).
//!
//! When the `encryption` feature of this crate is enabled, it provides
//! [`EncryptedPrivateKeyInfo::decrypt`] and [`PrivateKeyInfo::encrypt`]
//! functions which are able to decrypt/encrypt keys using the following
//! algorithms:
//!
//! - [PKCS#5v2 Password Based Encryption Scheme 2 (RFC 8018)]
//!   - Key derivation function: [scrypt] ([RFC 7914], also supports PBKDF2-HMAC-SHA256)
//!   - Symmetric encryption: AES-128-CBC or AES-256-CBC (best available options for PKCS#5v2)
//!
//! # Minimum Supported Rust Version
//!
//! This crate requires **Rust 1.51** at a minimum.
//!
//! [RFC 5208]: https://tools.ietf.org/html/rfc5208
//! [RFC 5958]: https://tools.ietf.org/html/rfc5958
//! [RFC 7914]: https://datatracker.ietf.org/doc/html/rfc7914
//! [PKCS#5v2 Password Based Encryption Scheme 2 (RFC 8018)]: https://tools.ietf.org/html/rfc8018#section-6.2
//! [scrypt]: https://en.wikipedia.org/wiki/Scrypt

#![no_std]
#![cfg_attr(docsrs, feature(doc_cfg))]
#![doc(
    html_logo_url = "https://raw.githubusercontent.com/RustCrypto/meta/master/logo.svg",
    html_favicon_url = "https://raw.githubusercontent.com/RustCrypto/meta/master/logo.svg",
    html_root_url = "https://docs.rs/pkcs8/0.7.3"
)]
#![forbid(unsafe_code, clippy::unwrap_used)]
#![warn(missing_docs, rust_2018_idioms, unused_qualifications)]

#[cfg(feature = "alloc")]
extern crate alloc;
#[cfg(feature = "std")]
extern crate std;

mod attributes;
mod error;
mod private_key_info;
mod traits;
mod version;

#[cfg(feature = "alloc")]
mod document;

#[cfg(feature = "pkcs5")]
pub(crate) mod encrypted_private_key_info;

pub use crate::{
    attributes::Attributes,
    error::{Error, Result},
    private_key_info::PrivateKeyInfo,
    traits::{FromPrivateKey, FromPublicKey},
    version::Version,
};
pub use der::{self, asn1::ObjectIdentifier};
pub use spki::{AlgorithmIdentifier, SubjectPublicKeyInfo};

#[cfg(feature = "alloc")]
pub use crate::{
    document::{private_key::PrivateKeyDocument, public_key::PublicKeyDocument},
    traits::{ToPrivateKey, ToPublicKey},
};

#[cfg(feature = "pem")]
use pem_rfc7468 as pem;

#[cfg(feature = "pkcs5")]
pub use encrypted_private_key_info::EncryptedPrivateKeyInfo;

#[cfg(feature = "pkcs5")]
pub use pkcs5;

#[cfg(all(feature = "alloc", feature = "pkcs5"))]
pub use crate::document::encrypted_private_key::EncryptedPrivateKeyDocument;
