//! Pure Rust implementation of Public-Key Cryptography Standards (PKCS) #8:
//!
//! Private-Key Information Syntax Specification (as defined in [RFC 5208]).
//!
//! # About
//! This library provides generalized PKCS#8 support designed to work with a
//! number of different algorithms. It supports `no_std` platforms including
//! ones without a heap (albeit with reduced functionality).
//!
//! It supports decoding/encoding the following types:
//!
//! - [`EncryptedPrivateKeyInfo`]: (with `pkcs5` feature) encrypted key.
//! - [`PrivateKeyInfo`]: algorithm identifier and data representing a private key.
//! - [`OneAsymmetricKey`]: a version, algorithm identifier, private key data,
//!   and optionally public key data.
//! - [`SubjectPublicKeyInfo`]: algorithm identifier and data representing a public key
//!   (re-exported from the [`spki`] crate)
//!
//! When the `alloc` feature is enabled, the following additional types are
//! available which provide more convenient decoding/encoding support:
//!
//! - [`EncryptedPrivateKeyDocument`]: (with `pkcs5` feature) heap-backed encrypted key.
//! - [`PrivateKeyDocument`]: heap-backed storage for serialized [`PrivateKeyInfo`].
//! - [`PublicKeyDocument`]: heap-backed storage for serialized [`SubjectPublicKeyInfo`].
//!
//! When the `pem` feature is enabled, it also supports decoding/encoding
//! documents from "PEM encoding" format as defined in RFC 7468.
//!
//! # Supported Algorithms
//! This crate has been tested against keys generated by OpenSSL for the
//! following algorithms:
//!
//! - ECC (`id-ecPublicKey`)
//! - Ed25519 (`Ed25519`)
//! - RSA (`rsaEncryption`)
//!
//! It may work with other algorithms which use an optional OID for
//! [`AlgorithmIdentifier`] parameters.
//!
//! # Encrypted Private Key Support
//! [`EncryptedPrivateKeyInfo`] supports decoding/encoding encrypted PKCS#8
//! private keys and is gated under the `pkcs5` feature. The corresponding
//! [`EncryptedPrivateKeyDocument`] type provides heap-backed storage
//! (`alloc` feature required).
//!
//! When the `encryption` feature of this crate is enabled, it provides
//! [`EncryptedPrivateKeyInfo::decrypt`] and [`PrivateKeyInfo::encrypt`]
//! functions which are able to decrypt/encrypt keys using the following
//! algorithms:
//!
//! - [PKCS#5v2 Password Based Encryption Scheme 2 (RFC 8018)]
//!   - Key derivation function: PBKDF2 with HMAC-SHA256 as the PRF
//!   - Symmetric encryption: AES-128-CBC or AES-256-CBC
//!
//! # Minimum Supported Rust Version
//!
//! This crate requires **Rust 1.47** at a minimum.
//!
//! [RFC 5208]: https://tools.ietf.org/html/rfc5208
//! [PKCS#5v2 Password Based Encryption Scheme 2 (RFC 8018)]: https://tools.ietf.org/html/rfc8018#section-6.2

#![no_std]
#![cfg_attr(docsrs, feature(doc_cfg))]
#![doc(
    html_logo_url = "https://raw.githubusercontent.com/RustCrypto/meta/master/logo.svg",
    html_favicon_url = "https://raw.githubusercontent.com/RustCrypto/meta/master/logo.svg",
    html_root_url = "https://docs.rs/pkcs8/0.6.1"
)]
#![forbid(unsafe_code, clippy::unwrap_used)]
#![warn(missing_docs, rust_2018_idioms)]

#[cfg(feature = "alloc")]
extern crate alloc;

#[cfg(feature = "std")]
extern crate std;

mod error;
mod one_asymmetric_key;
mod private_key_info;
mod traits;
mod version;

#[cfg(feature = "alloc")]
mod document;

#[cfg(feature = "pkcs5")]
pub(crate) mod encrypted_private_key_info;

#[cfg(feature = "pem")]
mod pem;

pub use crate::{
    error::{Error, Result},
    one_asymmetric_key::OneAsymmetricKey,
    private_key_info::PrivateKeyInfo,
    traits::{FromPrivateKey, FromPublicKey},
    version::Version,
};
pub use der::{self, ObjectIdentifier};
pub use spki::{AlgorithmIdentifier, SubjectPublicKeyInfo};

#[cfg(feature = "pkcs5")]
pub use encrypted_private_key_info::EncryptedPrivateKeyInfo;

#[cfg(feature = "pkcs5")]
pub use pkcs5;

#[cfg(all(feature = "alloc", feature = "pkcs5"))]
pub use crate::document::encrypted_private_key::EncryptedPrivateKeyDocument;

#[cfg(feature = "alloc")]
pub use crate::{
    document::{private_key::PrivateKeyDocument, public_key::PublicKeyDocument},
    traits::{ToPrivateKey, ToPublicKey},
};

/// Attributes as defined in [RFC 5958 Section 2].
///
/// > attributes is OPTIONAL.  It contains information corresponding to
/// > the public key (e.g., certificates).  The attributes field uses the
/// > class `ATTRIBUTE` which is restricted by the
/// > `OneAsymmetricKeyAttributes` information object set.
/// > `OneAsymmetricKeyAttributes` is an open ended set in this document.
/// > Others documents can constrain these values.  Attributes from
/// > RFC2985 MAY be supported.
///
/// Attributes have the following ASN.1 schema:
///
/// ```text
/// Attributes ::= SET OF Attribute { { OneAsymmetricKeyAttributes } }
/// ```
///
/// [RFC 5958 Section 2]: https://datatracker.ietf.org/doc/html/rfc5958
// TODO(tarcieri): support parsing attributes as a `der::SetOf`?
pub type Attributes<'a> = der::Any<'a>;
