//! Password-Based Encryption Scheme 2 tests

use core::convert::TryFrom;
use der::Encodable;
use hex_literal::hex;
use pkcs5::pbes2;

/// PBES2 + PBKDF2-SHA1 + AES-128-CBC `AlgorithmIdentifier` example.
///
/// Generated by OpenSSL and extracted from the `pkcs8` crate's
/// `tests/examples/ed25519-encpriv-aes128-sha1.der` test vector.
const PBES2_PBKDF2_SHA1_AES128CBC_ALG_ID: &[u8] = &hex!(
    "304906092a864886f70d01050d303c301b06092a864886f70d01050c300e0408
     e8765e01e43b6bad02020800301d06096086480165030401020410223080a71b
     cd2b9a256d876c924979d2"
);

/// PBES2 + PBKDF2-SHA256 + AES-256-CBC `AlgorithmIdentifier` example.
///
/// Generated by OpenSSL and extracted from the `pkcs8` crate's
/// `tests/examples/ed25519-encpriv-aes256-sha256.der` test vector.
const PBES2_PBKDF2_SHA256_AES256CBC_ALG_ID: &[u8] = &hex!(
    "305706092a864886f70d01050d304a302906092a864886f70d01050c301c0408
     79d982e70df91a8802020800300c06082a864886f70d02090500301d06096086
     4801650304012a0410b2d02d78b2efd9dff694cf8e0af40925"
);

/// Decoding tests
#[test]
fn decode_pbes2_pbkdf2_sha1_aes128cbc() {
    let scheme = pkcs5::EncryptionScheme::try_from(PBES2_PBKDF2_SHA1_AES128CBC_ALG_ID).unwrap();
    let params = scheme.pbes2().unwrap();

    let pbkdf2_params = params.kdf.pbkdf2().unwrap();
    assert_eq!(pbkdf2_params.salt, &hex!("e8765e01e43b6bad"));
    assert_eq!(pbkdf2_params.iteration_count, 2048);
    assert_eq!(pbkdf2_params.key_length, None);
    assert_eq!(pbkdf2_params.prf, pbes2::Pbkdf2Prf::HmacWithSha1);

    match params.encryption {
        pbes2::EncryptionScheme::Aes128Cbc { iv } => {
            assert_eq!(iv, &hex!("223080a71bcd2b9a256d876c924979d2"));
        }
        other => panic!("unexpected encryption scheme: {:?}", other),
    }
}

/// Decoding tests
#[test]
fn decode_pbes2_pbkdf2_sha256_aes256cbc() {
    let scheme = pkcs5::EncryptionScheme::try_from(PBES2_PBKDF2_SHA256_AES256CBC_ALG_ID).unwrap();
    let params = scheme.pbes2().unwrap();

    let pbkdf2_params = params.kdf.pbkdf2().unwrap();
    assert_eq!(pbkdf2_params.salt, &hex!("79d982e70df91a88"));
    assert_eq!(pbkdf2_params.iteration_count, 2048);
    assert_eq!(pbkdf2_params.key_length, None);
    assert_eq!(pbkdf2_params.prf, pbes2::Pbkdf2Prf::HmacWithSha256);

    match params.encryption {
        pbes2::EncryptionScheme::Aes256Cbc { iv } => {
            assert_eq!(iv, &hex!("b2d02d78b2efd9dff694cf8e0af40925"));
        }
        other => panic!("unexpected encryption scheme: {:?}", other),
    }
}

/// Encoding tests
#[test]
fn encode_pbes2_pbkdf2_sha1_aes128cbc() {
    let mut buffer = [0u8; 1024];

    let scheme = pkcs5::EncryptionScheme::try_from(PBES2_PBKDF2_SHA1_AES128CBC_ALG_ID).unwrap();
    let mut encoder = der::Encoder::new(&mut buffer);
    scheme.encode(&mut encoder).unwrap();

    let encoded_der = encoder.finish().unwrap();
    assert_eq!(encoded_der, PBES2_PBKDF2_SHA1_AES128CBC_ALG_ID);
}

/// Encoding tests
#[test]
fn encode_pbes2_pbkdf2_sha256_aes256cbc() {
    let mut buffer = [0u8; 1024];

    let scheme = pkcs5::EncryptionScheme::try_from(PBES2_PBKDF2_SHA256_AES256CBC_ALG_ID).unwrap();
    let mut encoder = der::Encoder::new(&mut buffer);
    scheme.encode(&mut encoder).unwrap();

    let encoded_der = encoder.finish().unwrap();
    assert_eq!(encoded_der, PBES2_PBKDF2_SHA256_AES256CBC_ALG_ID);
}
