"use strict";

exports.stringToBase64Url = stringToBase64Url;
exports.base64ToBase64Url = base64ToBase64Url;
exports.base64UrlToBase64 = base64UrlToBase64;
exports.base64UrlToString = base64UrlToString;
exports.stringToBuffer = stringToBuffer;
exports.base64UrlDecode = base64UrlDecode;
exports.bind = bind;
exports.isAbsoluteUrl = isAbsoluteUrl;
exports.isString = isString;
exports.isObject = isObject;
exports.isNumber = isNumber;
exports.isoToUTCString = isoToUTCString;
exports.toQueryParams = toQueryParams;
exports.genRandomString = genRandomString;
exports.extend = extend;
exports.removeNils = removeNils;
exports.clone = clone;
exports.omit = omit;
exports.find = find;
exports.getLink = getLink;
exports.getNativeConsole = getNativeConsole;
exports.getConsole = getConsole;
exports.warn = warn;
exports.deprecate = deprecate;
exports.deprecateWrap = deprecateWrap;
exports.removeTrailingSlash = removeTrailingSlash;
exports.isIE11OrLess = isIE11OrLess;
exports.isFunction = isFunction;
exports.delay = delay;
exports.isPromise = isPromise;

/*!
 * Copyright (c) 2015-present, Okta, Inc. and/or its affiliates. All rights reserved.
 * The Okta software accompanied by this notice is provided pursuant to the Apache License, Version 2.0 (the "License.")
 *
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 */

/* global window, document, btoa, atob */
// converts a string to base64 (url/filename safe variant)
function stringToBase64Url(str) {
  var b64 = btoa(str);
  return base64ToBase64Url(b64);
} // converts a standard base64-encoded string to a "url/filename safe" variant


function base64ToBase64Url(b64) {
  return b64.replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/, '');
} // converts a "url/filename safe" base64 string to a "standard" base64 string


function base64UrlToBase64(b64u) {
  return b64u.replace(/-/g, '+').replace(/_/g, '/');
}

function base64UrlToString(b64u) {
  var b64 = base64UrlToBase64(b64u);

  switch (b64.length % 4) {
    case 0:
      break;

    case 2:
      b64 += '==';
      break;

    case 3:
      b64 += '=';
      break;

    default:
      throw 'Not a valid Base64Url';
  }

  var utf8 = atob(b64);

  try {
    return decodeURIComponent(escape(utf8));
  } catch (e) {
    return utf8;
  }
}

function stringToBuffer(str) {
  var buffer = new Uint8Array(str.length);

  for (var i = 0; i < str.length; i++) {
    buffer[i] = str.charCodeAt(i);
  }

  return buffer;
}

function base64UrlDecode(str) {
  return atob(base64UrlToBase64(str));
}

function bind(fn, ctx) {
  var additionalArgs = Array.prototype.slice.call(arguments, 2);
  return function () {
    var args = Array.prototype.slice.call(arguments);
    args = additionalArgs.concat(args);
    return fn.apply(ctx, args);
  };
}

function isAbsoluteUrl(url) {
  return /^(?:[a-z]+:)?\/\//i.test(url);
}

function isString(obj) {
  return Object.prototype.toString.call(obj) === '[object String]';
}

function isObject(obj) {
  return Object.prototype.toString.call(obj) === '[object Object]';
}

function isNumber(obj) {
  return Object.prototype.toString.call(obj) === '[object Number]';
}

function isoToUTCString(str) {
  var parts = str.match(/\d+/g),
      isoTime = Date.UTC(parts[0], parts[1] - 1, parts[2], parts[3], parts[4], parts[5]),
      isoDate = new Date(isoTime);
  return isoDate.toUTCString();
}

function toQueryParams(obj) {
  var str = [];

  if (obj !== null) {
    for (var key in obj) {
      if (Object.prototype.hasOwnProperty.call(obj, key) && obj[key] !== undefined && obj[key] !== null) {
        str.push(key + '=' + encodeURIComponent(obj[key]));
      }
    }
  }

  if (str.length) {
    return '?' + str.join('&');
  } else {
    return '';
  }
}

function genRandomString(length) {
  var randomCharset = 'abcdefghijklnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
  var random = '';

  for (var c = 0, cl = randomCharset.length; c < length; ++c) {
    random += randomCharset[Math.floor(Math.random() * cl)];
  }

  return random;
} // TODO: replace all references with `Object.assign` then remove this function


function extend() {
  // First object will be modified!
  var obj1 = arguments[0]; // Properties from other objects will be copied over

  var objArray = [].slice.call(arguments, 1);
  objArray.forEach(function (obj) {
    for (var prop in obj) {
      // copy over all properties with defined values
      if (Object.prototype.hasOwnProperty.call(obj, prop) && obj[prop] !== undefined) {
        obj1[prop] = obj[prop];
      }
    }
  });
  return obj1; // return the modified object
}

function removeNils(obj) {
  var cleaned = {};

  for (var prop in obj) {
    if (Object.prototype.hasOwnProperty.call(obj, prop)) {
      var value = obj[prop];

      if (value !== null && value !== undefined) {
        cleaned[prop] = value;
      }
    }
  }

  return cleaned;
}

function clone(obj) {
  if (obj) {
    var str = JSON.stringify(obj);

    if (str) {
      return JSON.parse(str);
    }
  }

  return obj;
} // Analogous to _.omit


function omit(obj, ...props) {
  // var props = Array.prototype.slice.call(arguments, 1);
  var newobj = {};

  for (var p in obj) {
    if (Object.prototype.hasOwnProperty.call(obj, p) && props.indexOf(p) == -1) {
      newobj[p] = obj[p];
    }
  }

  return clone(newobj);
}

function find(collection, searchParams) {
  var c = collection.length;

  while (c--) {
    var item = collection[c];
    var found = true;

    for (var prop in searchParams) {
      if (!Object.prototype.hasOwnProperty.call(searchParams, prop)) {
        continue;
      }

      if (item[prop] !== searchParams[prop]) {
        found = false;
        break;
      }
    }

    if (found) {
      return item;
    }
  }
}

function getLink(obj, linkName, altName) {
  if (!obj || !obj._links) {
    return;
  }

  var link = clone(obj._links[linkName]); // If a link has a name and we have an altName, return if they match

  if (link && link.name && altName) {
    if (link.name === altName) {
      return link;
    }
  } else {
    return link;
  }
}

function getNativeConsole() {
  if (typeof window !== 'undefined') {
    return window.console;
  } else if (typeof console !== 'undefined') {
    return console;
  } else {
    return undefined;
  }
}

function getConsole() {
  var nativeConsole = getNativeConsole();

  if (nativeConsole && nativeConsole.log) {
    return nativeConsole;
  }

  return {
    log: function () {}
  };
}

function warn(text) {
  /* eslint-disable no-console */
  getConsole().log('[okta-auth-sdk] WARN: ' + text);
  /* eslint-enable */
}

function deprecate(text) {
  /* eslint-disable no-console */
  getConsole().log('[okta-auth-sdk] DEPRECATION: ' + text);
  /* eslint-enable */
}

function deprecateWrap(text, fn) {
  return function () {
    deprecate(text);
    return fn.apply(null, arguments);
  };
}

function removeTrailingSlash(path) {
  if (!path) {
    return;
  } // Remove any whitespace before or after string


  var trimmed = path.replace(/^\s+|\s+$/gm, ''); // Remove trailing slash(es)

  trimmed = trimmed.replace(/\/+$/, '');
  return trimmed;
}

function isIE11OrLess() {
  return !!document.documentMode && document.documentMode <= 11;
}

function isFunction(fn) {
  return !!fn && {}.toString.call(fn) === '[object Function]';
}

function delay(ms) {
  return new Promise(function (resolve) {
    setTimeout(resolve, ms);
  });
}

function isPromise(obj) {
  return obj && obj.finally && typeof obj.finally === 'function';
}